package com.stripe.android.stripe3ds2.transaction

import android.content.Intent
import com.stripe.android.stripe3ds2.init.ui.StripeUiCustomization
import com.stripe.android.stripe3ds2.transactions.ChallengeRequestData
import com.stripe.android.stripe3ds2.transactions.ChallengeResponseData
import com.stripe.android.stripe3ds2.views.ChallengeActivity
import com.stripe.android.stripe3ds2.views.ChallengeViewArgs

/**
 * Creates the [Intent] for starting a [ChallengeActivity], and starts it.
 */
class ChallengeStarter internal constructor(
    private val host: Stripe3ds2ActivityStarterHost,
    creqData: ChallengeRequestData,
    cresData: ChallengeResponseData,
    uiCustomization: StripeUiCustomization,
    creqExecutorConfig: ChallengeRequestExecutor.Config,
    creqExecutorFactory: ChallengeRequestExecutor.Factory,
    errorExecutorFactory: ErrorRequestExecutor.Factory,
    challengeCompletionIntent: Intent? = null,
    private val challengeCompletionRequestCode: Int = 0
) {
    private val args = ChallengeViewArgs(
        cresData,
        creqData,
        uiCustomization,
        creqExecutorConfig,
        creqExecutorFactory,
        errorExecutorFactory,
        challengeCompletionIntent = challengeCompletionIntent,
        challengeCompletionRequestCode = challengeCompletionRequestCode
    )

    fun start() {
        if (challengeCompletionRequestCode > 0) {
            // when starting the first challenge screen, use `startActivityForResult`
            host.startActivityForResult(
                challengeIntent,
                challengeCompletionRequestCode
            )
        } else {
            // when starting subsequent challenge screens, a result is already expected, so
            // use `Intent.FLAG_ACTIVITY_FORWARD_RESULT` to forward it
            host.startActivity(
                challengeIntent
                    .addFlags(Intent.FLAG_ACTIVITY_FORWARD_RESULT)
            )
        }
    }

    private val challengeIntent: Intent
        get() = Intent(host.activity, ChallengeActivity::class.java)
            .putExtras(args.toBundle())

    companion object {
        fun create(
            host: Stripe3ds2ActivityStarterHost,
            creqData: ChallengeRequestData,
            cresData: ChallengeResponseData,
            uiCustomization: StripeUiCustomization,
            creqExecutorConfig: ChallengeRequestExecutor.Config,
            creqExecutorFactory: ChallengeRequestExecutor.Factory =
                StripeChallengeRequestExecutor.Factory(),
            errorRequestExecutor: ErrorRequestExecutor.Factory =
                StripeErrorRequestExecutor.Factory(),
            challengeCompletionIntent: Intent? = null,
            challengeCompletionRequestCode: Int = 0
        ): ChallengeStarter {
            return ChallengeStarter(
                host,
                creqData,
                cresData,
                uiCustomization,
                creqExecutorConfig,
                creqExecutorFactory,
                errorRequestExecutor,
                challengeCompletionIntent = challengeCompletionIntent,
                challengeCompletionRequestCode = challengeCompletionRequestCode
            )
        }
    }
}
