/*
 * Decompiled with CFR 0.152.
 */
package com.spotify.annoy;

import com.spotify.annoy.AnnoyIndex;
import com.spotify.annoy.IndexType;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteOrder;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.PriorityQueue;

public class ANNIndex
implements AnnoyIndex {
    private final ArrayList<Integer> roots;
    private MappedByteBuffer annBuf;
    private final int DIMENSION;
    private final int MIN_LEAF_SIZE;
    private final IndexType INDEX_TYPE;
    private final int INDEX_TYPE_OFFSET;
    private final int K_NODE_HEADER_STYLE;
    private final int NODE_SIZE;
    private final int INT_SIZE = 4;
    private final int FLOAT_SIZE = 4;

    public ANNIndex(int dimension, String filename, IndexType indexType) throws IOException {
        this.DIMENSION = dimension;
        this.INDEX_TYPE = indexType;
        this.INDEX_TYPE_OFFSET = this.INDEX_TYPE == IndexType.ANGULAR ? 4 : 8;
        this.K_NODE_HEADER_STYLE = this.INDEX_TYPE == IndexType.ANGULAR ? 12 : 16;
        this.MIN_LEAF_SIZE = this.DIMENSION + 2;
        this.NODE_SIZE = this.K_NODE_HEADER_STYLE + 4 * this.DIMENSION;
        this.roots = new ArrayList();
        this.load(filename);
    }

    public ANNIndex(int dimension, String filename) throws IOException {
        this(dimension, filename, IndexType.ANGULAR);
    }

    private void load(String filename) throws IOException {
        RandomAccessFile memoryMappedFile = new RandomAccessFile(filename, "r");
        int fileSize = (int)memoryMappedFile.length();
        this.annBuf = memoryMappedFile.getChannel().map(FileChannel.MapMode.READ_ONLY, 0L, fileSize);
        this.annBuf.order(ByteOrder.LITTLE_ENDIAN);
        int m = -1;
        for (int i = fileSize - this.NODE_SIZE; i >= 0; i -= this.NODE_SIZE) {
            int k = this.annBuf.getInt(i);
            if (m != -1 && k != m) break;
            this.roots.add(i);
            m = k;
        }
    }

    @Override
    public void getNodeVector(int nodeOffset, float[] v) {
        for (int i = 0; i < this.DIMENSION; ++i) {
            v[i] = this.annBuf.getFloat(nodeOffset + this.K_NODE_HEADER_STYLE + i * 4);
        }
    }

    @Override
    public void getItemVector(int itemIndex, float[] v) {
        this.getNodeVector(itemIndex * this.NODE_SIZE, v);
    }

    private float getNodeBias(int nodeOffset) {
        return this.annBuf.getFloat(nodeOffset + 4);
    }

    @Override
    public final float[] getItemVector(int itemIndex) {
        return this.getNodeVector(itemIndex * this.NODE_SIZE);
    }

    public float[] getNodeVector(int nodeOffset) {
        float[] v = new float[this.DIMENSION];
        this.getNodeVector(nodeOffset, v);
        return v;
    }

    private static float norm(float[] u) {
        float n = 0.0f;
        for (float x : u) {
            n += x * x;
        }
        return (float)Math.sqrt(n);
    }

    private static float euclideanDistance(float[] u, float[] v) {
        float[] diff = new float[u.length];
        for (int i = 0; i < u.length; ++i) {
            diff[i] = u[i] - v[i];
        }
        return ANNIndex.norm(diff);
    }

    public static float cosineMargin(float[] u, float[] v) {
        double d = 0.0;
        for (int i = 0; i < u.length; ++i) {
            d += (double)(u[i] * v[i]);
        }
        return (float)(d / (double)(ANNIndex.norm(u) * ANNIndex.norm(v)));
    }

    public static float euclideanMargin(float[] u, float[] v, float bias) {
        float d = bias;
        for (int i = 0; i < u.length; ++i) {
            d += u[i] * v[i];
        }
        return d;
    }

    private static boolean isZeroVec(float[] v) {
        for (int i = 0; i < v.length; ++i) {
            if (v[i] == 0.0f) continue;
            return false;
        }
        return true;
    }

    @Override
    public final List<Integer> getNearest(float[] queryVector, int nResults) {
        PriorityQueue<PQEntry> pq = new PriorityQueue<PQEntry>(this.roots.size() * 4);
        float kMaxPriority = 1.0E30f;
        for (int r : this.roots) {
            pq.add(new PQEntry(1.0E30f, r));
        }
        HashSet<Integer> nearestNeighbors = new HashSet<Integer>();
        while (nearestNeighbors.size() < this.roots.size() * nResults && !pq.isEmpty()) {
            PQEntry top = (PQEntry)pq.poll();
            int topNodeOffset = top.nodeOffset;
            int nDescendants = this.annBuf.getInt(topNodeOffset);
            float[] v = this.getNodeVector(topNodeOffset);
            if (ANNIndex.isZeroVec(v)) continue;
            if (nDescendants == 1) {
                nearestNeighbors.add(topNodeOffset / this.NODE_SIZE);
                continue;
            }
            if (nDescendants <= this.MIN_LEAF_SIZE) {
                for (int i = 0; i < nDescendants; ++i) {
                    int j = this.annBuf.getInt(topNodeOffset + this.INDEX_TYPE_OFFSET + i * 4);
                    nearestNeighbors.add(j);
                }
                continue;
            }
            float margin = this.INDEX_TYPE == IndexType.ANGULAR ? ANNIndex.cosineMargin(v, queryVector) : ANNIndex.euclideanMargin(v, queryVector, this.getNodeBias(topNodeOffset));
            int childrenMemOffset = topNodeOffset + this.INDEX_TYPE_OFFSET;
            int lChild = this.NODE_SIZE * this.annBuf.getInt(childrenMemOffset);
            int rChild = this.NODE_SIZE * this.annBuf.getInt(childrenMemOffset + 4);
            pq.add(new PQEntry(-margin, lChild));
            pq.add(new PQEntry(margin, rChild));
        }
        ArrayList<PQEntry> sortedNNs = new ArrayList<PQEntry>();
        int i = 0;
        Iterator i$ = nearestNeighbors.iterator();
        while (i$.hasNext()) {
            int nn = (Integer)i$.next();
            float[] v = this.getItemVector(nn);
            if (ANNIndex.isZeroVec(v)) continue;
            sortedNNs.add(new PQEntry(this.INDEX_TYPE == IndexType.ANGULAR ? ANNIndex.cosineMargin(v, queryVector) : -ANNIndex.euclideanDistance(v, queryVector), nn));
        }
        Collections.sort(sortedNNs);
        ArrayList<Integer> result = new ArrayList<Integer>(nResults);
        for (i = 0; i < nResults && i < sortedNNs.size(); ++i) {
            result.add(((PQEntry)sortedNNs.get(i)).nodeOffset);
        }
        return result;
    }

    public static void main(String[] args) throws IOException {
        String indexPath = args[0];
        int dimension = Integer.parseInt(args[1]);
        IndexType indexType = null;
        if (args[2].toLowerCase().equals("angular")) {
            indexType = IndexType.ANGULAR;
        } else if (args[2].toLowerCase().equals("euclidean")) {
            indexType = IndexType.EUCLIDEAN;
        } else {
            throw new RuntimeException("wrong index type specified");
        }
        int queryItem = Integer.parseInt(args[3]);
        ANNIndex annIndex = new ANNIndex(dimension, indexPath, indexType);
        float[] u = annIndex.getItemVector(queryItem);
        System.out.printf("vector[%d]: ", queryItem);
        for (float x : u) {
            System.out.printf("%2.2f ", Float.valueOf(x));
        }
        System.out.printf("\n", new Object[0]);
        List<Integer> nearestNeighbors = annIndex.getNearest(u, 10);
        for (int nn : nearestNeighbors) {
            float[] v = annIndex.getItemVector(nn);
            System.out.printf("%d %d %f\n", queryItem, nn, Float.valueOf(indexType == IndexType.ANGULAR ? ANNIndex.cosineMargin(u, v) : ANNIndex.euclideanDistance(u, v)));
        }
    }

    private class PQEntry
    implements Comparable<PQEntry> {
        private float margin;
        private int nodeOffset;

        PQEntry(float margin, int nodeOffset) {
            this.margin = margin;
            this.nodeOffset = nodeOffset;
        }

        @Override
        public int compareTo(PQEntry o) {
            return Float.compare(o.margin, this.margin);
        }
    }
}

