package com.shitu.epathmap.utils.voice.aws;

import android.content.Context;
import android.media.AudioManager;
import android.media.MediaPlayer;
import android.util.Log;

import com.amazonaws.auth.CognitoCachingCredentialsProvider;
import com.amazonaws.regions.Regions;
import com.amazonaws.services.polly.AmazonPollyPresigningClient;
import com.amazonaws.services.polly.model.DescribeVoicesRequest;
import com.amazonaws.services.polly.model.DescribeVoicesResult;
import com.amazonaws.services.polly.model.OutputFormat;
import com.amazonaws.services.polly.model.SynthesizeSpeechPresignRequest;
import com.amazonaws.services.polly.model.Voice;
import com.shitu.epathmap.utils.voice.SpeechListener;
import com.shitu.epathmap.utils.voice.Speecher;

import java.io.IOException;
import java.net.URL;
import java.util.List;
import java.util.Locale;

public class AwsSpeecher implements Speecher {
    // Cognito pool ID. Pool needs to be unauthenticated pool with

    // Amazon Polly permissions.
    private String COGNITO_POOL_ID;
    // Region of Amazon Polly.
    Regions MY_REGION;


    private URL presignedSynthesizeSpeechUrl;
    private String TAG = "AwsSpeecher";

    private Context context;

    // Media player
    MediaPlayer mediaPlayer;
    // Backend resources
    private AmazonPollyPresigningClient client;


    private Voice enVoice;
    private Voice zhVoice;
    private SpeechListener speechListener;

    public AwsSpeecher(Context context, List<String> keys) {
        this.context = context.getApplicationContext();

        COGNITO_POOL_ID = keys.get(0);
        MY_REGION = Regions.fromName(keys.get(1));

        new Thread(){
            @Override
            public void run() {
                initPolly();
            }
        }.start();
        setupNewMediaPlayer();

    }



    private void initPolly() {


        // Initialize the Amazon Cognito credentials provider.
        CognitoCachingCredentialsProvider credentialsProvider = new CognitoCachingCredentialsProvider(
                context,
                COGNITO_POOL_ID,
                MY_REGION
        );

        // Create a client that supports generation of presigned URLs.
        client = new AmazonPollyPresigningClient(credentialsProvider);


        // Create describe voices request.
        DescribeVoicesRequest describeVoicesRequest = new DescribeVoicesRequest();

// Synchronously ask Amazon Polly to describe available TTS voices.
        DescribeVoicesResult describeVoicesResult = client.describeVoices(describeVoicesRequest);
        List<Voice> voices = describeVoicesResult.getVoices();


        enVoice = null;
        for (Voice voice : voices) {
            if("Salli".equals(voice.getName())){
                enVoice = voice;
            }else if("Zhiyu".equals(voice.getName())){
                zhVoice = voice;
            }
        }


    }


    void setupNewMediaPlayer() {
        mediaPlayer = new MediaPlayer();
        mediaPlayer.setOnCompletionListener(new MediaPlayer.OnCompletionListener() {
            @Override
            public void onCompletion(MediaPlayer mp) {
                if(speechListener!=null){
                    speechListener.onSpeechFinish();
                }
                mp.release();
                setupNewMediaPlayer();
            }
        });
        mediaPlayer.setOnPreparedListener(new MediaPlayer.OnPreparedListener() {
            @Override
            public void onPrepared(MediaPlayer mp) {
                if(speechListener!=null) {
                    speechListener.onSpeechStart();
                }
                mp.start();
            }
        });
        mediaPlayer.setOnErrorListener(new MediaPlayer.OnErrorListener() {
            @Override
            public boolean onError(MediaPlayer mp, int what, int extra) {
                if(speechListener!=null){
                    speechListener.onError();
                }
                return false;
            }
        });
    }


    public void playVoice(String textToRead) {


        // Create speech synthesis request.
        SynthesizeSpeechPresignRequest synthesizeSpeechPresignRequest =
                new SynthesizeSpeechPresignRequest()
                        // Set text to synthesize.
                        .withText(textToRead)
                        // Set voice selected by the user.
                        .withVoiceId(Locale.getDefault().getLanguage().equals("zh")? zhVoice.getId():enVoice.getId())
                        // Set format to MP3.
                        .withOutputFormat(OutputFormat.Mp3);

        // Get the presigned URL for synthesized speech audio stream.
        URL presignedSynthesizeSpeechUrl =
                client.getPresignedSynthesizeSpeechUrl(synthesizeSpeechPresignRequest);

        Log.i(TAG, "Playing speech from presigned URL: " + presignedSynthesizeSpeechUrl);

        // Create a media player to play the synthesized audio stream.
        if (mediaPlayer.isPlaying()) {
            mediaPlayer.stop();
            mediaPlayer.release();
            setupNewMediaPlayer();
        }
        mediaPlayer.setAudioStreamType(AudioManager.STREAM_MUSIC);

        try {
            // Set media player's data source to previously obtained URL.
            mediaPlayer.setDataSource(presignedSynthesizeSpeechUrl.toString());
        } catch (IOException e) {
            Log.e(TAG, "Unable to set data source for the media player! " + e.getMessage());
        } catch (IllegalStateException e){
            e.printStackTrace();
        }

        // Start the playback asynchronously (since the data source is a network stream).
        try {
            mediaPlayer.prepareAsync();
        }catch (IllegalStateException e){

        }
    }

    @Override
    public int speak(String text) {
        playVoice(text);
        return 0;
    }

    @Override
    public int speak(String text, String utteranceld) {
        playVoice(text);
        return 0;
    }

    @Override
    public int pause() {
        if(mediaPlayer!=null && mediaPlayer.isPlaying()){
            mediaPlayer.pause();
        }
        return 0;
    }

    @Override
    public int resume() {
        if(mediaPlayer!=null){
            mediaPlayer.start();
        }
        return 0;
    }

    @Override
    public int stop() {
        if(mediaPlayer!=null){
            mediaPlayer.stop();
            mediaPlayer.release();
            setupNewMediaPlayer();
        }
        return 0;
    }

    @Override
    public void release() {
        if(mediaPlayer!=null){
            mediaPlayer.release();
            setupNewMediaPlayer();
        }
    }

    @Override
    public void setSpeechListener(SpeechListener speechListener) {
        this.speechListener = speechListener;
    }
}
