package com.shitu.epathmap.ui;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Typeface;
import android.os.Handler;
import android.os.Message;

import com.sails.engine.SAILS;
import com.sails.engine.SAILSMapView;
import com.sails.engine.core.model.GeoPoint;
import com.sails.engine.overlay.ListOverlay;
import com.sails.engine.overlay.OrientationMarker;
import com.sails.engine.overlay.TextOverlay;

import java.io.InputStream;
import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.List;

/**
 * author:lfei
 * 次楼层路径预览
 * 请于PathRoutingManager中的onTotalDistanceRefresh上呼叫updatePath，本类别将自动处理所有路径预览细节
 */

public class GhostDirectionGuide {
    final ListOverlay ghostOverlay;
    private Paint defaultTextStroke;
    private Paint defaultTextFill;
    private Context context;
    final private SAILSMapView sailsMapView;
    private OrientationMarker marker;
    private TextOverlay text;

    private List<SAILS.GeoNode> path;
    private int oneRunTime = 5;
    private int repeat = 1;
    private int delay = 0;
    private String title;
    private double DELTA_LENGTH = 0.15;
    private int currentFloor = Integer.MAX_VALUE;
    private int notificationDistance = 8;

    /**
     * 设置次楼层切换距离点
     *
     * @param distance 单位为米
     * @return
     */
    public GhostDirectionGuide setNotifcationDistance(int distance) {
        notificationDistance = distance;
        return this;
    }

    /**
     * 设定动画速度
     *
     * @param factor factor=1为正常速度，factor愈高速度愈快
     * @return
     */
    public GhostDirectionGuide setSpeedFactor(float factor) {
        DELTA_LENGTH = 0.15 * factor;
        return this;
    }

    /**
     * 开始路径预览动画
     *
     * @return
     */
    public GhostDirectionGuide start() {
        if (!generateNextFloorPath())
            return this;
        if (text == null)
            this.text = new TextOverlay(new GeoPoint(0, 0), title, defaultTextFill, defaultTextStroke);
        else
            text.setText(title);
        refreshHandler.start();
        return this;

    }

    /**
     * @param drawableId 设定动画图标的资源id
     * @param dps        设定图标大小
     * @return
     */
    public GhostDirectionGuide setMarker(int drawableId, int dps) {
        Bitmap bitmap = resize(getLocalBitmap(context, drawableId), dps);
        this.marker = new OrientationMarker(new GeoPoint(0, 0), bitmap);
        this.marker.setTransparency(128);
        return this;
    }

    /**
     * @param delay 设定路径预览动画延迟秒数，单位为秒
     * @return
     */
    public GhostDirectionGuide setDelay(int delay) {
        this.delay = delay;
        return this;
    }

    /**
     * @param repeatTimes 设定路径动画重复次数
     * @return
     */
    public GhostDirectionGuide setRepeatTimes(int repeatTimes) {
        this.repeat = repeatTimes;
        return this;
    }

    /**
     * @param path 设定路径节点，本类别将自动抓取次楼层的节点资讯
     * @return
     */
    public GhostDirectionGuide setPath(List<SAILS.GeoNode> path) {
        this.path = path;
        return this;
    }

    /**
     * @param oneRunTime 设定预览路径动画一次的时间，单位为秒
     * @return
     */
    public GhostDirectionGuide setOneRunTime(int oneRunTime) {
        this.oneRunTime = oneRunTime;
        return this;
    }

    public GhostDirectionGuide(Context context, SAILSMapView sailsMapView) {
        this.context = context;
        this.sailsMapView = sailsMapView;
        ghostOverlay = new ListOverlay();
        final float scale = context.getResources().getDisplayMetrics().density;
        int pixels = (int) (18 * scale + 0.5f);
        defaultTextFill = new Paint(Paint.ANTI_ALIAS_FLAG);
        defaultTextFill.setColor(Color.DKGRAY);
        defaultTextFill.setStyle(Paint.Style.FILL);
        defaultTextFill.setTextSize(pixels);
        defaultTextFill.setTextAlign(Paint.Align.CENTER);
        defaultTextFill.setAlpha(128);
        defaultTextFill.setStrokeWidth(6);
        defaultTextFill.setTypeface(Typeface.create(Typeface.DEFAULT, Typeface.BOLD));

        defaultTextStroke = new Paint(Paint.ANTI_ALIAS_FLAG);
        defaultTextStroke.setColor(Color.WHITE);
        defaultTextStroke.setTextSize(pixels);
        defaultTextStroke.setTextAlign(Paint.Align.CENTER);
        defaultTextStroke.setStyle(Paint.Style.STROKE);
        defaultTextStroke.setAlpha(128);
        defaultTextStroke.setStrokeWidth(8);
        defaultTextStroke.setTypeface(Typeface.create(Typeface.DEFAULT, Typeface.BOLD));

    }

    private boolean generateNextFloorPath() {
        if (path == null || path.size() <= 2)
            return false;
        int initfloorNumber = path.get(0).floornumber;
        int saveFloorNumber = Integer.MAX_VALUE;
        List<SAILS.GeoNode> nextFloorPath = new ArrayList<>();
        for (int i = 1; i < path.size(); i++) {
            if (path.get(i).floornumber == initfloorNumber)
                continue;
            if (saveFloorNumber == Integer.MAX_VALUE) {
                //若为电梯则不显示鬼影导航A
                if (path.get(i).BelongsRegion != null &&
                        "elevator".equals(path.get(i).BelongsRegion.subtype)) {
                    initfloorNumber = path.get(i).floornumber;
                }

                saveFloorNumber = path.get(i).floornumber;
                title = path.get(i).floorDesc + " 路径预览 -   -   -   -   -   -  ";
            } else if (path.get(i).floornumber != saveFloorNumber)
                break;
            nextFloorPath.add(path.get(i));
        }
        if (nextFloorPath.size() >= 2) {
            List<SAILS.GeoNode> equalLengthPath = new ArrayList<>();
            for (int i = 0; i < nextFloorPath.size() - 1; i++) {
                double len = computeDistance(
                        nextFloorPath.get(i).longitude,
                        nextFloorPath.get(i).latitude,
                        nextFloorPath.get(i + 1).longitude,
                        nextFloorPath.get(i + 1).latitude);
                if (len < 0.1) {
                    equalLengthPath.add(nextFloorPath.get(i));
                    continue;
                }
                int nodes = (int) (len / DELTA_LENGTH);
                double deltaLon = (nextFloorPath.get(i + 1).longitude - nextFloorPath.get(i).longitude) / nodes;
                double deltaLat = (nextFloorPath.get(i + 1).latitude - nextFloorPath.get(i).latitude) / nodes;
                for (int j = 0; j < nodes; j++) {
                    SAILS.GeoNode geoNode = new SAILS.GeoNode(
                            nextFloorPath.get(i).longitude + j * deltaLon,
                            nextFloorPath.get(i).latitude + j * deltaLat);
                    geoNode.floornumber = nextFloorPath.get(i).floornumber;
                    equalLengthPath.add(geoNode);
                }
            }
            path = equalLengthPath;
            return true;
        }
        return false;
    }

    RefreshHandler refreshHandler = new RefreshHandler(this);

    public void updatePath(List<SAILS.GeoNode> pathNodes) {
        if (pathNodes == null || pathNodes.size() < 1)
            return;
        int initFloor = pathNodes.get(0).floornumber;
        double len = 0;
        for (int i = 0; i < pathNodes.size() - 1; i++) {
            if (pathNodes.get(i + 1).floornumber != initFloor)
                break;
            len += computeDistance(
                    pathNodes.get(i).longitude,
                    pathNodes.get(i).latitude,
                    pathNodes.get(i + 1).longitude,
                    pathNodes.get(i + 1).latitude);


        }
        if (currentFloor != initFloor) {
            stopAnimation();
        }
        if (len < notificationDistance && currentFloor != initFloor) {
            currentFloor = initFloor;
            path = pathNodes;
            start();

        }
    }

    static class RefreshHandler extends Handler {
        boolean enable = false;
        WeakReference<GhostDirectionGuide> mG;
        private int index;
        private int count;
        private long lastTimeStamp;
        private int transparency;

        RefreshHandler(GhostDirectionGuide ghostDirectionGuide) {
            this.mG = new WeakReference<>(ghostDirectionGuide);
        }

        void start() {
            count = 0;
            index = 0;
            transparency = 0;
            GhostDirectionGuide ghostDirectionGuide = mG.get();
            if (ghostDirectionGuide == null)
                return;
            enable = true;
            lastTimeStamp = System.currentTimeMillis() + ghostDirectionGuide.delay * 1000;
            if (!ghostDirectionGuide.sailsMapView.getDynamicOverlays()
                    .contains(ghostDirectionGuide.ghostOverlay)) {
                ghostDirectionGuide.sailsMapView.getDynamicOverlays()
                        .add(ghostDirectionGuide.ghostOverlay);
            }
            if (!ghostDirectionGuide.ghostOverlay.getOverlayItems()
                    .contains(ghostDirectionGuide.marker)) {
                ghostDirectionGuide.ghostOverlay.getOverlayItems().add(ghostDirectionGuide.marker);
                ghostDirectionGuide.ghostOverlay.getOverlayItems().add(ghostDirectionGuide.text);
            }
            sleep(ghostDirectionGuide.delay);

        }

        @Override
        public void handleMessage(Message msg) {
            GhostDirectionGuide ghostDirectionGuide = mG.get();
            if (ghostDirectionGuide == null)
                return;
            if (ghostDirectionGuide.path.size() > index) {
                double bearAngle = SAILS.GetBearDegree(
                        ghostDirectionGuide.path.get(index).longitude,
                        ghostDirectionGuide.path.get(index).latitude,
                        ghostDirectionGuide.path.get(index + 1).longitude,
                        ghostDirectionGuide.path.get(index + 1).latitude);
                ghostDirectionGuide.marker.setRotation((float) bearAngle);
                ghostDirectionGuide.marker.setGeoPoint(
                        new GeoPoint(ghostDirectionGuide.path.get(index).latitude
                                , ghostDirectionGuide.path.get(index).longitude));
                ghostDirectionGuide.text.setGeoPoint(
                        new GeoPoint(ghostDirectionGuide.path.get(index).latitude
                                , ghostDirectionGuide.path.get(index).longitude));
                ghostDirectionGuide.marker.setTransparency(transparency);
                ghostDirectionGuide.defaultTextFill.setAlpha(transparency);
                ghostDirectionGuide.defaultTextStroke.setAlpha(transparency);
                ghostDirectionGuide.text.setPaintFill(ghostDirectionGuide.defaultTextFill);
                ghostDirectionGuide.text.setPaintStroke(ghostDirectionGuide.defaultTextStroke);
                if (System.currentTimeMillis() - lastTimeStamp < 2000) {
                    transparency += 5;
                    if (transparency >= 128)
                        transparency = 128;
                }
                if (index > ghostDirectionGuide.path.size() - 3
                        || System.currentTimeMillis() - lastTimeStamp > ghostDirectionGuide.oneRunTime * 1000) {
                    index = 0;
                    count++;
                    transparency = 0;
                    lastTimeStamp = System.currentTimeMillis();
                } else {
                    if (System.currentTimeMillis() - lastTimeStamp > ghostDirectionGuide.oneRunTime * 1000 - 20 * 50 * 2) {
                        transparency -= 5;
                        if (transparency < 0)
                            transparency = 0;
                    }
                    index++;
                }
            }
            if (count == ghostDirectionGuide.repeat) {
                ghostDirectionGuide.stopAnimation();
            }
            ghostDirectionGuide.sailsMapView.invalidate();
            if (enable)
                sleep(10);

        }

        public void sleep(long delayMillis) {
            this.removeMessages(0);
            sendMessageDelayed(obtainMessage(0), delayMillis);
        }
    }

    public void stopAnimation() {
        refreshHandler.enable = false;
        sailsMapView.getDynamicOverlays().remove(ghostOverlay);
        if (marker != null)
            marker.setGeoPoint(new GeoPoint(0, 0));
        if (text != null)
            text.setGeoPoint(new GeoPoint(0, 0));
        ghostOverlay.getOverlayItems().remove(marker);
        ghostOverlay.getOverlayItems().remove(text);
    }

    static float computeDistance(double lon1, double lat1, double lon2, double lat2) {
        // Based on http://www.ngs.noaa.gov/PUBS_LIB/inverse.pdf
        // using the "Inverse Formula" (section 4)

        int MAXITERS = 20;
        // Convert lat/long to radians
        lat1 *= Math.PI / 180.0;
        lat2 *= Math.PI / 180.0;
        lon1 *= Math.PI / 180.0;
        lon2 *= Math.PI / 180.0;

        double a = 6378137.0; // WGS84 major axis
        double b = 6356752.3142; // WGS84 semi-major axis
        double f = (a - b) / a;
        double aSqMinusBSqOverBSq = (a * a - b * b) / (b * b);

        double L = lon2 - lon1;
        double A = 0.0;
        double U1 = Math.atan((1.0 - f) * Math.tan(lat1));
        double U2 = Math.atan((1.0 - f) * Math.tan(lat2));

        double cosU1 = Math.cos(U1);
        double cosU2 = Math.cos(U2);
        double sinU1 = Math.sin(U1);
        double sinU2 = Math.sin(U2);
        double cosU1cosU2 = cosU1 * cosU2;
        double sinU1sinU2 = sinU1 * sinU2;

        double sigma = 0.0;
        double deltaSigma = 0.0;
        double cosSqAlpha = 0.0;
        double cos2SM = 0.0;
        double cosSigma = 0.0;
        double sinSigma = 0.0;
        double cosLambda = 0.0;
        double sinLambda = 0.0;

        double lambda = L; // initial guess
        for (int iter = 0; iter < MAXITERS; iter++) {
            double lambdaOrig = lambda;
            cosLambda = Math.cos(lambda);
            sinLambda = Math.sin(lambda);
            double t1 = cosU2 * sinLambda;
            double t2 = cosU1 * sinU2 - sinU1 * cosU2 * cosLambda;
            double sinSqSigma = t1 * t1 + t2 * t2; // (14)
            sinSigma = Math.sqrt(sinSqSigma);
            cosSigma = sinU1sinU2 + cosU1cosU2 * cosLambda; // (15)
            sigma = Math.atan2(sinSigma, cosSigma); // (16)
            double sinAlpha = (sinSigma == 0) ? 0.0 :
                    cosU1cosU2 * sinLambda / sinSigma; // (17)
            cosSqAlpha = 1.0 - sinAlpha * sinAlpha;
            cos2SM = (cosSqAlpha == 0) ? 0.0 :
                    cosSigma - 2.0 * sinU1sinU2 / cosSqAlpha; // (18)

            double uSquared = cosSqAlpha * aSqMinusBSqOverBSq; // defn
            A = 1 + (uSquared / 16384.0) * // (3)
                    (4096.0 + uSquared *
                            (-768 + uSquared * (320.0 - 175.0 * uSquared)));
            double B = (uSquared / 1024.0) * // (4)
                    (256.0 + uSquared *
                            (-128.0 + uSquared * (74.0 - 47.0 * uSquared)));
            double C = (f / 16.0) *
                    cosSqAlpha *
                    (4.0 + f * (4.0 - 3.0 * cosSqAlpha)); // (10)
            double cos2SMSq = cos2SM * cos2SM;
            deltaSigma = B * sinSigma * // (6)
                    (cos2SM + (B / 4.0) *
                            (cosSigma * (-1.0 + 2.0 * cos2SMSq) -
                                    (B / 6.0) * cos2SM *
                                            (-3.0 + 4.0 * sinSigma * sinSigma) *
                                            (-3.0 + 4.0 * cos2SMSq)));

            lambda = L +
                    (1.0 - C) * f * sinAlpha *
                            (sigma + C * sinSigma *
                                    (cos2SM + C * cosSigma *
                                            (-1.0 + 2.0 * cos2SM * cos2SM))); // (11)

            double delta = (lambda - lambdaOrig) / lambda;
            if (Math.abs(delta) < 1.0e-12) {
                break;
            }
        }

        float distance = (float) (b * A * (sigma - deltaSigma));

        return distance;
    }

    private Bitmap getLocalBitmap(Context con, int resourceId) {
        InputStream inputStream = con.getResources()
                .openRawResource(resourceId);
        return BitmapFactory.decodeStream(inputStream, null,
                getBitmapOptions(1));
    }

    private BitmapFactory.Options getBitmapOptions(int scale) {
        BitmapFactory.Options options = new BitmapFactory.Options();
        // options.inPurgeable = true;
        // options.inInputShareable = true;
        options.inSampleSize = scale;
        return options;
    }

    private Bitmap resize(Bitmap b, int dps) {

        float scale = 1;
        if (context.getResources() != null)
            scale = context.getResources().getDisplayMetrics().density;
        int pixels = (int) (dps * scale + 0.5f);
//        Bitmap b = ((BitmapDrawable)image).getBitmap();
        return Bitmap.createScaledBitmap(b, pixels, pixels, false);
//        return new BitmapDrawable(getResources(),bitmapResized);
    }

}
