/*
 * Decompiled with CFR 0.152.
 */
package com.sensorberg.sdk.location;

import android.content.Context;
import android.content.SharedPreferences;
import android.database.SQLException;
import android.location.Location;
import android.os.Bundle;
import android.os.Looper;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import com.google.android.gms.common.api.GoogleApiClient;
import com.google.android.gms.common.api.PendingResult;
import com.google.android.gms.common.api.ResultCallback;
import com.google.android.gms.common.api.Status;
import com.google.android.gms.location.Geofence;
import com.google.android.gms.location.GeofencingRequest;
import com.google.android.gms.location.LocationListener;
import com.google.android.gms.location.LocationRequest;
import com.google.android.gms.location.LocationServices;
import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;
import com.sensorberg.sdk.Logger;
import com.sensorberg.sdk.location.GeofenceData;
import com.sensorberg.sdk.location.GeofenceListener;
import com.sensorberg.sdk.location.GeofenceReceiver;
import com.sensorberg.sdk.location.GeofenceStorage;
import com.sensorberg.sdk.location.LocationStorage;
import com.sensorberg.sdk.location.PlayServiceManager;
import com.sensorberg.sdk.settings.SettingsManager;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.UUID;

public class GeofenceManager
implements GeofenceListener {
    private Context context;
    private SettingsManager settings;
    private SharedPreferences prefs;
    private Gson gson;
    private GeofenceStorage storage;
    private PlayServiceManager play;
    private List<GeofenceListener> listeners = new ArrayList<GeofenceListener>();
    private HashMap<String, String> entered;
    private Location previous;
    private Location current;
    private boolean updating = false;
    private boolean enabled = false;
    private boolean registered = false;
    private final boolean playServicesNotAvailable;
    private final GoogleApiClient.ConnectionCallbacks connectionCallbacks = new GoogleApiClient.ConnectionCallbacks(){

        public void onConnected(@Nullable Bundle bundle) {
            if (GeofenceManager.this.playServicesNotAvailable) {
                return;
            }
            if (GeofenceManager.this.enabled) {
                Logger.log.geofence("Event: Play services connection");
                GeofenceManager.this.requestSingleUpdate();
            }
        }

        public void onConnectionSuspended(int i) {
        }
    };
    private final LocationListener locationListener = new LocationListener(){

        public void onLocationChanged(Location incoming) {
            if (GeofenceManager.this.playServicesNotAvailable) {
                return;
            }
            if (incoming != null) {
                GeofenceManager.this.current = incoming;
                GeofenceManager.this.storeLastKnown(GeofenceManager.this.current);
                Logger.log.geofence("Update: location change at " + incoming);
                if (GeofenceManager.this.trigger(incoming)) {
                    GeofenceManager.this.removeGeofences(incoming);
                }
            }
        }
    };

    public GeofenceManager(Context context, SettingsManager settings, SharedPreferences prefs, Gson gson, PlayServiceManager play) {
        this.context = context;
        this.prefs = prefs;
        this.gson = gson;
        this.play = play;
        this.settings = settings;
        boolean bl = this.playServicesNotAvailable = play == null;
        if (this.playServicesNotAvailable) {
            return;
        }
        this.entered = this.loadEntered();
        this.storage = new GeofenceStorage(context, settings, prefs);
        play.addListener(this.connectionCallbacks);
        this.previous = this.restorePrevious();
        this.current = this.restoreLastKnown();
        if (this.storage.getCount() > 0) {
            Logger.log.geofence("Geofences restored from DB");
            this.enable();
        }
    }

    public void clear() {
        if (this.playServicesNotAvailable) {
            return;
        }
        this.previous = null;
        this.storePrevious(this.previous);
        this.entered = new HashMap();
        this.onFencesChanged(new ArrayList<String>());
    }

    private void enable() {
        if (this.playServicesNotAvailable) {
            return;
        }
        if (!this.enabled) {
            this.enabled = true;
            Logger.log.geofence("Enable GEOFENCING: Geofences appeared");
            this.play.connect();
            this.requestLocationUpdates();
        }
    }

    private void disable() {
        if (this.playServicesNotAvailable) {
            return;
        }
        if (this.enabled && this.storage.getCount() == 0) {
            this.enabled = false;
            Logger.log.geofence("Disable GEOFENCING: No geofences in layout");
            this.removeLocationUpdates();
            this.play.disconnect();
        }
    }

    public void onFencesChanged(List<String> fences) {
        if (this.playServicesNotAvailable) {
            return;
        }
        this.prefs.edit().putLong("com.sensorberg.preferences.lastDbUpdated", System.currentTimeMillis()).apply();
        Logger.log.geofence("Update: layout change");
        this.storage.updateFences(fences);
        if (fences.size() == 0 && this.storage.getCount() == 0) {
            this.disable();
        } else {
            this.enable();
        }
        this.registered = false;
        if (this.storage.getCount() <= 100) {
            this.requestSingleUpdate();
        }
        if (this.trigger(this.current)) {
            this.removeGeofences(this.current);
        }
    }

    public boolean shouldForceUpdate() {
        long lastUpdate = this.prefs.getLong("com.sensorberg.preferences.lastDbUpdated", 0L);
        return System.currentTimeMillis() - lastUpdate > this.settings.getLayoutUpdateInterval();
    }

    public void ping() {
        if (this.playServicesNotAvailable) {
            return;
        }
        Logger.log.geofence("Update: ping at " + this.current);
        if (this.trigger(this.current)) {
            this.removeGeofences(this.current);
        }
    }

    @Override
    public void onLocationChanged(Location incoming) {
        this.locationListener.onLocationChanged(incoming);
    }

    public void onGeofenceNotAvailable() {
        if (this.playServicesNotAvailable) {
            return;
        }
        if (this.enabled) {
            Logger.log.geofence("Event: Location state changed");
            this.registered = false;
            this.requestSingleUpdate();
        }
    }

    private boolean trigger(Location location) {
        if (!this.enabled) {
            Logger.log.geofence("Deny: No geofences in layout");
            return false;
        }
        if (!this.play.isGeofencingAvailable()) {
            Logger.log.geofenceError("Deny: Service is not available", null);
            return false;
        }
        if (!this.play.isConnected()) {
            this.play.connect();
            Logger.log.geofenceError("Deny: Service is not connected, will retry", null);
            return false;
        }
        if (this.updating) {
            Logger.log.geofence("Deny: Already updating");
            return false;
        }
        if (this.storage.getCount() <= 100) {
            if (this.registered) {
                Logger.log.geofence("Deny: Below 100 and all registered");
                return false;
            }
            Logger.log.geofence("Allow: Below 100 and not registered");
            return true;
        }
        if (this.previous == null) {
            if (location == null) {
                Logger.log.geofence("Deny: No location available");
                return false;
            }
            Logger.log.geofence("Allow: New location at " + location);
            return true;
        }
        if (location != null) {
            float change = this.previous.distanceTo(location);
            if (change < (float)this.storage.getRadius()) {
                Logger.log.geofence("Deny: Location change too small, was: " + change + "m, needed: " + this.storage.getRadius() + "m");
                return false;
            }
            Logger.log.geofence("Allow: Location change large enough, was: " + change + "m, needed: " + this.storage.getRadius() + "m");
            return true;
        }
        Logger.log.geofence("Allow: Just in case, at " + location);
        return true;
    }

    private void removeGeofences(final Location location) {
        this.updating = true;
        try {
            LocationServices.GeofencingApi.removeGeofences(this.play.getClient(), GeofenceReceiver.getGeofencePendingIntent(this.context)).setResultCallback((ResultCallback)new ResultCallback<Status>(){

                public void onResult(@NonNull Status status) {
                    if (status.isSuccess()) {
                        GeofenceManager.this.registerGeofences(location);
                    } else {
                        GeofenceManager.this.onGeofencesFailed(null, status.getStatusCode());
                    }
                }
            });
        }
        catch (IllegalStateException | SecurityException ex) {
            this.onGeofencesFailed(ex, 0);
        }
    }

    private void registerGeofences(final Location location) {
        List<GeofencingRequest> requests = this.getGeofencingRequests(location);
        if (requests.isEmpty()) {
            this.onGeofencesRemoved(location);
            return;
        }
        try {
            for (final GeofencingRequest request : requests) {
                LocationServices.GeofencingApi.addGeofences(this.play.getClient(), request, GeofenceReceiver.getGeofencePendingIntent(this.context)).setResultCallback((ResultCallback)new ResultCallback<Status>(){

                    public void onResult(@NonNull Status status) {
                        if (status.isSuccess()) {
                            GeofenceManager.this.onGeofencesAdded(location, request.getGeofences(), request.getInitialTrigger());
                        } else {
                            GeofenceManager.this.onGeofencesFailed(null, status.getStatusCode());
                        }
                    }
                });
            }
        }
        catch (IllegalStateException | SecurityException ex) {
            this.onGeofencesFailed(ex, 0);
        }
    }

    private void onGeofencesAdded(Location location, List<Geofence> geofences, int initialTrigger) {
        this.registered = true;
        this.updating = false;
        this.previous = location;
        this.storePrevious(this.previous);
        Logger.log.geofence("Successfully added " + geofences.size() + " geofences, initial trigger: " + initialTrigger);
        this.requestLocationUpdates();
    }

    private void onGeofencesRemoved(Location location) {
        this.registered = true;
        this.updating = false;
        this.previous = location;
        this.storePrevious(this.previous);
        Logger.log.geofence("No geofences around, nothing tracked at " + location);
        this.disable();
    }

    private void onGeofencesFailed(Exception ex, int status) {
        this.updating = false;
        if (ex != null) {
            Logger.log.geofenceError("Failed to add geofences, error code: " + status, ex);
        } else {
            Logger.log.geofenceError("Failed to add geofences, error code: " + status, ex);
        }
    }

    @Override
    public void onGeofenceEvent(GeofenceData geofenceData, boolean entry, String pairingIdNotUsedHere) {
        if (this.playServicesNotAvailable) {
            return;
        }
        String pairingId = this.entered.get(geofenceData.getFence());
        if (entry) {
            if (pairingId != null) {
                Logger.log.geofence("Duplicate entry, skipping geofence: " + geofenceData.getFence());
                return;
            }
            pairingId = UUID.randomUUID().toString();
            this.entered.put(geofenceData.getFence(), pairingId);
            this.saveEntered(this.entered);
        } else {
            if (pairingId == null) {
                Logger.log.geofence("Duplicate exit, skipping geofence: " + geofenceData.getFence());
                return;
            }
            this.entered.remove(geofenceData.getFence());
            this.saveEntered(this.entered);
        }
        this.notifyListeners(geofenceData, entry, pairingId);
    }

    private List<GeofencingRequest> getGeofencingRequests(Location location) {
        ArrayList<GeofencingRequest> result = new ArrayList<GeofencingRequest>(2);
        try {
            GeofencingRequest.Builder builder;
            HashMap<String, Geofence> triggerEnter = this.storage.getGeofences(location);
            HashMap<String, Geofence> triggerExit = new HashMap<String, Geofence>();
            Iterator<Map.Entry<String, String>> iterator = this.entered.entrySet().iterator();
            while (iterator.hasNext()) {
                Map.Entry<String, String> inside = iterator.next();
                Geofence temp = triggerEnter.get(inside.getKey());
                if (temp != null) {
                    triggerEnter.remove(inside.getKey());
                    triggerExit.put(inside.getKey(), temp);
                    continue;
                }
                iterator.remove();
                Logger.log.geofenceError("Exit event for " + inside.getKey() + " not triggered, probably not relevant anymore", null);
            }
            this.saveEntered(this.entered);
            if (triggerEnter.size() > 0) {
                builder = new GeofencingRequest.Builder();
                builder.setInitialTrigger(1);
                builder.addGeofences(new ArrayList<Geofence>(triggerEnter.values()));
                result.add(builder.build());
            }
            if (triggerExit.size() > 0) {
                builder = new GeofencingRequest.Builder();
                builder.setInitialTrigger(2);
                builder.addGeofences(new ArrayList(triggerExit.values()));
                result.add(builder.build());
            }
        }
        catch (SQLException ex) {
            Logger.log.geofenceError("Can't build geofencing reqest", ex);
        }
        return result;
    }

    private void requestSingleUpdate() {
        LocationRequest request = LocationRequest.create();
        request.setNumUpdates(1);
        request.setPriority(102);
        try {
            PendingResult result = LocationServices.FusedLocationApi.requestLocationUpdates(this.play.getClient(), request, this.locationListener, Looper.myLooper());
            result.setResultCallback((ResultCallback)new ResultCallback<Status>(){

                public void onResult(@NonNull Status status) {
                    if (!status.isSuccess()) {
                        Logger.log.geofenceError("Requesting single location update failed " + status.getStatusCode(), null);
                    }
                }
            });
        }
        catch (SecurityException ex) {
            Logger.log.geofence("Insufficient permission for location updates");
        }
        catch (IllegalStateException ex) {
            Logger.log.geofence("Play service client is not connected");
        }
    }

    private void requestLocationUpdates() {
        if (this.storage.getCount() <= 100) {
            return;
        }
        final float displacement = Math.max(this.storage.getRadius(), this.settings.getGeofenceMinUpdateDistance());
        final long time = Math.max((long)displacement / (long)this.settings.getGeofenceMaxDeviceSpeed() * 1000L, this.settings.getGeofenceMinUpdateTime());
        LocationRequest request = LocationRequest.create();
        request.setPriority(102);
        request.setInterval(time);
        request.setFastestInterval(time / 2L);
        request.setMaxWaitTime(2L * time);
        request.setSmallestDisplacement(displacement);
        try {
            PendingResult result = LocationServices.FusedLocationApi.requestLocationUpdates(this.play.getClient(), request, GeofenceReceiver.getLocationPendingIntent(this.context));
            result.setResultCallback((ResultCallback)new ResultCallback<Status>(){

                public void onResult(@NonNull Status status) {
                    if (!status.isSuccess()) {
                        Logger.log.geofenceError("Requesting location updates failed " + status.getStatusCode(), null);
                    } else {
                        Logger.log.geofence("Registered location updates with time: " + time + " displacement: " + displacement);
                    }
                }
            });
        }
        catch (SecurityException ex) {
            Logger.log.geofence("Insufficient permission for location updates");
        }
        catch (IllegalStateException ex) {
            Logger.log.geofence("Play service client is not connected");
        }
    }

    private void removeLocationUpdates() {
        try {
            PendingResult result = LocationServices.FusedLocationApi.removeLocationUpdates(this.play.getClient(), GeofenceReceiver.getLocationPendingIntent(this.context));
            result.setResultCallback((ResultCallback)new ResultCallback<Status>(){

                public void onResult(@NonNull Status status) {
                    if (!status.isSuccess()) {
                        Logger.log.geofenceError("Removing location updates failed " + status.getStatusCode(), null);
                    }
                }
            });
        }
        catch (SecurityException ex) {
            Logger.log.geofence("Insufficient permission for location updates");
        }
        catch (IllegalStateException ex) {
            Logger.log.geofence("Play service client is not connected");
        }
    }

    public void onLocationPermissionGranted() {
    }

    public void addListener(GeofenceListener listener) {
        if (this.playServicesNotAvailable) {
            return;
        }
        for (GeofenceListener previous : this.listeners) {
            if (previous != listener) continue;
            return;
        }
        this.listeners.add(listener);
    }

    public void removeListener(GeofenceListener listener) {
        if (this.playServicesNotAvailable) {
            return;
        }
        Iterator<GeofenceListener> iterator = this.listeners.iterator();
        while (iterator.hasNext()) {
            GeofenceListener existing = iterator.next();
            if (existing != listener) continue;
            iterator.remove();
            return;
        }
    }

    private void notifyListeners(GeofenceData geofenceData, boolean entry, String pairingId) {
        for (GeofenceListener listener : this.listeners) {
            listener.onGeofenceEvent(geofenceData, entry, pairingId);
        }
    }

    private void saveEntered(HashMap<String, String> entered) {
        String key = "com.sensorberg.preferences.enteredGeofencesMap";
        this.prefs.edit().putString(key, this.gson.toJson(entered)).apply();
    }

    private HashMap<String, String> loadEntered() {
        String key = "com.sensorberg.preferences.enteredGeofencesMap";
        String json = this.prefs.getString(key, null);
        if (json == null || json.isEmpty()) {
            return new HashMap<String, String>();
        }
        Type type = new TypeToken<HashMap<String, String>>(){}.getType();
        return (HashMap)this.gson.fromJson(json, type);
    }

    private void storeLastKnown(Location location) {
        LocationStorage.save(this.gson, this.prefs, "com.sensorberg.preferences.lastKnownLocation", location);
    }

    private Location restoreLastKnown() {
        return LocationStorage.load(this.gson, this.prefs, "com.sensorberg.preferences.lastKnownLocation");
    }

    private void storePrevious(Location location) {
        LocationStorage.save(this.gson, this.prefs, "com.sensorberg.preferences.previousLocation", location);
    }

    private Location restorePrevious() {
        return LocationStorage.load(this.gson, this.prefs, "com.sensorberg.preferences.previousLocation");
    }

    public void setRegistered(boolean registered) {
        this.registered = registered;
    }
}

