package com.segway.robot.mobile.sdk.connectivity;

import android.os.Handler;
import android.os.HandlerThread;
import android.util.Log;

import com.segway.robot.sdk.base.bind.ServiceBinder;
import com.segway.robot.sdk.baseconnectivity.ByteMessage;
import com.segway.robot.sdk.baseconnectivity.MessageConnection.ConnectionStateListener;
import com.segway.robot.sdk.baseconnectivity.MessageConnection.MessageListener;
import com.segway.robot.sdk.baseconnectivity.MessageRouter;
import com.segway.robot.sdk.baseconnectivity.MessageUtil;

import org.apache.mina.core.future.ConnectFuture;
import org.apache.mina.core.session.IdleStatus;
import org.apache.mina.core.session.IoSession;
import org.apache.mina.filter.codec.ProtocolCodecFilter;
import org.apache.mina.filter.codec.serialization.ObjectSerializationCodecFactory;
import org.apache.mina.filter.keepalive.KeepAliveFilter;
import org.apache.mina.filter.keepalive.KeepAliveMessageFactory;
import org.apache.mina.filter.logging.LoggingFilter;
import org.apache.mina.transport.socket.nio.NioSocketConnector;
import org.json.JSONException;
import org.json.JSONObject;

import java.net.InetSocketAddress;
import java.util.Set;

/**
 * Created by gaofeng on 2016/4/20.
 */
public class MobileMessageSocket {

    private static final String TAG = "MobileMessageSocket";
    private MobileMessageHandler mMobileMessageHandler = null;
    private NioSocketConnector mConnector = null;
    private static MobileMessageSocket mMobileMessageSocket = null;
    private ConnectFuture mConnectFuture = null;
    private IoSession mIoSession = null;
    private String mPackageName = null;
    private String mIp = null;
    private HandlerThread connectThread = null;
    private Handler mHandler = null;
    private ConnectRunnable mConnectRunnable = null;
    private boolean isSessionOpened = false;
    private static Object mLock = new Object();
    private ServiceBinder.BindStateListener mBindStateListener = null;
    private boolean isRepeatedConnect = false;

    private static final int PORT = 7798;
    private static final int CONNECTION_TIMEOUT = 30 * 1000;//30s
    private static final int RECONNECT_INTERVAL = 10 * 1000;//10s

    public static synchronized MobileMessageSocket getInstance() {
        if (mMobileMessageSocket == null) {
            mMobileMessageSocket = new MobileMessageSocket();
        }
        return mMobileMessageSocket;
    }

    private MobileMessageSocket() {
        Log.d(TAG, "MobileMessageSocket: init");
        mMobileMessageHandler = new MobileMessageHandler();
        connectThread = new HandlerThread("ConnectThread");
        connectThread.start();
        mHandler = new Handler(connectThread.getLooper());
        // Create ClientIo
        mConnector = new NioSocketConnector();
        mConnector.getFilterChain().addLast("logger", new LoggingFilter());
        ObjectSerializationCodecFactory codecFactory = new ObjectSerializationCodecFactory();
        codecFactory.setDecoderMaxObjectSize(MessageUtil.MAX_CODER_LENGTH);
        codecFactory.setEncoderMaxObjectSize(MessageUtil.MAX_CODER_LENGTH);
        mConnector.getFilterChain().addLast("codec", new ProtocolCodecFilter(codecFactory));
        KeepAliveMessageFactory heartBeatFactory = new MobileKeepAliveImpl();
        KeepAliveFilter heartBeatFilter = new KeepAliveFilter(heartBeatFactory, IdleStatus.BOTH_IDLE);
        heartBeatFilter.setForwardEvent(true);
        heartBeatFilter.setRequestInterval(MessageUtil.HEART_BEAT_INTERVAL);//mina default 60s
        heartBeatFilter.setRequestTimeout(MessageUtil.HEART_BEAT_TIMEOUT);//mina default 30s
        mConnector.getFilterChain().addLast("heart", heartBeatFilter);
        // set time out
        mConnector.setConnectTimeoutMillis(CONNECTION_TIMEOUT);
        mConnector.getSessionConfig().setWriteTimeout(MessageUtil.WRITE_TIMEOUT);
        mConnector.setHandler(mMobileMessageHandler);

        setSessionStateChange(sessionStateCallback);
    }

    private SessionStateCallback sessionStateCallback = new SessionStateCallback() {
        @Override
        public void onSessionClosed() {
            Log.d(TAG, "onSessionClosed: ");
            mIoSession = null;
            isSessionOpened = false;
            if (isRepeatedConnect) {
                mBindStateListener.onUnbind(MessageError.REPEATED_CONNECTION);
            } else {
                mBindStateListener.onUnbind(MessageError.SESSION_CLOSED);
                if (!mConnector.isDisposed() && mMobileMessageHandler != null) {
                    if (!isConnected()) {
                        connect();
                    }
                }
            }
        }

        @Override
        public void onSessionOpened() {
            isSessionOpened = true;
            isRepeatedConnect = false;
            mBindStateListener.onBind();
            mHandler.removeCallbacks(mConnectRunnable);
            synchronized (mLock) {
                mIoSession = mConnectFuture.getSession();
            }
        }

        @Override
        public void onSessionRepeated(IoSession ioSession) {
            isRepeatedConnect = true;
            ioSession.closeNow();
        }
    };

    public synchronized void disconnect() {
        if (mConnectRunnable != null) {
            mHandler.removeCallbacks(mConnectRunnable);
            Log.d(TAG, "disconnect: remove connect runnable");
        }

        new Thread(new Runnable() {
            @Override
            public void run() {
                synchronized (mLock) {
                    if (!mConnector.isDisposed() && mIoSession != null) {
                        mIoSession.getCloseFuture().awaitUninterruptibly(100);
                        mIoSession = null;
                        isSessionOpened = false;
                        mConnector.dispose();
                        Log.d(TAG, "dispose thread run: over");
                    }
                }
            }
        }).start();

        mMobileMessageHandler = null;
        mHandler = null;
        mConnectRunnable = null;
        connectThread.quit();
        mMobileMessageSocket = null;

        Log.d(TAG, "disconnect: MobileMessageSocket set to null!");
    }

    private class ConnectRunnable implements Runnable {
        private String mIp = null;

        public ConnectRunnable(String ip) {
            mIp = ip;
        }

        @Override
        public void run() {
            synchronized (mLock) {
                if (isSessionOpened || mIoSession != null) {
                    return;
                }

                if (!mConnector.isDisposed()) {
                    mConnectFuture = mConnector.connect(new InetSocketAddress(mIp, PORT));
                    mConnectFuture.awaitUninterruptibly();
                    mHandler.postDelayed(this, RECONNECT_INTERVAL);
                }
                Log.d(TAG, "run: ConnectRunnable");
            }
        }
    }

    public synchronized void initSocketClient(String packageName, String ip, ServiceBinder.BindStateListener listener) {
        if (mConnector == null) {
            throw new NullPointerException("Connector is null pointer!");
        }

        mIp = ip;
        mPackageName = packageName;
        mBindStateListener = listener;
    }

    public synchronized void connect() {
        if (isSessionOpened || mIoSession != null) {
            return;
        }

        if (mConnectRunnable == null) {
            mConnectRunnable = new ConnectRunnable(mIp);
        }

        mHandler.post(mConnectRunnable);
    }

    public boolean isConnected() {
        return mMobileMessageHandler != null && mMobileMessageHandler.isConnected();
    }

    public void sendMessage(String to, StringMessage body) throws MobileException {
        JSONObject jsonMessage = new JSONObject();
        try {
            jsonMessage.put(MessageUtil.MSG_ID, body.getId());
            jsonMessage.put(MessageUtil.TIMESTAMP, MessageUtil.getTimestamp());
            jsonMessage.put(MessageUtil.MSG_FROM, mPackageName);
            jsonMessage.put(MessageUtil.MSG_TO, to);
            jsonMessage.put(MessageUtil.MSG_BODY, body.getContent());
        } catch (JSONException e) {
            throw MobileException.getJsonException(e.getMessage(), e);
        }

        if (mMobileMessageHandler != null && mConnector != null) {
            mMobileMessageHandler.sendMessage(jsonMessage.toString());
        }
    }

    public void sendMessage(String to, BufferMessage body) throws MobileException {
        ByteMessage byteMessage = new ByteMessage(body.getId(), MessageUtil.getTimestamp(), mPackageName, to, body.getContent());
        byte[] bufferMessage = byteMessage.packByteMessage(byteMessage);

        if (mMobileMessageHandler != null && mConnector != null) {
            mMobileMessageHandler.sendMessage(bufferMessage);
        }
    }

    private void setSessionStateChange(SessionStateCallback callback) {
        mMobileMessageHandler.setSessionChangedCallback(callback);
    }

    public void setConnectionListener(MessageRouter.MessageConnectionListener listener) {
        mMobileMessageHandler.setMessageConnectionListener(listener);
    }

    public void setMetaData(String packageName, Set set) {
        mMobileMessageHandler.setMetaData(packageName, set);
    }

    public void setListenersReady(String from, String to, ConnectionStateListener connectionStateListener, MessageListener messageListener) {
        mMobileMessageHandler.setListenersReady(from, to, connectionStateListener, messageListener);
    }
}