package com.segway.robot.mobile.sdk.connectivity;

import android.util.Log;

import com.segway.robot.sdk.baseconnectivity.ByteMessage;
import com.segway.robot.sdk.baseconnectivity.ConnectionInfo;
import com.segway.robot.sdk.baseconnectivity.MessageConnection.ConnectionStateListener;
import com.segway.robot.sdk.baseconnectivity.MessageConnection.MessageListener;
import com.segway.robot.sdk.baseconnectivity.MessageRouter;
import com.segway.robot.sdk.baseconnectivity.MessageUtil;

import org.apache.mina.core.service.IoHandlerAdapter;
import org.apache.mina.core.session.IdleStatus;
import org.apache.mina.core.session.IoSession;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Created by gaofeng on 2016/5/9.
 */
public class MobileMessageHandler extends IoHandlerAdapter {

    private static final String TAG = "MobileMessageHandler";
    private static final String MOBILE_HEARTBEAT = "mobile";

    private boolean isSessionEnable = false;
    private MessageRouter.MessageConnectionListener mMessageConnectionListener = null;
    private Map<String, String> mRobotMetaDataMap = new HashMap<>();//<hosts,list<data>>
    private IoSession mIoSession = null;
    private SessionStateCallback mSessionStateCallback = null;
    private ConnectionInfo mConnectionInfo = new ConnectionInfo();
    private Map<String, MobileMessageConnection> mMessageConnectionMap = new HashMap<>();//<robot name , info>
    private List<ConnectionReadyInfo> mConnectionsReadyInfo = new ArrayList<>();//<robot name, info>

    @Override
    public void messageReceived(IoSession session, Object message) throws Exception {
        super.messageReceived(session, message);
        Log.d(TAG, "Mobile messageReceived: " + message.toString());
        if (message instanceof byte[]) {
            ByteMessage byteMessage = new ByteMessage();
            byteMessage.unpackMessage((byte[]) message, byteMessage);
            String from = byteMessage.mFrom;
            String to = byteMessage.mTo;
            int id = byteMessage.mId;
            long timestamp = byteMessage.mTimestamp;
            byte[] messageByte = byteMessage.mMessage;
            Log.d(TAG, "id=" + id + ";timestamp=" + timestamp + ";from=" + from + ";to=" + to + ";m-length=" + messageByte.length);
            for (ConnectionReadyInfo info : mConnectionsReadyInfo) {
                if (info.getPackage().equals(to) && info.getPackageTo().equals(from)) {
                    MessageListener listener = info.getMessageListener();
                    if (listener != null) {
                        BufferMessage bufferMessage = new BufferMessage(timestamp, messageByte, id);
                        listener.onMessageReceived(bufferMessage);
                    }
                }
            }
        } else {
            JSONObject jsonObject = new JSONObject(message.toString());
            if (jsonObject.has(MessageUtil.ADD_NEW)) {//add new host
                String data = jsonObject.getString(MessageUtil.ADD_NEW);
                String from = jsonObject.getString(MessageUtil.MSG_FROM);
                Log.d(TAG, MessageUtil.ADD_NEW + "=" + data);
                mRobotMetaDataMap.put(from, data);
                if (mConnectionInfo.getMetadata().contains(from)) {
                    MobileMessageConnection messageConnection = new MobileMessageConnection(from);
                    mMessageConnectionMap.put(from, messageConnection);
                    mMessageConnectionListener.onConnectionCreated(messageConnection);
                }
            } else if (jsonObject.has(MessageUtil.REMOVE_HOST)) {//remove host
                String from = jsonObject.getString(MessageUtil.MSG_FROM);
                Log.d(TAG, MessageUtil.REMOVE_HOST + "=" + from);
                mRobotMetaDataMap.remove(from);
                ConnectionStateListener listener = mMessageConnectionMap.get(from).getConnectionStateListener();
                if (listener != null) {
                    listener.onClosed(MessageError.ROBOT_APP_DISCONNECTED);
                }
                mMessageConnectionMap.remove(from);

                synchronized (mConnectionsReadyInfo) {
                    Iterator<ConnectionReadyInfo> iterator = mConnectionsReadyInfo.iterator();
                    while (iterator.hasNext()) {
                        ConnectionReadyInfo info = iterator.next();
                        if (from.equals(info.getPackageTo())) {
                            iterator.remove();
                        }
                    }
                }
            } else if (jsonObject.has(MessageUtil.UPDATE_LIST)) {//update host app list
                JSONObject object = new JSONObject(jsonObject.getString(MessageUtil.UPDATE_LIST));
                mRobotMetaDataMap = MessageUtil.packJsonToMap(object);
                Log.d(TAG, MessageUtil.UPDATE_LIST + "=" + mRobotMetaDataMap.toString());
                for (String robot : mConnectionInfo.getMetadata()) {
                    if (mRobotMetaDataMap.containsKey(robot) && mRobotMetaDataMap.get(robot).contains(mConnectionInfo.getPackageName())) {
                        if (mMessageConnectionListener != null) {
                            MobileMessageConnection messageConnection = new MobileMessageConnection(robot);
                            mMessageConnectionMap.put(robot, messageConnection);
                            mMessageConnectionListener.onConnectionCreated(messageConnection);
                        }
                    }
                }
            } else if (jsonObject.has(MessageUtil.REMOVE_DEAD)) {
                String name = jsonObject.getString(MessageUtil.REMOVE_DEAD);
                String hostMeta = mRobotMetaDataMap.get(name);
                Log.d(TAG, "remove dead host " + name + ";" + hostMeta);
                if (hostMeta.contains(mConnectionInfo.getPackageName()) && mConnectionInfo.getMetadata().contains(name)) {
                    ConnectionStateListener listener = mMessageConnectionMap.get(name).getConnectionStateListener();
                    if (listener != null) {
                        listener.onClosed(MessageError.ROBOT_APP_DISCONNECTED);
                    }
                    mMessageConnectionMap.remove(name);
                }

                synchronized (mConnectionsReadyInfo) {
                    Iterator<ConnectionReadyInfo> iterator = mConnectionsReadyInfo.iterator();
                    while (iterator.hasNext()) {
                        ConnectionReadyInfo info = iterator.next();
                        if (name.equals(info.getPackageTo())) {
                            iterator.remove();
                        }
                    }
                }
                mRobotMetaDataMap.remove(name);
            } else if (jsonObject.has(MessageUtil.MESSAGE_LISTENER_READY)) {
                String from = jsonObject.getString(MessageUtil.MSG_FROM);
                String to = jsonObject.getString(MessageUtil.MESSAGE_LISTENER_READY);
                Log.d(TAG, MessageUtil.MESSAGE_LISTENER_READY + ": from= " + from + "; to=" + to);
                synchronized (mConnectionsReadyInfo) {
                    for (ConnectionReadyInfo info : mConnectionsReadyInfo) {
                        if (info.getPackage().equals(to) && info.getPackageTo().equals(from)) {
                            ConnectionStateListener listener = info.getConnectionStateListener();
                            if (listener != null) {
                                listener.onOpened();
                                Log.d(TAG, "on opened = " + from);
                            }
                        }
                    }
                }
            } else if (jsonObject.has(MessageUtil.REPEATED_CONNECTION)) {
                mSessionStateCallback.onSessionRepeated(session);
            } else {
                String from = jsonObject.getString(MessageUtil.MSG_FROM);
                String to = jsonObject.getString(MessageUtil.MSG_TO);
                String body = jsonObject.getString(MessageUtil.MSG_BODY);
                long time = jsonObject.getLong(MessageUtil.TIMESTAMP);
                int id = jsonObject.getInt(MessageUtil.MSG_ID);
                synchronized (mConnectionsReadyInfo) {
                    for (ConnectionReadyInfo connection : mConnectionsReadyInfo) {
                        if (from.equals(connection.getPackageTo()) && to.equals(connection.getPackage())) {
                            MessageListener listener = connection.getMessageListener();
                            if (listener != null) {
                                StringMessage stringMessage = new StringMessage(time, body, id);
                                listener.onMessageReceived(stringMessage);
                            }
                        }
                    }
                }
            }
        }
    }

    @Override
    public void messageSent(IoSession session, Object message) throws Exception {
        super.messageSent(session, message);

        if (message instanceof byte[]) {
            ByteMessage byteMessage = new ByteMessage();
            byteMessage.unpackMessage((byte[]) message, byteMessage);
            String from = byteMessage.mFrom;
            String to = byteMessage.mTo;
            int id = byteMessage.mId;
            long timestamp = byteMessage.mTimestamp;
            byte[] messageByte = byteMessage.mMessage;
            Log.d(TAG, "byte[] messageSent: id=" + id + ";timestamp=" + timestamp + ";from=" + from + ";to=" + to);
            for (MobileMessageConnection connection : mMessageConnectionMap.values()) {
                if (to.equals(connection.getName())) {
                    MobileMessageConnection.MessageListener listener = connection.getMessageListener();
                    if (listener != null) {
                        BufferMessage bufferMessage = new BufferMessage(timestamp, messageByte, id);
                        listener.onMessageSent(bufferMessage);
                    }
                }
            }
        } else {
            JSONObject json = new JSONObject(message.toString());
            if (checkMessageSent(json)) {    //to check message sent
                String to = json.getString(MessageUtil.MSG_TO);
                String from = json.getString(MessageUtil.MSG_FROM);
                String body = json.getString(MessageUtil.MSG_BODY);
                long timestamp = json.getLong(MessageUtil.TIMESTAMP);
                int id = json.getInt(MessageUtil.MSG_ID);
                Log.d(TAG, "String messageSent: id=" + id + ";timestamp=" + timestamp + ";from=" + from + ";to=" + to);
                for (MobileMessageConnection connection : mMessageConnectionMap.values()) {
                    if (to.equals(connection.getName())) {
                        MobileMessageConnection.MessageListener listener = connection.getMessageListener();
                        if (listener != null) {
                            StringMessage stringMessage = new StringMessage(timestamp, body, id);
                            listener.onMessageSent(stringMessage);
                        }
                    }
                }
            }
        }
    }

    @Override
    public void sessionOpened(IoSession session) throws Exception {
        super.sessionOpened(session);
        Log.d(TAG, "Mobile sessionOpened:" + session);
        isSessionEnable = true;
        mIoSession = session;
        mSessionStateCallback.onSessionOpened();
        mConnectionInfo.setIoSession(session);
        session.write(setSessionOpenMessage());
    }

    @Override
    public void sessionClosed(IoSession session) throws Exception {
        Log.w(TAG, "Mobile sessionClosed: " + session);
        super.sessionClosed(session);
        isSessionEnable = false;
        mSessionStateCallback.onSessionClosed();
        for (MobileMessageConnection connection : mMessageConnectionMap.values()) {
            connection.getConnectionStateListener().onClosed(MessageError.SESSION_CLOSED);
        }
        mMessageConnectionMap.clear();
        synchronized (mConnectionsReadyInfo) {
            mConnectionsReadyInfo.clear();
        }
    }

    @Override
    public void sessionIdle(IoSession session, IdleStatus status) throws Exception {
        super.sessionIdle(session, status);
    }

    @Override
    public void exceptionCaught(IoSession session, Throwable cause) throws Exception {
        super.exceptionCaught(session, cause);
        Log.e(TAG, "Mobile exceptionCaught: ", cause);
        throw MobileException.getIoSessionException("IoSession exception!", cause);
    }

    public boolean isConnected() {
        return isSessionEnable;
    }

    public void sendMessage(String message) {
        mIoSession.write(message);
    }

    public void sendMessage(byte[] message) {
        mIoSession.write(message);
    }

    public void setSessionChangedCallback(SessionStateCallback callback) {
        mSessionStateCallback = callback;
    }

    public void setMessageConnectionListener(MessageRouter.MessageConnectionListener listener) {
        mMessageConnectionListener = listener;
    }

    public void setMetaData(String packageName, Set set) {
        Log.d(TAG, "setMetaData: name=" + packageName + "set =" + set.toString());
        mConnectionInfo.setPackageName(packageName);
        mConnectionInfo.setMetadata(set);
    }

    private String setSessionOpenMessage() {
        JSONObject json = new JSONObject();
        try {
            json.put(MessageUtil.MSG_FROM, mConnectionInfo.getPackageName());
            json.put(MessageUtil.SESSION_OPEN, MessageUtil.packSetToString(mConnectionInfo.getMetadata()));
        } catch (JSONException e) {
            try {
                throw MobileException.getJsonException(e.getMessage(), e);
            } catch (MobileException mobileException) {
                Log.e(TAG, "setSessionMessage: " + mobileException.getMessage());
            }
        }
        return json.toString();
    }

    public void setListenersReady(String from, String to, ConnectionStateListener connectionStateListener, MessageListener messageListener) {
        Log.d(TAG, "setListenersReady() called with: " + "from = [" + from + "], to = [" + to + "]");
        ConnectionReadyInfo info = new ConnectionReadyInfo(from, to, connectionStateListener, messageListener);
        mConnectionsReadyInfo.add(info);
        sendListenerReadyInfo(from, to);
    }

    private void sendListenerReadyInfo(String packageName, String to) {
        if (isConnected()) {
            JSONObject object = new JSONObject();
            try {
                object.put(MessageUtil.TIMESTAMP, MessageUtil.getTimestamp());
                object.put(MessageUtil.MSG_FROM, packageName);
                object.put(MessageUtil.MESSAGE_LISTENER_READY, to);
            } catch (JSONException e) {
                Log.e(TAG, "Message is not json", e);
            }

            mConnectionInfo.getIoSession().write(object.toString());
        }
    }

    private boolean checkMessageSent(JSONObject json) {
        return !json.has(MOBILE_HEARTBEAT) && !json.has(MessageUtil.ADD_NEW) &&
                !json.has(MessageUtil.UPDATE_LIST) && !json.has(MessageUtil.REMOVE_HOST) &&
                !json.has(MessageUtil.REMOVE_DEAD) && !json.has(MessageUtil.MESSAGE_LISTENER_READY) && !json.has(MessageUtil.SESSION_OPEN);
    }
}