package com.seeq.link.sdk.utilities;

import java.text.DecimalFormat;
import java.time.Duration;
import java.time.format.DateTimeParseException;

public class TimeInterval {

    public TimeInterval(TimeInstant start, TimeInstant end) {
        this.startTime = start;
        this.endTime = end;
    }

    private final TimeInstant startTime;

    public TimeInstant getStartTime() {
        return this.startTime;
    }

    private final TimeInstant endTime;

    public TimeInstant getEndTime() {
        return this.endTime;
    }

    public static Duration parseFriendlyDuration(String durationString) {
        try {
            return Duration.parse(("PT" + durationString).toUpperCase());
        } catch (DateTimeParseException e) {
            throw new IllegalArgumentException(
                    String.format("'%s' not a valid duration. Examples of valid durations: '1h', '5m', '30s'",
                            durationString), e);
        }
    }

    public static String toFriendlyString(Duration duration) {
        String str = "";
        long hours = duration.toHours();
        if (hours != 0) {
            str += String.format("%dh", hours);
        }
        long minutes = duration.toMinutes() % 60;
        if (minutes != 0) {
            str += String.format("%dm", minutes);
        }
        double seconds = duration.getSeconds() % 60 + duration.getNano() / 1000000000d;
        if (seconds != 0) {
            DecimalFormat df = new DecimalFormat("#.######");
            str += String.format("%ss", df.format(seconds));
        }
        return str;
    }

    /**
     * Returns a string representation in the form HH:MM:SS, which is the same as the C# TimeSpan.ToString() method.
     *
     * @param duration
     *         The Duration object to convert to a string
     * @return A string in the form HH:MM:SS
     */
    public static String toFriendlyStringMatchDotNetTimeSpan(Duration duration) {
        StringBuilder str = new StringBuilder();
        if (duration.toHours() > 24) {
            str.append(duration.toHours() / 24);
            str.append(".");
        }
        str.append(String.format("%02d:%02d:%02d", duration.toHours() % 24, duration.toMinutes() % 60,
                duration.getSeconds() % 60));
        double fractionalSeconds = duration.getNano() / 1000000000d;
        if (fractionalSeconds != 0) {
            DecimalFormat df = new DecimalFormat("#.0000000");
            String fractionalSecondsString = df.format(fractionalSeconds);
            str.append(fractionalSecondsString.substring(fractionalSecondsString.lastIndexOf('.')));
        }
        return str.toString();
    }
}
