package com.seeq.link.sdk.utilities;

import java.time.Duration;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.google.common.math.LongMath;

import lombok.EqualsAndHashCode;

// Developer Note:  If the time conversions don't make sense to you, check your computer's time settings.
// Adjust Date and Time -> Change Time Zone -> Automatically adjust clock for Daylight Savings should be checked
// and was not that way on my Parallels Windows VM.  That setting influences the results.

/**
 * Representation of a Seeq timestamp which is a count of nanoseconds elapsed since Unix Epoch (January 1, 1970 UTC).
 */
@EqualsAndHashCode
public class TimeInstant {
    private final long timestamp;

    public static final TimeInstant MIN = new TimeInstant(Long.MIN_VALUE);
    public static Pattern ISO_TIMESTAMP_REGEX =
            // Note that the .* is present at the end of the RegEx due to Java adding the timezone in square brackets
            // as part of ZonedDateTime.toString(). (This is different than C#.)
            // See https://stackoverflow.com/questions/43634226/zoneddatetime-tostring-compatability-with-iso-8601
            Pattern.compile("(\\d\\d\\d\\d)-(\\d\\d)-(\\d\\d)T(\\d\\d):(\\d\\d):(\\d\\d)(\\.(\\d+))?Z.*");

    /**
     * Construct a new TimeInstant with the count of nanoseconds elapsed since Unix Epoch (January 1, 1970 UTC).
     *
     * @param timestamp
     *         Number of nanoseconds elapsed since Unix Epoch (January 1, 1970 UTC).
     */
    public TimeInstant(long timestamp) {
        this.timestamp = timestamp;
    }

    /**
     * Construct a new TimeInstant from a Java ZonedDateTime.
     *
     * @param dateTime
     *         The time that the TimeInstant is to represent.
     */
    public TimeInstant(ZonedDateTime dateTime) {
        this.timestamp = this.getNanosecondsSinceUnixEpoch(dateTime);
    }

    /**
     * Represents a Seeq timestamp which is a count of nanoseconds elapsed since Unix Epoch (January 1, 1970 UTC). Seeq
     * timestamps are always in UTC.
     *
     * @return the timestamp
     */
    public long getTimestamp() {
        return this.timestamp;
    }

    /**
     * Converts a Java ZonedDateTime to a Seeq timestamp which is a count of nanoseconds elapsed since Unix Epoch
     * (January 1, 1970 UTC).
     *
     * @param dateTime
     *         The Java ZonedDateTime that the timestamp is to represent.
     * @return Number of nanoseconds elapsed since Unix Epoch (January 1, 1970 UTC).
     */
    private long getNanosecondsSinceUnixEpoch(ZonedDateTime dateTime) {
        ZonedDateTime unixEpoch = ZonedDateTime.of(1970, 1, 1, 0, 0, 0, 0, ZoneId.of("UTC"));
        if (!dateTime.getZone().equals(ZoneId.of("UTC"))) {
            dateTime = dateTime.withZoneSameInstant(ZoneId.of("UTC"));
        }

        Duration duration = Duration.between(unixEpoch, dateTime);
        return duration.get(ChronoUnit.SECONDS) * 1_000_000_000 + duration.get(ChronoUnit.NANOS);
    }

    /**
     * Retrieves the TimeInstant as a Java UTC DateTime.
     *
     * @return The Java UTC DateTime representation of the TimeInstant.
     */
    public ZonedDateTime toDateTime() {
        ZonedDateTime returnValue = ZonedDateTime.of(1970, 1, 1, 0, 0, 0, 0, ZoneId.of("UTC"));
        returnValue = returnValue.plusNanos(this.timestamp);
        return returnValue;
    }

    /**
     * Converts a Seeq timestamp to a Java UTC DateTime. Seeq timestamps are a count of nanoseconds elapsed since Unix
     * Epoch (January 1, 1970 UTC).
     *
     * @param timestamp
     *         Number of nanoseconds elapsed since Unix Epoch (January 1, 1970 UTC).
     * @return The UTC DateTime representation of the Seeq timestamp.
     */
    public static ZonedDateTime timestampToDateTime(long timestamp) {
        TimeInstant instant = new TimeInstant(timestamp);
        return instant.toDateTime();
    }

    /**
     * Converts a Java ZonedDateTime to a Seeq timestamp which is a count of nanoseconds elapsed since Unix Epoch
     * (January 1, 1970 UTC).
     *
     * @param dateTime
     *         The Java DateTime that the timestamp is to represent.
     * @return The Seeq timestamp representation of the .Net DateTime.
     */
    public static long dateTimeToTimestamp(ZonedDateTime dateTime) {
        TimeInstant instant = new TimeInstant(dateTime);
        return instant.getTimestamp();
    }

    /**
     * Converts a Java LocalDateTime to a Seeq timestamp which is a count of nanoseconds elapsed since Unix Epoch
     * (January 1, 1970 UTC).
     *
     * @param dateTime
     *         The Java DateTime that the timestamp is to represent.
     * @return The Seeq timestamp representation of the .Net DateTime.
     */
    public static long dateTimeToTimestamp(LocalDateTime dateTime) {
        TimeInstant instant = new TimeInstant(dateTime.atZone(ZoneId.of("UTC")));
        return instant.getTimestamp();
    }

    /**
     * Round up a TimeInstant to a multiple of nanoseconds.<br>
     * Example: TimeInstant = 1232 ns. TimeInstant.roundUp(100) = 1300 ns<br>
     * Example: TimeInstant = -1232 ns. TimeInstant.roundUp(100)= -1200 ns
     *
     * @param multiple_ns
     *         The multiple of nanoseconds to round to such as 1, 10, 100, 1000 etc
     * @return The rounded result
     */
    public TimeInstant roundUp(long multiple_ns) {
        return new TimeInstant(roundTimestampUp(this.timestamp, multiple_ns));
    }

    /**
     * Round up a timestamp to a multiple of 1, 10, 100, etc.<br>
     * Example: roundTimestampUp( 1232,100) = 1300<br>
     * Example: roundTimestampUp(-1232,100) = -1200
     *
     * @param timestamp
     *         The timestamp to round
     * @param multiple
     *         The multiple to round to such as 1, 10, 100, 1000 etc
     * @return The rounded result
     */
    public static long roundTimestampUp(long timestamp, long multiple) {
        if (timestamp < 0) {
            return roundDown(timestamp, multiple);
        } else {
            return roundUp(timestamp, multiple);
        }
    }

    private static long roundUp(long timestamp, long multiple) {
        long delta = multiple - 1;

        if (timestamp < 0) {
            delta = -delta;
        }

        if (timestamp % multiple != 0) {
            return ((timestamp + delta) / multiple) * multiple;
        } else {
            return timestamp;
        }
    }

    /**
     * Round down a TimeInstant to a multiple of nanoseconds.<br>
     * Example: TimeInstant = 1232 ns. TimeInstant.roundUp(100) = 1200 ns<br>
     * Example: TimeInstant = -1232 ns. TimeInstant.roundUp(100)= -1300 ns
     *
     * @param multiple_ns
     *         The multiple of nanoseconds to round to such as 1, 10, 100, 1000 etc
     * @return The rounded result
     */
    public TimeInstant roundDown(long multiple_ns) {
        return new TimeInstant(roundTimestampDown(this.timestamp, multiple_ns));
    }

    /**
     * Subtract a nanosecond from this time.
     *
     * @return The time exactly 1 nanosecond previous to the time represented by this instance.
     * @throws ArithmeticException
     *         If this timestamp is already at the minimum representable time.
     */
    public TimeInstant decrement() {
        return new TimeInstant(LongMath.checkedSubtract(this.timestamp, 1));
    }

    /**
     * Add a nanosecond to this time.
     *
     * @return The time exactly 1 nanosecond after the time represented by this instance.
     * @throws ArithmeticException
     *         If this timestamp is already at the maximum representable time.
     */
    public TimeInstant increment() {
        return new TimeInstant(LongMath.checkedAdd(this.timestamp, 1));
    }

    /**
     * Round down a timestamp to a multiple of 1, 10, 100, etc.<br>
     * Example: roundTimestampDown( 1232,100) = 1200<br>
     * Example: roundTimestampDown(-1232,100) = -1300<br>
     *
     * @param timestamp
     *         The timestamp to round
     * @param multiple
     *         The multiple to round to such as 1, 10, 100, 1000 etc
     * @return The rounded result
     */
    public static long roundTimestampDown(long timestamp, long multiple) {
        if (timestamp < 0) {
            return roundUp(timestamp, multiple);
        } else {
            return roundDown(timestamp, multiple);
        }
    }

    private static long roundDown(long timestamp, long multiple) {
        if (timestamp % multiple != 0) {
            return (timestamp / multiple) * multiple;
        } else {
            return timestamp;
        }
    }

    @Override
    public String toString() {
        return this.toDateTime().format(DateTimeFormatter.ISO_INSTANT);
    }

    /**
     * Parses a string in the ISO format "2021-02-14T15:23:54.384758575Z" into a TimeInstant. Note
     * that only this string format is supported. Sub-nanosecond precision will be ignored (floored).
     *
     * @param str
     *         A string in ISO format "2021-02-14T15:23:54.384758575Z"
     * @return The TimeInstant corresponding to the specified string.
     */
    public static TimeInstant parseIso(String str) {
        Matcher match = ISO_TIMESTAMP_REGEX.matcher(str);
        if (!match.matches()) {
            throw new IllegalArgumentException(
                    "ISO timestamp string not recognized, must be in format 2021-02-14T15:23:54.384758575Z");
        }

        ZonedDateTime dateTime = ZonedDateTime.parse(str);

        return new TimeInstant(dateTime);
    }
}
