package com.seeq.link.sdk.utilities;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.seeq.utilities.exception.OperationCanceledException;
import com.seeq.utilities.process.StackTraceInfo;

import lombok.Data;
import lombok.extern.slf4j.Slf4j;

/**
 * A facility that manages a collection of background threads that need to be spawned and potentially all interrupted in
 * a controlled way.
 */
@Slf4j
public class ThreadCollection {
    // A timeout of -1 means we will never time out.
    public static final int NO_TIMEOUT = -1;

    // A request ID of -1 means it can't be interrupted by ID. Appserver-generated request IDs are >= 0.
    public static final int UNINTERRUPTABLE_ID = -1;

    private static final long DEFAULT_TIMEOUT_CHECK_INTERVAL = 60;
    private static final TimeUnit DEFAULT_TIMEOUT_CHECK_UNIT = TimeUnit.SECONDS;

    private final Object lockObj = new Object();
    private String id;
    private ScheduledExecutorService scheduledExecutorService;
    private final long timeoutCheckInterval;
    private final TimeUnit timeoutCheckUnit;

    public ThreadCollection() {
        this("Thread Collection ID not set");
    }

    public ThreadCollection(String id) {
        this(id, DEFAULT_TIMEOUT_CHECK_INTERVAL, DEFAULT_TIMEOUT_CHECK_UNIT);
    }

    @VisibleForTesting
    ThreadCollection(String id, long timeoutCheckInterval, TimeUnit timeoutCheckUnit) {
        this.id = id;
        this.timeoutCheckInterval = timeoutCheckInterval;
        this.timeoutCheckUnit = timeoutCheckUnit;
    }

    /**
     * Starts a new monitor thread for the thread collection if one isn't already running.
     */
    private void startMonitorIfNotRunning() {
        synchronized (this.lockObj) {
            if (this.scheduledExecutorService == null) {
                this.scheduledExecutorService = Executors.newSingleThreadScheduledExecutor(
                        new ThreadFactoryBuilder()
                                .setNameFormat("Seeq ThreadCollection [" + this.id + "] - Thread %d")
                                .build());
                this.scheduledExecutorService.scheduleAtFixedRate(() -> this.threads.values().forEach(threadInfo -> {
                    if (threadInfo.isTimedOut()) {
                        LOG.warn("Interrupting request {} which has run for longer than the timeout of {}ms",
                                threadInfo.getThread().getName(), threadInfo.getTimeoutMillis());
                        threadInfo.getThread().interrupt();
                    }
                }), this.timeoutCheckInterval, this.timeoutCheckInterval, this.timeoutCheckUnit);
            }
        }
    }

    /**
     * An identifier used in log output for this object.
     *
     * @return an identifier used in log output for this object
     */
    public String getID() {
        return this.id;
    }

    /**
     * An identifier used in log output for this object.
     *
     * @param id
     *         an identifier used in log output for this object
     */
    public void setID(String id) {
        this.id = id;
    }

    private final ConcurrentHashMap<Thread, ThreadInfo> threads = new ConcurrentHashMap<>();

    /**
     * Spawns a thread that executes the supplied callback.
     *
     * @param callback
     *         The callback that represents the task to be performed
     * @return the Thread object that was spawned
     */
    public Thread spawn(Runnable callback) {
        return this.spawn(callback, NO_TIMEOUT, UNINTERRUPTABLE_ID, Thread.NORM_PRIORITY);
    }

    /**
     * Spawns a thread that executes the supplied callback.
     *
     * @param callback
     *         The callback that represents the task to be performed
     * @param priority
     *         The desired thread priority; see {@link Thread#setPriority}
     * @return the Thread object that was spawned
     */
    public Thread spawn(Runnable callback, int priority) {
        return this.spawn(callback, NO_TIMEOUT, UNINTERRUPTABLE_ID, priority);
    }

    /**
     * Spawns a thread that executes the supplied callback.
     *
     * @param callback
     *         The callback that represents the task to be performed
     * @param timeoutMillis
     *         How long the request remains valid. After timeoutMillis milliseconds, the
     *         request should be interrupted, to free up resources for future requests.
     * @param requestId
     *         An identifier for the request which must be greater than or equal to 0 and unique to the connection. This
     *         ID allows for subsequent cancellation messages to be associated with the same request.
     * @return the Thread object that was spawned
     */
    public Thread spawn(Runnable callback, long timeoutMillis, long requestId) {
        return this.spawn(callback, timeoutMillis, requestId, Thread.NORM_PRIORITY);
    }

    /**
     * Spawns a thread that executes the supplied callback.
     *
     * @param callback
     *         The callback that represents the task to be performed
     * @param timeoutMillis
     *         How long the request remains valid. After timeoutMillis milliseconds, the
     *         request should be interrupted, to free up resources for future requests.
     * @param requestId
     *         An identifier for the request which must be greater than or equal to 0 and unique to the connection. This
     *         ID allows for subsequent cancellation messages to be associated with the same request.
     * @param priority
     *         The desired thread priority; see {@link Thread#setPriority}
     * @return the Thread object that was spawned
     */
    public Thread spawn(Runnable callback, long timeoutMillis, long requestId, int priority) {
        this.startMonitorIfNotRunning();

        Thread thread = new Thread(() -> {
            try {
                // Make sure our dynamic classloader is set for the current thread (otherwise Resource.getResource
                // may not work as expected, by default a different classloader being used)
                Thread.currentThread().setContextClassLoader(ClassPathUtilities.instance().getClassLoader());

                callback.run();
            } catch (Throwable e) {
                // Handle interrupt/cancellation exceptions with DEBUG logs instead of ERROR since they are expected
                if (e instanceof OperationCanceledException) {
                    LOG.debug("Request {} canceled", requestId);
                } else {
                    LOG.error("Error in thread:", e);
                }
            } finally {
                ThreadCollection.this.threads.remove(Thread.currentThread());
            }
        });

        thread.setPriority(priority);

        this.threads.put(thread, new ThreadInfo(thread, timeoutMillis, requestId));

        thread.start();

        return thread;
    }

    /**
     * Interrupt a request, given the identifier supplied when the request was created.
     *
     * Note: interrupting a request does not mean it will terminate immediately, or at all. It only means that the
     * interrupted flag will be set on the thread. Some connectors will watch for this flag and terminate the
     * request, but others may not.
     *
     * @param requestId
     *         the identifier supplied when the request was created
     */
    public void interrupt(long requestId) {
        this.threads.values().stream()
                .filter(threadInfo -> threadInfo.getRequestId() == requestId)
                .forEach(threadInfo -> {
                    LOG.info("Interrupting request with request ID {} due to cancellation or timeout", requestId);
                    threadInfo.getThread().interrupt();
                });
    }

    @Data
    private static class ThreadInfo {
        private final Thread thread;
        private final long timeoutMillis;
        private final long requestId;
        private final Stopwatch stopwatch;

        ThreadInfo(Thread thread, long timeoutMillis, long requestId) {
            this.thread = thread;
            this.timeoutMillis = timeoutMillis;
            this.requestId = requestId;
            this.stopwatch = new Stopwatch();
            this.stopwatch.start();
        }

        boolean isTimedOut() {
            return this.timeoutMillis >= 0
                    && this.stopwatch.elapsed(TimeUnit.MILLISECONDS) > this.timeoutMillis;
        }
    }

    /**
     * Shuts down all threads being managed in this collection and does not return until they are verified to have died.
     */
    public void shutDownAll() {
        synchronized (this.lockObj) {
            for (Thread thread : this.threads.keySet()) {
                this.shutdownThread(thread);
            }

            this.threads.clear();

            if (this.scheduledExecutorService != null) {
                LOG.debug("ThreadCollection[{}]: Shutting down monitor thread", this.id);
                this.scheduledExecutorService.shutdownNow();
                try {
                    this.scheduledExecutorService
                            .awaitTermination(DEFAULT_TIMEOUT_CHECK_INTERVAL, DEFAULT_TIMEOUT_CHECK_UNIT);
                } catch (InterruptedException e) {
                    // Ignore
                }
                this.scheduledExecutorService = null;
            }
        }
    }

    /**
     * The count of all background threads being managed by this collection.
     *
     * @return the count of all background threads
     */
    public int getCount() {
        return this.threads.size();
    }

    private void shutdownThread(Thread thread) {
        final int timeoutInMilliseconds = 10000;
        LOG.info("{} shutting down thread: {}", this.id, thread.getName());
        while (true) {
            thread.interrupt();
            try {
                thread.join(timeoutInMilliseconds);
                if (!thread.isAlive()) {
                    break;
                }

                LOG.info("{} continuing to try to shut down thread '{}'. Current stack trace:\n{}",
                        this.id, thread.getName(), StackTraceInfo.getFullStackTrace(thread.getStackTrace()));
            } catch (InterruptedException e) {
                break;
            }
        }
        LOG.info("{} successfully shut down thread: {}", this.id, thread.getName());
    }

    @VisibleForTesting
    protected boolean isShutdown() {
        return this.threads.isEmpty() && this.scheduledExecutorService == null;
    }
}
