package com.seeq.link.sdk.utilities;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;

import com.seeq.utilities.exception.RuntimeIOException;

import lombok.extern.slf4j.Slf4j;
import lombok.val;

/**
 * Utility for accessing secret configuration fields from external files.
 */
@Slf4j
public final class SecretsHelper {
    private static Path basePath;

    private SecretsHelper() {}

    /**
     * Gets the secret from the secretInputString. The input may contain a reference to a secret file in
     * the form SECRETS_FILE:/path/to/the/secrets/file.
     * The file may be also relative to the agent data_dir.
     *
     * @param secretInputString
     *         the input as it comes from the configuration file (can be null)
     * @return null if the input is null
     *         the input if it doesn't start with the prefix "SECRETS_FILE:"
     *         the string from file if the input has syntax "SECRETS_FILE:/path/to/the/credentials/file"
     * @throws RuntimeIOException
     *         if an I/O error occurs reading from the file or a malformed or unmappable byte sequence
     *         is read
     */
    public static String resolveSecret(String secretInputString) {
        if (secretInputString == null) {
            return null;
        }

        val SECRETS_MARKER_PREFIX = "SECRETS_FILE:";
        if (secretInputString.startsWith(SECRETS_MARKER_PREFIX)) {
            try {
                String secretFilePathString = secretInputString.substring(SECRETS_MARKER_PREFIX.length());
                Path path = Paths.get(secretFilePathString);

                if (! path.isAbsolute()) {
                    path = basePath.resolve(path);
                }

                return new String(Files.readAllBytes(path)).trim();
            } catch (IOException e) {
                throw new RuntimeIOException(e);
            }
        } else {
            return secretInputString;
        }
    }

    /**
     * Sets the base path for relative secret file resolution (e.g. SECRETS_FILE:relative/path).
     * This should be only called by DefaultAgent during initialization.
     */
    public static void setBasePath(Path newBasePath) {
        basePath = newBasePath;
    }
}
