package com.seeq.link.sdk.utilities;

import java.io.IOException;
import java.util.Collections;

import com.google.common.base.Charsets;
import com.google.common.io.Resources;
import com.vladsch.flexmark.ast.Node;
import com.vladsch.flexmark.ext.toc.TocExtension;
import com.vladsch.flexmark.html.HtmlRenderer;
import com.vladsch.flexmark.parser.Parser;
import com.vladsch.flexmark.util.options.MutableDataHolder;
import com.vladsch.flexmark.util.options.MutableDataSet;

public final class Markdown {
    private static final MutableDataHolder MARKDOWN_OPTIONS = new MutableDataSet()
            .set(HtmlRenderer.RENDER_HEADER_ID, true)
            .set(Parser.EXTENSIONS, Collections.singletonList(TocExtension.create()));

    private Markdown() {}

    /**
     * Compile a markdown formatted document to a user presentable html document with inlined css.
     *
     * @param markdown
     *         The document to render as html.
     * @return The html (with inline css) version of the provided markdown document.
     */
    public static String compileMarkdownToStyledHtml(String markdown) {
        try {
            String html = compileMarkdownToHtml(markdown);
            String css = Resources.toString(Resources.getResource("markdown.css"), Charsets.UTF_8);

            html = "<head><style>\n" + css + "\n</style></head>\n" +
                    "<article class=\"markdown-body\">\n" +
                    html +
                    "</article>";

            return html;
        } catch (IOException e) {
            throw new IllegalStateException("Internal error resolving the markdown css file");
        }
    }

    /**
     * Compile a markdown formatted document to an html document.
     *
     * @param markdown
     *         The document to render as html.
     * @return The html version of the provided markdown document.
     */
    public static String compileMarkdownToHtml(String markdown) {
        Parser parser = Parser.builder(MARKDOWN_OPTIONS).build();
        Node document = parser.parse(markdown);
        HtmlRenderer renderer = HtmlRenderer.builder(MARKDOWN_OPTIONS).build();

        return renderer.render(document);
    }
}
