package com.seeq.link.sdk.utilities;

import java.io.File;
import java.io.IOException;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.List;

import lombok.extern.slf4j.Slf4j;

/**
 * Utility to perform a file search using Glob syntax via {@link #find(String, Path)} static member. Refer to the
 * following article for syntax: https://docs.oracle.com/javase/tutorial/essential/io/fileOps.html#glob
 */
@Slf4j
public final class FileSystemGlob {

    private static List<Path> walkDirectoryTree(Path root) {
        return walkDirectoryTree(root, root.toString());
    }

    private static List<Path> walkDirectoryTree(Path root, String relativizeWith) {
        if (relativizeWith.charAt(relativizeWith.length() - 1) != File.separatorChar) {
            relativizeWith = relativizeWith + File.separatorChar;
        }

        final List<Path> fileList = new ArrayList<>();

        final String relativizeWithFinal = relativizeWith;
        try {
            Files.walkFileTree(root, new SimpleFileVisitor<Path>() {
                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
                    fileList.add(Paths.get(file.toString().replace(relativizeWithFinal, "")));
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) {
                    if (dir.equals(root) == false) {
                        fileList.add(Paths.get(dir.toString().replace(relativizeWithFinal, "")));
                    }
                    return FileVisitResult.CONTINUE;
                }
            });
        } catch (Exception e) {
            return new ArrayList<>();
        }

        return fileList;
    }

    /**
     * Finds all files in the system that match the globPattern. If the globPattern is not rooted, then
     * rootForRelativePaths will be used to establish the search root.
     *
     * @param globPattern
     *         the pattern to search for, which can include standard "glob" wildcards including * and **
     * @param rootForRelativePaths
     *         the root to use for relative paths
     * @return a list of matching paths
     */
    public static List<Path> find(String globPattern, Path rootForRelativePaths) {
        // Handle case of zero-length String
        if (globPattern.length() == 0) {
            return new ArrayList<>();
        }

        // Make sure we're only dealing with OS directory separators
        globPattern = globPattern.replaceAll("/", "\\" + File.separatorChar);

        String[] pieces = globPattern.split("\\" + File.separatorChar); // This is a regex
        List<String> rootPieces = new ArrayList<>();
        List<String> patternPieces = null;
        for (String piece : pieces) {
            // Stop at the first wildcard
            if (patternPieces != null || piece.contains("*") || piece.contains("?")) {
                if (patternPieces == null) {
                    patternPieces = new ArrayList<>();
                }

                patternPieces.add(piece);
            } else {
                rootPieces.add(piece);
            }
        }

        Path root = Paths.get(String.join(Character.toString(File.separatorChar), rootPieces));

        String pattern;
        if (patternPieces == null) {
            // No pattern elements were found-- this must be a hard path
            pattern = root.getFileName().toString();
            root = root.getParent();

            // In Java, Path.getParent() returns null, whereas C# Path.GetDirectoryName() returns "".
            if (root == null) {
                root = Paths.get("");
            }
        } else {
            // Note: The Java PathMatcher class MUST be given paths with slashes on all platforms.
            // i.e., you can't give it a path with backslashes on Windows.
            pattern = String.join("/", patternPieces);
        }

        if (root.isAbsolute() == false) {
            // Calling getCanonicalFile() removes all the (potential) ".." path pieces
            try {
                root = Paths.get(rootForRelativePaths.resolve(root).toFile().getCanonicalFile().toString());
            } catch (IOException e) {
                return new ArrayList<>();
            }
        }

        List<Path> potentialFiles = walkDirectoryTree(root);

        PathMatcher pathMatcher = FileSystems.getDefault().getPathMatcher("glob:" + pattern);
        LOG.info("Filter all items in '{}' via '{}' matcher", root, pattern);
        List<Path> filteredFiles = new ArrayList<>();
        for (Path potentialFile : potentialFiles) {
            if (pathMatcher.matches(potentialFile)) {
                filteredFiles.add(potentialFile);
            }
        }

        List<Path> finalFiles = new ArrayList<>();
        for (Path filteredFile : filteredFiles) {
            Path fullPath = root.resolve(filteredFile);
            finalFiles.add(fullPath);
            LOG.info("Found next matched file: '{}'", fullPath);
        }

        return finalFiles;
    }
}
