package com.seeq.link.sdk.utilities;

import java.time.Duration;
import java.time.Period;
import java.time.temporal.TemporalAmount;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public class DurationHelper {

    /**
     * Parse a configuration string (e.g. "30 min", "1 y") into a duration object.
     * If the provided duration string is not correct it defaults to the duration given in defaultDurationString
     *
     * @param durationString
     *         string to parse
     * @param defaultDurationString
     *         default value to use if the parsing fails
     * @param variableName
     *         name of the variable (for logging)
     * @return the duration as object and as configuration string
     */
    public static ParsedDuration parseConfigString(String durationString, String defaultDurationString,
            String variableName) {
        if (durationString == null) {
            LOG.info("{} is null, using default ({})", variableName, defaultDurationString);
            return parseConfigString(defaultDurationString, defaultDurationString, variableName);
        }

        Pattern regex = Pattern.compile("\\s*(\\d+)\\s*(\\w+)\\s*");
        Matcher matcher = regex.matcher(durationString);

        if (!matcher.matches()) {
            // If we can't parse it, then just set it to the default
            LOG.error("Could not parse {} '{}', setting to default ({})", variableName, durationString,
                    defaultDurationString);
            durationString = defaultDurationString;
            matcher = regex.matcher(defaultDurationString);
            matcher.matches();
        }

        int quantity = Integer.parseInt(matcher.group(1));
        String unit = matcher.group(2).toLowerCase();

        if (unit.startsWith("y")) {
            return new ParsedDuration(Period.ofYears(quantity), durationString);
        } else if (unit.startsWith("mo")) {
            return new ParsedDuration(Period.ofMonths(quantity), durationString);
        } else if (unit.startsWith("w")) {
            return new ParsedDuration(Period.ofWeeks(quantity), durationString);
        } else if (unit.startsWith("d")) {
            return new ParsedDuration(Period.ofDays(quantity), durationString);
        } else if (unit.startsWith("h")) {
            return new ParsedDuration(Duration.ofHours(quantity), durationString);
        } else if (unit.startsWith("min")) {
            return new ParsedDuration(Duration.ofMinutes(quantity), durationString);
        } else if (unit.startsWith("s")) {
            return new ParsedDuration(Duration.ofSeconds(quantity), durationString);
        } else {
            // If we can't parse it, then just set it to the default
            LOG.error("Unrecognized time unit on {} '{}', setting to default ({})", variableName, durationString,
                    defaultDurationString);

            return parseConfigString(defaultDurationString, defaultDurationString, variableName);
        }
    }

    @Data
    @AllArgsConstructor
    public static class ParsedDuration {
        TemporalAmount amount;
        String usedDurationString;
        public Duration asDuration() {
            return Duration.from(this.amount);
        }
    }
}
