package com.seeq.link.sdk.utilities;

import java.util.concurrent.Semaphore;
import java.util.concurrent.atomic.AtomicInteger;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.seeq.link.sdk.interfaces.ConcurrentRequestsHandler;
import com.seeq.utilities.ManualResetEvent;
import com.seeq.utilities.exception.OperationCanceledException;

import lombok.extern.slf4j.Slf4j;

/**
 * There are two main use-cases for concurrent requests
 * 1. Limit the number of concurrent requests per connection
 * In this case each connection will have its own instance of this class.
 *
 * 2. Limit the number of concurrent requests per connector
 * In this case the connector will create an instance of this class and share it between its connections. This is
 * useful for connectors like Python ExtCalc where each script corresponds to a connection, and therefore, for
 * optimal CPU usage we should have the possibility to set maxConcurrentRequests per connector. Basically in this
 * case we limit the number of parallel Python processes.
 */
@Slf4j
public class DefaultConcurrentRequestsHandler implements ConcurrentRequestsHandler {

    private final AtomicInteger currentConcurrentRequests = new AtomicInteger(0);
    private final int maxConcurrentRequests;
    private final Semaphore concurrentRequestsSemaphore;

    public DefaultConcurrentRequestsHandler(int maxConcurrentRequests) {
        Preconditions.checkArgument(maxConcurrentRequests > 0, "Max concurrent requests should be at least 1");
        this.maxConcurrentRequests = maxConcurrentRequests;
        this.concurrentRequestsSemaphore = new Semaphore(this.maxConcurrentRequests);
    }

    @Override
    public Thread runWhenPermitted(Runnable request, ThreadCollection threadCollection, long timeoutMillis,
            long requestId, ManualResetEvent requestThreadStartedEvent) {
        int registeredRequests = this.registerRequest();
        int maxConcurrentRequests = this.getMaxConcurrentRequests();
        int overflowRequests = registeredRequests - maxConcurrentRequests;

        return threadCollection.spawn(() -> {
            // keep this line inside request thread for easier request tracking in the logs
            LOG.debug(
                    "Received concurrent request with Id {}, {} work thread. {} work thread(s) running. {} work " +
                            "thread(s) queued. Request timeout {}ms",
                    requestId,
                    overflowRequests > 0 ? "queuing" : "spawning",
                    Math.min(registeredRequests, maxConcurrentRequests),
                    Math.max(0, overflowRequests),
                    timeoutMillis);

            // Do not move acquireProcessingPermit in the next try block.
            // releaseProcessingPermit must not be called if acquireProcessingPermit was not successful
            try {
                LOG.debug("Acquiring execution permit for request with Id {}", requestId);
                DefaultConcurrentRequestsHandler.this.acquireProcessingPermit(requestThreadStartedEvent);
            } catch (InterruptedException e) {
                throw new OperationCanceledException();
            }

            try {
                LOG.debug("Running request with Id {}", requestId);
                request.run();
            } finally {
                LOG.debug("Releasing execution permit for request with Id {}", requestId);
                DefaultConcurrentRequestsHandler.this.releaseProcessingPermit();
            }
        }, timeoutMillis, requestId);
    }

    @Override
    public int getMaxConcurrentRequests() { return this.maxConcurrentRequests; }

    @Override
    public int getRegisteredRequestsCount() { return this.currentConcurrentRequests.get(); }

    /**
     * Register a new request. Non-blocking method which register the request and immediately returns
     *
     * @return the number of registered requests. May be lower than the number of requests running in parallel
     */
    @VisibleForTesting
    int registerRequest() {
        return this.currentConcurrentRequests.incrementAndGet();
    }

    /**
     * Allocate a processing permit. This method is blocking the execution until a permit can be achieved
     *
     * @param beforeAcquireEvent
     *         An event which is set right before trying to acquire the semaphore. This is used internally by
     *         {@code runWhenPermitted}
     * @throws InterruptedException
     *         if the current thread is interrupted
     */
    @VisibleForTesting
    void acquireProcessingPermit(ManualResetEvent beforeAcquireEvent) throws InterruptedException {
        try {
            beforeAcquireEvent.set();
            this.concurrentRequestsSemaphore.acquire();
        } catch (InterruptedException e) {
            // intentionally not releasing a permit because none was acquired
            this.currentConcurrentRequests.decrementAndGet();
            throw e;
        }
    }

    /**
     * Release a processing permit. Non-blocking method which releases the permit and immediately returns
     *
     * @return the number of remaining registered requests
     */
    @VisibleForTesting
    int releaseProcessingPermit() {
        this.concurrentRequestsSemaphore.release();
        return this.currentConcurrentRequests.decrementAndGet();
    }
}