package com.seeq.link.sdk.utilities;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;

import com.seeq.link.sdk.interfaces.Connector;
import com.seeq.link.sdk.interfaces.ConnectorServiceV2;
import com.seeq.utilities.SeeqNames;

/**
 * Connector connection utilities.
 */
public class ConnectorHelper {
    private final Map<String, String> uniqueIdMap = new HashMap<>();
    private final Set<String> uniqueNameSet = new HashSet<>();

    /**
     * Check to make sure the ID for each connection is unique within that connector.
     *
     * @param connectionId
     *         The ID of the connection be tested for uniqueness.
     * @param connectionName
     *         The name of the connection be tested for uniqueness.
     * @param connectorName
     *         The host connector.
     * @param LOG
     *         Error logging mechanism.
     * @return true if the ID is unique.
     */
    public boolean isIdUnique(String connectionId, String connectionName, String connectorName, Logger LOG) {
        if (this.uniqueIdMap.containsKey(connectionId)) {
            LOG.error("'{}' has the same ID {} as '{}'. " +
                            "To connect, edit the configuration file '{}.json' to give '{}' a unique ID.",
                    connectionName, connectionId, this.uniqueIdMap.get(connectionId), connectorName, connectionName);
            return false;
        } else if (this.uniqueNameSet.contains(connectionName)) {
            LOG.error("'{}' has the same name as another connection of the same type. " +
                            "To connect, edit the configuration file '{}.json' to give it a unique name.",
                    connectionName, connectorName);
            return false;
        } else {
            this.uniqueIdMap.put(connectionId, connectionName);
            this.uniqueNameSet.add(connectionName);
            return true;
        }
    }

    public static String generateConnectorDeveloperInfo(Connector connector) {
        if (connector instanceof ConnectorServiceV2) {
            ConnectorServiceV2 connectorService = (ConnectorServiceV2) connector;
            String developer = connectorService.getConnectorDeveloperName();
            if (!StringUtils.isBlank(developer)) {
                String supportUrl = connectorService.getConnectorDeveloperSupportUrl();
                return developer + (!StringUtils.isBlank(supportUrl) ? " (" + supportUrl + ")" : "");
            }
        }
        return SeeqNames.Connectors.UnknownThirdPartyDeveloperName;
    }
}
