package com.seeq.link.sdk.utilities;


import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.seeq.link.messages.connector.condition.ConditionConnectionMessages.ConditionResponseMessage.CapsuleData;

import lombok.Data;

@Data
public class Capsule {
    public final TimeInstant start;
    public final TimeInstant end;
    public final List<Property> properties;

    public Capsule(TimeInstant start, TimeInstant end, List<Property> properties) {
        Preconditions.checkArgument(start.getTimestamp() <= end.getTimestamp(),
                "Start must be less than or equal to end");

        // This is intentionally done without checkArgument to avoid a performance penalty
        if (properties.stream().map(Property::getName).distinct().count() != properties.size()) {
            String propertiesList;
            try {
                ObjectMapper mapper = new ObjectMapper();
                mapper.disable(SerializationFeature.INDENT_OUTPUT);
                propertiesList = mapper.writeValueAsString(properties);
            } catch (Exception e) {
                propertiesList = "<Could not stringify properties list>";
            }
            String errorMessage = String.format(
                    "Properties should only contain one of each name, which is not the case for the capsule: "
                            + "{\"start\":\"%s\",\"end\":\"%s\",\"properties\":%s}",
                    start, end, propertiesList);
            throw new IllegalArgumentException(errorMessage);
        }

        this.start = start;
        this.end = end;

        // Silently drop invalid properties. We don't want one bad property to red triangle an entire condition.
        this.properties = properties.stream()
                .filter(p -> p.getName() != null)
                .filter(p -> p.getValue() != null)
                .filter(p -> !p.getValue().trim().isEmpty())
                .filter(p -> !p.getName().trim().isEmpty())
                .collect(ImmutableList.toImmutableList());
    }

    public CapsuleData toMessage() {
        return CapsuleData.newBuilder()
                .setStart(this.start.getTimestamp())
                .setEnd(this.end.getTimestamp())
                .addAllProperty(this.properties.stream().map(Property::toMessage).collect(
                        Collectors.toList()))
                .build();
    }

    @Data
    public static class Property {
        /**
         * The name of the capsule property.
         */
        final String name;
        /**
         * The value of the capsule property. Will be parsed as a timestamp, numeric or boolean value unless the unit
         * is 'string'.
         */
        final String value;
        /**
         * The unit of the capsule property. Should be 'string' if the property is a string, or leave null for unitless.
         */
        final String unit;

        public Property(String name, String value, String unit) {
            this.name = name;
            this.value = value;
            this.unit = Optional.ofNullable(unit).orElse(StringUtils.EMPTY);
        }

        CapsuleData.Property toMessage() {
            return CapsuleData.Property.newBuilder()
                    .setName(this.name).setValue(this.value).setUnits(this.unit).build();
        }
    }
}
