package com.seeq.link.sdk.utilities;

import java.time.Duration;
import java.util.concurrent.TimeUnit;

/**
 * Monitors the time it takes to process a batch (via the Start() and Stop() methods) and recommends a new batch size
 * based on preferences for MaximumDuration.
 */
public class BatchSizeHelper {
    private final Stopwatch stopwatch;
    private int currentSize;
    private int lastSize;
    private Duration lastDuration = Duration.ZERO;
    private final Duration maximumDuration;

    /**
     * Creates a new BatchSizeHelper with specified constraints.
     *
     * @param initialSize
     *         The initial value for the recommended batch size.
     * @param maximumDuration
     *         The maximum duration for processing that the caller wants a single batch to take.
     */
    public BatchSizeHelper(int initialSize, Duration maximumDuration) {
        this(initialSize, maximumDuration, new Stopwatch());
    }

    /**
     * Creates a new BatchSizeHelper with specified constraints. This constructor is only used by tests.
     *
     * @param initialSize
     *         The initial value for the recommended batch size.
     * @param maximumDuration
     *         The maximum duration for processing that the caller wants a single batch to take.
     * @param stopwatch
     *         The stopwatch class to be used to measure time elapsed. Used for testing.
     */
    public BatchSizeHelper(int initialSize, Duration maximumDuration, Stopwatch stopwatch) {
        this.currentSize = initialSize;
        this.maximumDuration = maximumDuration;
        this.stopwatch = stopwatch;
    }

    /**
     * Called to signal that a batch has started processing.
     */
    public void start() {
        this.stopwatch.restart();
    }

    /**
     * Called to signal that a batch has stopped processing. A new BatchSize will be calculated during this call. This
     * overload should be called if the actual number of items returned in a batch never varies until the final call.
     */
    public void stop() {
        this.stop(this.currentSize);
    }

    /**
     * Called to signal that a batch has stopped processing. A new BatchSize will be calculated during this call.
     *
     * @param actualSize
     *         The number of items in the batch, which may differ from the recommended size
     */
    public void stop(int actualSize) {
        this.stopwatch.stop();

        Duration headroom = this.maximumDuration
                .minus(Duration.ofMillis(this.stopwatch.elapsed(TimeUnit.MILLISECONDS)));
        this.lastSize = actualSize;
        this.lastDuration = Duration.ofMillis(this.stopwatch.elapsed(TimeUnit.MILLISECONDS));

        double itemsPerMillisecond = ((double) actualSize) / this.stopwatch.elapsed(TimeUnit.MILLISECONDS);
        double newSize = this.currentSize + (headroom.toMillis() * itemsPerMillisecond);

        final double MaximumGrowthFactor = 5;
        final double MaximumShrinkFactor = 0.2;
        if (newSize > this.lastSize * MaximumGrowthFactor) {
            newSize = this.lastSize * MaximumGrowthFactor;
        } else if (newSize < this.lastSize * MaximumShrinkFactor) {
            newSize = this.lastSize * MaximumShrinkFactor;
        }

        this.stopwatch.reset();

        this.currentSize = Math.max(1, (int) newSize);
    }

    /**
     * The current recommended batch size given the history of time taken to process a batch.
     *
     * @return current recommended batch size
     */
    public int getBatchSize() {
        return this.currentSize;
    }

    /**
     * The amount of time the last batch took to process.
     *
     * @return time the last batch took to process
     */
    public Duration getLastDuration() {
        return this.lastDuration;
    }

    /**
     * The rate of items processed in the last batch.
     *
     * @return rate of items processed in the last batch
     */
    public double getLastItemsPerSecond() {
        return this.lastSize / (this.lastDuration.toMillis() / 1000d);
    }
}
