package com.seeq.link.sdk.utilities;

import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.AbstractIterator;
import com.google.common.collect.Iterators;
import com.google.common.collect.Streams;
import com.seeq.api.DatasourcesApi;
import com.seeq.link.sdk.interfaces.DatasourceConnectionServiceV2;
import com.seeq.model.DatasourceOutputListV1;
import com.seeq.model.DatasourceOutputV1;
import com.seeq.model.ScalarPropertyV1;
import com.seeq.utilities.SeeqNames;

/**
 * Auth connection utilities.
 */
public class AuthConnectionHelper {

    /**
     * Retrieves the GroupsToSync information for a given datasource
     * Only non archived datasources are considered.
     *
     * @param datasourceClass
     *         the datasource class of the datasource
     * @param datasourceId
     *         the datasourceId of the datasource
     * @param connectionService
     *         the connectionService to use to obtain the datasources from Seeq
     * @return set of groups to be synchronized
     */
    public static Set<String> getGroupsToSyncForDatasource(String datasourceClass, String datasourceId,
            DatasourceConnectionServiceV2 connectionService) {
        DatasourcesApi datasourcesApi =
                connectionService.getAgentService().getApiProvider().createDatasourcesApi();

        Iterator<Iterator<DatasourceOutputV1>> allDatasourcesIterators = new AbstractIterator
                // Note: '<>' with anonymous inner classes is not supported in -source 8
                <Iterator<DatasourceOutputV1>>() {
            private static final int CHUNK_SIZE = 100;
            private int offset = 0;
            private boolean reachedTheEnd = false;

            @Override
            protected Iterator<DatasourceOutputV1> computeNext() {
                if (this.reachedTheEnd) {
                    return this.endOfData();
                }

                DatasourceOutputListV1 datasources =
                        datasourcesApi.getDatasources(null, null, this.offset, CHUNK_SIZE, false);

                this.offset += CHUNK_SIZE;

                if (datasources.getNext() == null) {
                    this.reachedTheEnd = true;
                }

                return datasources.getDatasources().iterator();
            }
        };

        ScalarPropertyV1 referencedDatasourceClass = new ScalarPropertyV1()
                .name(SeeqNames.Properties.ReferencedDatasourceClass)
                .value(datasourceClass)
                .unitOfMeasure("string");

        ScalarPropertyV1 referencedDatasourceId = new ScalarPropertyV1()
                .name(SeeqNames.Properties.ReferencedDatasourceId)
                .value(datasourceId)
                .unitOfMeasure("string");

        return Streams.stream(Iterators.concat(allDatasourcesIterators))
                .filter(d -> {
                    List<ScalarPropertyV1> additionalProperties = d.getAdditionalProperties();
                    return additionalProperties.contains(referencedDatasourceClass) &&
                            additionalProperties.contains(referencedDatasourceId);
                })
                .flatMap(d -> d.getAdditionalProperties().stream())
                .filter(p -> p.getName().equals(SeeqNames.Properties.AdditionalGroupsToSync))
                .map(ScalarPropertyV1::getValue)
                .filter(v -> v instanceof String)
                .map(v -> (String) v)
                .map(g2syncStr -> Arrays.asList(g2syncStr.split("\\s*,\\s*")))
                .flatMap(Collection::stream)
                .collect(Collectors.toSet());
    }

}
