package com.seeq.link.sdk.services;

import java.util.ArrayList;
import java.util.List;

import com.seeq.link.sdk.utilities.Capsule;
import com.seeq.link.sdk.utilities.TimeInstant;
import com.seeq.utilities.SeeqNames;

/**
 * Transformer for Capsules.
 * To ensure consistency with the datasource the Capsule's properties cannot be mutated, but new properties can
 * be created using any Capsule property as input.
 */
public class TransformableCapsule implements Transformable<Capsule> {

    // We provide the Condition Data ID as a property to allow transforming Capsules of a specific Condition
    private static final String CONDITION_DATA_ID_PROPERTY = "Condition Metadata: Data ID";

    private final TimeInstant start;
    private final TimeInstant end;
    private final List<Capsule.Property> properties;
    private final String conditionDataId;

    public TransformableCapsule(Capsule input, String conditionDataId) {
        this(input.start, input.end, input.properties, conditionDataId);
    }

    public TransformableCapsule(TimeInstant start, TimeInstant end, List<Capsule.Property> properties,
            String conditionDataId) {
        this.start = start;
        this.end = end;
        // Capsule.properties is an immutable list, so we make a mutable copy to store new properties
        // and create a new Capsule when getInputObject() is called. We only make a shallow copy
        // because the properties themselves are immutable too.
        this.properties = properties == null ? new ArrayList<>() : new ArrayList<>(properties);
        this.conditionDataId = conditionDataId;
    }

    @Override
    public Transformable<Capsule> copy() {
        return new TransformableCapsule(this.start, this.end, this.properties, this.conditionDataId);
    }

    @Override
    public Capsule getInputObject() {
        return new Capsule(this.start, this.end, this.properties);
    }

    @Override
    public Object getProperty(String propertyName) {
        switch (propertyName) {
        case SeeqNames.Properties.Type:
            return SeeqNames.Types.Capsule;
        case SeeqNames.CapsuleProperties.Start:
            return this.start;
        case SeeqNames.CapsuleProperties.End:
            return this.end;
        case CONDITION_DATA_ID_PROPERTY:
            return this.conditionDataId;
        }
        Capsule.Property property = this.findProperty(propertyName);
        return property != null ? property.getValue() : null;
    }

    @Override
    public void setProperty(String propertyName, Object propertyValue) {
        this.setProperty(propertyName, propertyValue, null);
    }

    @Override
    public void setProperty(String propertyName, Object propertyValue, String unitOfMeasure) {
        if (propertyName.equals(SeeqNames.Properties.Type)
                || propertyName.equals(SeeqNames.CapsuleProperties.Start)
                || propertyName.equals(SeeqNames.CapsuleProperties.End)
                || propertyName.equals(CONDITION_DATA_ID_PROPERTY)
                || this.findProperty(propertyName) != null) {
            throw new IllegalStateException(String.format("Property transform tried to change the Capsule " +
                    "property '%s', which is not allowed. Capsule property transforms are only allowed to create new " +
                    "properties.", propertyName));
        }
        this.properties.add(new Capsule.Property(propertyName, PropertyTransformer.toStr(propertyValue),
                unitOfMeasure));
    }

    @Override
    public String toString() {
        return String.format("Capsule(start = %s, end = %s)", this.start, this.end);
    }

    private Capsule.Property findProperty(String propertyName) {
        return this.properties.stream()
                .filter(p -> propertyName.equals(p.getName()))
                .findFirst()
                .orElse(null);
    }
}
