package com.seeq.link.sdk.services;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.code.regexp.Matcher;
import com.google.code.regexp.Pattern;
import com.seeq.link.sdk.utilities.Capsule;
import com.seeq.model.PutAssetInputV1;
import com.seeq.model.ConditionUpdateInputV1;
import com.seeq.model.PutScalarInputV1;
import com.seeq.model.SignalWithIdInputV1;
import com.seeq.model.UserGroupWithIdInputV1;
import com.seeq.utilities.SeeqNames;

import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.experimental.UtilityClass;
import lombok.extern.slf4j.Slf4j;

@Slf4j
@UtilityClass
public class PropertyTransformer {
    @Data
    @NoArgsConstructor
    @AllArgsConstructor
    public static class Input {
        private String property;
        private Object value;
    }

    @Data
    @NoArgsConstructor
    @AllArgsConstructor
    public static class Output {
        public Output(String property, Object value) {
            this(property, value, null);
        }

        private String property;
        private Object value;
        private String unitOfMeasure;
    }

    @Data
    public static class Spec {
        private List<Input> inputs = new ArrayList<>();
        private List<Output> outputs = new ArrayList<>();
        private boolean enabled = true;
        private boolean log;
    }

    public static PutScalarInputV1 transform(PutScalarInputV1 input, List<Spec> transforms) {
        return transform(new TransformableScalar(input), transforms).getInputObject();
    }

    public static SignalWithIdInputV1 transform(SignalWithIdInputV1 input, List<Spec> transforms) {
        return transform(new TransformableSignal(input), transforms).getInputObject();
    }

    public static UserGroupWithIdInputV1 transform(UserGroupWithIdInputV1 input, List<Spec> transforms) {
        return transform(new TransformableUserGroup(input), transforms).getInputObject();
    }

    public static ConditionUpdateInputV1 transform(ConditionUpdateInputV1 input, List<Spec> transforms) {
        return transform(new TransformableCondition(input), transforms).getInputObject();
    }

    public static PutAssetInputV1 transform(PutAssetInputV1 input, List<Spec> transforms) {
        return transform(new TransformableAsset(input), transforms).getInputObject();
    }

    public static Capsule transform(Capsule input, String conditionDataId, List<Spec> transforms) {
        return transform(new TransformableCapsule(input, conditionDataId), transforms).getInputObject();
    }

    public static <T> Transformable<T> transform(Transformable<T> input, List<Spec> transforms) {
        Transformable<T> output = input.copy();

        for (Spec transform : transforms) {
            Map<String, String> variables = new HashMap<>();
            boolean isMatch = true;

            for (Input transformInput : transform.getInputs()) {
                Object inputValueObject = transformInput.getValue();

                Object propertyValueObject = output.getProperty(transformInput.getProperty());
                if (inputValueObject instanceof String) {
                    String propertyValueString = toStr(propertyValueObject);
                    if (propertyValueString == null) {
                        isMatch = false;
                        break;
                    }

                    String inputValueString = (String) inputValueObject;
                    Pattern regex = Pattern.compile(inputValueString);
                    Matcher match = regex.matcher(propertyValueString);
                    if (!match.matches()) {
                        isMatch = false;
                        break;
                    }

                    for (String groupName : regex.groupNames()) {
                        if (groupName.equals("0")) {
                            // This is the "whole String" capture, which we skip
                            continue;
                        }

                        String group = match.group(groupName);
                        variables.put(groupName, group);
                    }
                } else {
                    isMatch = Objects.equals(inputValueObject, propertyValueObject);
                }
            }

            if (!isMatch) {
                continue;
            }

            for (Output transformOutput : transform.getOutputs()) {
                Object outputValueObject = transformOutput.getValue();
                Object newPropertyObject = outputValueObject;
                String newPropertyUnits = transformOutput.getUnitOfMeasure();
                if (outputValueObject instanceof String) {
                    String newPropertyString = (String) outputValueObject;
                    for (Map.Entry<String, String> variable : variables.entrySet()) {
                        newPropertyString =
                                newPropertyString.replace("${" + variable.getKey() + "}", variable.getValue());

                        if (newPropertyUnits != null) {
                            newPropertyUnits =
                                    newPropertyUnits.replace("${" + variable.getKey() + "}", variable.getValue());
                        }
                    }

                    if (transform.isLog()) {
                        if (Pattern.compile("\\$\\{[^\\}]+\\}").matcher(newPropertyString).matches()) {
                            LOG.error("Variable(s) in output were not found in capture groups: {}", newPropertyString);
                        }

                        LOG.debug("Transforming: '{}' Property: '{}' Transform: '{}' --> '{}{}' {}",
                                input, transformOutput.getProperty(),
                                output.getProperty(transformOutput.getProperty()),
                                newPropertyString,
                                newPropertyUnits == null ? "" : newPropertyUnits,
                                transform.isEnabled() ? "" : "<<DISABLED>>");
                    }

                    newPropertyObject = newPropertyString;
                }

                // String properties can only have the unit 'string'. So if we have units, try to parse as numeric.
                if (newPropertyUnits != null && !newPropertyUnits.equals("string") &&
                        newPropertyObject instanceof String) {
                    try {
                        newPropertyObject = Long.parseLong((String) newPropertyObject);
                    } catch (NumberFormatException e) {
                        try {
                            newPropertyObject = Double.parseDouble((String) newPropertyObject);
                        } catch (NumberFormatException e2) {
                            if (transform.isLog()) {
                                LOG.warn("Unable to parse {} as numeric; units {} will be ignored",
                                        newPropertyObject, newPropertyUnits);
                            }
                            newPropertyUnits = null;
                        }
                    }
                }

                if (transform.isEnabled()) {
                    output.setProperty(transformOutput.getProperty(), newPropertyObject, newPropertyUnits);
                }
            }
        }

        return output;
    }

    public static String toStr(Object s) {
        return (s == null) ? null : s.toString();
    }

    /**
     * @return the transforms from 'transforms' that have an Input with property == 'Type' and value == type or
     *         null if 'transforms' is null
     */
    public static List<PropertyTransformer.Spec> findTransformsWithInputType(String type,
            @Nullable List<PropertyTransformer.Spec> transforms) {
        if (transforms == null) {
            return null;
        }
        Predicate<Spec> hasRequestedInputType = transform -> transform.getInputs().stream().anyMatch(
                input -> SeeqNames.Properties.Type.equals(input.getProperty()) && type.equals(input.getValue()));
        return transforms.stream().filter(hasRequestedInputType).collect(Collectors.toList());
    }
}
