package com.seeq.link.sdk.services;

import java.io.IOException;
import java.util.Objects;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.cfg.MapperConfig;
import com.fasterxml.jackson.databind.exc.MismatchedInputException;
import com.fasterxml.jackson.databind.introspect.AnnotatedField;
import com.fasterxml.jackson.databind.introspect.AnnotatedMethod;
import com.seeq.link.sdk.ConfigObject;
import com.seeq.link.sdk.ConfigObjectWrapper;

public class JsonConfigObjectMapper {
    private final ObjectMapper objectMapper = new ObjectMapper()
            .setPropertyNamingStrategy(new PascalCaseNamingStrategy())
            .configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);


    /**
     * Converts a json into a {@link ConfigObject} and wraps it into a {@link ConfigObjectWrapper}. The wrapper
     * provides additional information about the configuration (like last modified date).
     * This function requires an array of config objects to use as a means of discovering what Object type is encoded
     * in the store.
     *
     * @param jsonString
     *         The json to convert
     * @param defaultConfigObjects
     *         An array of ConfigObject instances that represent the possible Object types that can be successfully
     *         retrieved from the store. If the persisted Object type does not match anything in the list, then the
     *         first item in the array is passed back (being effectively a means to have a 'default' config).
     * @param parseErrorMessage
     *         Error message logged if json parsing fails.
     * @param lastModified
     *         A nullable Long that will be set in the resulting {@link ConfigObjectWrapper}
     * @return The resulting configuration wrapper
     * @throws IOException
     *         If the received string is not having a valid json format.
     */
    public static ConfigObjectWrapper toConfigObjectWrapper(String jsonString, ConfigObject[] defaultConfigObjects,
            String parseErrorMessage, Long lastModified) throws IOException {
        ObjectMapper objectMapper = new ObjectMapper()
                .setPropertyNamingStrategy(new PascalCaseNamingStrategy())
                .configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        ConfigObject configObject = null;
        if (jsonString != null) {
            for (ConfigObject defaultConfigObject : defaultConfigObjects) {
                try {
                    configObject = objectMapper.readValue(jsonString, defaultConfigObject.getClass());
                } catch (JsonProcessingException e) {
                    String msg = parseErrorMessage;
                    if (e.getLocation() != null) {
                        msg += String.format(" at line %d, column %d. ",
                                e.getLocation().getLineNr(), e.getLocation().getColumnNr());
                    }

                    if (e.getOriginalMessage() != null) {
                        msg += "Details: " + e.getOriginalMessage() + ". ";
                    }

                    if (e instanceof MismatchedInputException) {
                        MismatchedInputException mismatchedException = (MismatchedInputException) e;
                        final String path =
                                mismatchedException.getPath().stream()
                                        .map(JsonMappingException.Reference::getFieldName)
                                        .filter(Objects::nonNull)
                                        .collect(Collectors.joining("/"));
                        msg += "Path: " + path;
                    }
                    throw new IOException(msg);
                }

                if (configObject.getVersion().equals(defaultConfigObject.getVersion())) {
                    break;
                }

                configObject = null;
            }
        }

        if (configObject == null) {
            configObject = defaultConfigObjects[0];
        }

        return new ConfigObjectWrapper(configObject, lastModified);
    }

    /**
     * Converts a configObject into a json
     *
     * @param configObject
     *         The configuration object to be converted to json
     * @return The configuration object as json string
     * @throws JsonProcessingException
     */
    public static String toJson(Object configObject) throws JsonProcessingException {
        ObjectMapper objectMapper = new ObjectMapper();
        objectMapper.setPropertyNamingStrategy(new PascalCaseNamingStrategy());
        objectMapper.enable(SerializationFeature.INDENT_OUTPUT);
        objectMapper.disable(SerializationFeature.FAIL_ON_EMPTY_BEANS);
        String newJson = objectMapper.writeValueAsString(configObject);

        if (!newJson.endsWith(System.getProperty("line.separator"))) {
            // Add a new line, because it will get added on write anyways.
            // This makes the comparison work properly.
            newJson += System.getProperty("line.separator");
        }
        return newJson;
    }

    /**
     * We have our own PascalCase naming strategy so that it matches .NET Link
     */
    private static class PascalCaseNamingStrategy extends PropertyNamingStrategy {
        private static final long serialVersionUID = -522841738265888214L;

        @Override
        public String nameForField(MapperConfig<?> config, AnnotatedField field, String defaultName) {
            return this.convert(defaultName);

        }

        @Override
        public String nameForGetterMethod(MapperConfig<?> config, AnnotatedMethod method, String defaultName) {
            return this.convert(defaultName);
        }

        @Override
        public String nameForSetterMethod(MapperConfig<?> config, AnnotatedMethod method, String defaultName) {
            return this.convert(defaultName);
        }

        public String convert(String defaultName) {
            char[] arr = defaultName.toCharArray();
            if (arr.length != 0) {
                if (Character.isLowerCase(arr[0])) {
                    char upper = Character.toUpperCase(arr[0]);
                    arr[0] = upper;
                }
            }
            return String.valueOf(arr);
        }
    }
}
