package com.seeq.link.sdk.services;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Duration;
import java.util.HashMap;
import java.util.function.Consumer;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;

import com.seeq.link.sdk.ConfigObject;
import com.seeq.link.sdk.ConfigObjectWrapper;
import com.seeq.link.sdk.interfaces.FileConfigObjectProvider;
import com.seeq.utilities.FileChangeListener;
import com.seeq.utilities.FileWatcher;

import lombok.extern.slf4j.Slf4j;

/**
 * See {@link FileConfigObjectProvider} for documentation.
 */
@Slf4j
public class DefaultFileConfigObjectProvider implements FileConfigObjectProvider, FileChangeListener {
    private Duration changeDebouncePeriod = Duration.ofSeconds(1);

    public DefaultFileConfigObjectProvider() {}

    public DefaultFileConfigObjectProvider(Duration changeDebouncePeriod) {
        this.changeDebouncePeriod = changeDebouncePeriod;
    }

    private Path dataPath;

    private Path getFullPath(String name) {
        return this.dataPath.resolve(name + ".json");
    }

    @Override
    public void initialize(Path dataPath) {
        this.dataPath = dataPath;
    }

    @Override
    public ConfigObjectWrapper loadConfigObject(String name, ConfigObject[] defaultConfigObjects) throws IOException {
        Path filename = null;
        String json = null;
        Long fileLastModified = null;
        try {
            filename = this.getFullPath(name);
            File file = filename.toFile();
            if (file.exists()) {
                json = FileUtils.readFileToString(file, "UTF-8");

                // milliseconds since the epoch (00:00:00 GMT, January 1, 1970)
                fileLastModified = file.lastModified();
            }
        } catch (Exception e) {
            if (filename == null) {
                LOG.error("Could not resolve config Object name \"{}\" to file path:\n{}", name, e);
            } else {
                throw new IOException(String.format("Could not read json file \"%s\":\n%s", filename, e));
            }
        }

        return JsonConfigObjectMapper.toConfigObjectWrapper(json, defaultConfigObjects,
                String.format("Error parsing JSON file \"%s\"", filename), fileLastModified);
    }

    @Override
    public void saveConfigObject(String name, Object configObject) {
        Path filename = null;

        try {
            filename = this.getFullPath(name);
            if (!Files.exists(this.dataPath)) {
                Files.createDirectories(this.dataPath);
            }

            String newJson = JsonConfigObjectMapper.toJson(configObject);

            String existingJson = "";
            if (Files.exists(filename)) {
                existingJson = FileUtils.readFileToString(filename.toFile(), "UTF-8");
            }

            if (!existingJson.equals(newJson)) {
                // Disable watcher while we're writing
                if (this.watchers.containsKey(name)) {
                    this.watchers.get(name).stop();
                }

                try {
                    LOG.debug("Saving configuration file '{}'", filename);
                    FileUtils.writeStringToFile(filename.toFile(), newJson, "UTF-8");
                } finally {
                    // Ensure we always re-enable the watcher.
                    if (this.watchers.containsKey(name)) {
                        this.watchers.get(name).start();
                    }
                }
            }
        } catch (Exception e) {
            if (filename == null) {
                LOG.error("Could not resolve config Object name \"{}\" to file path:\n{}", name, e);
            } else {
                LOG.error("Could not write config file \"{}\":\n{}", filename, e);
            }
        }
    }

    private final HashMap<String, FileWatcher> watchers = new HashMap<>();
    private final HashMap<String, Consumer<String>> callbacks = new HashMap<>();

    @Override
    public void registerChangeCallback(String name, Consumer<String> callback) {
        Path fullPath = this.getFullPath(name);
        FileWatcher fileWatcher = new FileWatcher(fullPath.getParent(), fullPath.getFileName().toString(), this,
                this.changeDebouncePeriod);
        this.watchers.put(name, fileWatcher);
        this.callbacks.put(name, callback);
        try {
            fileWatcher.start();
        } catch (IOException e) {
            LOG.error("Error encountered while registering FileWatcher", e);
        }
    }

    @Override
    public void unregisterChangeCallback(String name) {
        if (this.watchers.containsKey(name)) {
            this.watchers.get(name).stop();
        }

        this.watchers.remove(name);
        this.callbacks.remove(name);
    }

    private void fireRegisterChangeCallback(Path filePath) {
        String name = FilenameUtils.getBaseName(filePath.getFileName().toString());
        LOG.debug(name + " has been changed");
        Consumer<String> callback = this.callbacks.get(name);
        if (callback != null) {
            callback.accept(name);
        }
    }

    @Override
    public void onFileModify(Path filePath) {
        this.fireRegisterChangeCallback(filePath);
    }

    @Override
    public void onFileDelete(Path filePath) {
        this.fireRegisterChangeCallback(filePath);
    }
}
