package com.seeq.link.sdk.interfaces;

import java.util.function.Consumer;

import com.seeq.link.messages.connector.signal.SignalConnectionMessages;
import com.seeq.link.sdk.utilities.TimeInstant;

public class GetSamplesParameters {
    private final String dataId;
    private final TimeInstant startTime;
    private final TimeInstant endTime;
    private final long maxInterpolation;
    private final int sampleLimit;
    private final String valueUom;
    private final Consumer<TimeInstant> cursorCallback;

    public GetSamplesParameters(SignalConnectionMessages.SignalRequestMessage signalRequestMessage,
            Consumer<TimeInstant> cursorCallback) {
        this(signalRequestMessage.getSignalId(),
                new TimeInstant(signalRequestMessage.getStartTime()),
                new TimeInstant(signalRequestMessage.getEndTime()),
                signalRequestMessage.getMaxInterpolation(),
                signalRequestMessage.getSampleLimit(),
                cursorCallback,
                signalRequestMessage.getValueUom()
        );
    }

    public GetSamplesParameters(String dataId, TimeInstant startTime, TimeInstant endTime, long maxInterpolation,
            int sampleLimit, Consumer<TimeInstant> cursorCallback, String valueUom) {
        this.dataId = dataId;
        this.startTime = startTime;
        this.endTime = endTime;
        this.sampleLimit = sampleLimit;
        this.maxInterpolation = maxInterpolation;
        this.valueUom = valueUom;
        this.cursorCallback = cursorCallback;
    }

    /**
     * A connector-defined string that identifies the signal to retrieve data for.
     *
     * @return connector-defined string that identifies the signal to retrieve data for.
     */
    public String getDataId() {
        return this.dataId;
    }

    /**
     * The start time for the requested data. Note: This function must return "boundary values" as well, which are the
     * nearest values immediately outside the requested interval. If your datasource does not return such "boundary
     * values" efficiently, use the {@link #getExpandedStartTime()} field for your datasource query.
     *
     * @return start time for the requested data
     */
    public TimeInstant getStartTime() {
        return this.startTime;
    }

    /**
     * The start time for the requested data minus the "maximum interpolation" for the signal. This value represents the
     * earliest possible time that a sample may be relevant for a given request. This is useful when the datasource does
     * not have a facility for efficiently returning a sample that is "on or before" the start time. You can just use
     * the ExpandedStartTime/ExpandedEndTime fields instead and return all the samples within the expanded range. Seeq
     * Server will ignore the values it doesn't need.
     *
     * @return expanded start time for the requested data
     */
    public TimeInstant getExpandedStartTime() {
        return new TimeInstant(this.startTime.getTimestamp() - this.maxInterpolation);
    }

    /**
     * The end time for the requested data. Note: This function must return "boundary values" as well, which are the
     * nearest values immediately outside the requested interval.
     *
     * @return end time for the requested data
     */
    public TimeInstant getEndTime() {
        return this.endTime;
    }

    /**
     * The end time for the requested data plus the "maximum interpolation" for the signal. This value represents the
     * latest possible time that a sample may be relevant for a given request. This is useful when the datasource does
     * not have a facility for efficiently returning a sample that is "on or after" the end time. You can just use the
     * ExpandedStartTime/ExpandedEndTime fields instead and return all the samples within the expanded range. Seeq
     * Server will ignore the values it doesn't need.
     *
     * @return expanded end time for the requested data
     */
    public TimeInstant getExpandedEndTime() {
        return new TimeInstant(this.endTime.getTimestamp() + this.maxInterpolation);
    }

    /**
     * @return the maximum interpolation for the requested signal in nanoseconds.
     */
    public long getMaxInterpolation() {
        return this.maxInterpolation;
    }

    /**
     * The maximum samples that can be returned as part of this request. Note that the connection need not enforce this
     * limit, as no more samples than the limit will be read from the sample stream that is returned from this method.
     * This limit is provided for informational purposes to enable optimization where possible.
     *
     * @return sample limit for this request
     */
    public int getSampleLimit() {
        return this.sampleLimit;
    }

    /**
     * If {@code true}, you can specify a time instant to {@link #setLastCertainKey(TimeInstant)} to indicate to Seeq
     * that any samples after a time instant are uncertain. If {@code false}, you cannot make such a method call (and
     * should not try to calculate a time) because Seeq will not use the value.
     */
    public boolean isLastCertainKeyRequested() {
        return this.cursorCallback != null;
    }

    public String getValueUom() {
        return this.valueUom;
    }

    /**
     * If {@link #isLastCertainKeyRequested()} returns {@code true}, you can invoke this method to indicate to Seeq that
     * any samples after a time instant are uncertain. There does not need to be a sample present at the key specified;
     * for instance, if a historian guarantees that all data is recorded within three hours, and the most recent sample
     * is two weeks ago, this value may be {@code now - three hours}. Any samples after this key will be treated as
     * uncertain by Seeq, and any samples before or at the key will be treated as certain.
     * <p>
     * If this method is not invoked, Seeq will assume that all samples returned are certain, and that no new samples
     * will change or appear at or before the latest sample key seen. This inference carries no performance penalty, so
     * for a datasource where no historical changes are expected, not invoking this method is a reasonable default.
     * <p>
     * If you supply null for the key, it will clear any key set by a previous call.
     * <p>
     * You can invoke this method anytime after {@link SignalPullDatasourceConnection#getSamples(GetSamplesParameters)}
     * is entered, and you may invoke it multiple times. If you do, the last invocation takes precedence.
     */
    public void setLastCertainKey(TimeInstant key) {
        if (this.cursorCallback == null) {
            throw new IllegalStateException("setLastCertainKey() can be called only if isLastCertainKeyRequested() is" +
                    " true");
        }

        this.cursorCallback.accept(key);
    }
}
