package com.seeq.link.sdk.interfaces;

import java.util.function.Consumer;

import com.seeq.link.messages.connector.condition.ConditionConnectionMessages;
import com.seeq.link.sdk.utilities.TimeInstant;

public class GetCapsulesParameters {
    private final String dataId;
    private final TimeInstant startTime;
    private final TimeInstant endTime;
    private final int capsuleLimit;
    private final long maxDuration;
    private final Consumer<TimeInstant> cursorCallback;

    public GetCapsulesParameters(ConditionConnectionMessages.ConditionRequestMessage conditionRequestMessage,
            Consumer<TimeInstant> cursorCallback) {
        this.dataId = conditionRequestMessage.getConditionId();
        this.startTime = new TimeInstant(conditionRequestMessage.getStartTime());
        this.endTime = new TimeInstant(conditionRequestMessage.getEndTime());
        this.capsuleLimit = conditionRequestMessage.getCapsuleLimit();
        this.maxDuration = conditionRequestMessage.getMaxDuration();
        this.cursorCallback = cursorCallback;
    }

    public GetCapsulesParameters(String dataId, TimeInstant startTime, TimeInstant endTime, int capsuleLimit,
            long maxDuration, Consumer<TimeInstant> cursorCallback) {
        this.dataId = dataId;
        this.startTime = startTime;
        this.endTime = endTime;
        this.capsuleLimit = capsuleLimit - 1; // Subtract one because it will be added back later
        this.maxDuration = maxDuration;
        this.cursorCallback = cursorCallback;
    }

    /**
     * A connector-defined String that identifies the condition to retrieve data for.
     *
     * @return connector-defined string that identifies the signal to retrieve data for.
     */
    public String getDataId() {
        return this.dataId;
    }

    /**
     * The start time for the requested data. Return capsules that start at or after this time and start at or before
     * {@link #getEndTime()}.
     * <br><br>
     * This is defined as the Seeq server query start minus the "maximum duration" for the condition, and represents
     * the earliest possible time that a capsule may be relevant. This is useful when it is most natural to query
     * capsules by start time; if querying by overlaps is more natural, use {@link #getOverlappingStartTime()} instead.
     *
     * @return start time for the requested capsules queried by their starts
     * @see #getOverlappingStartTime()
     */
    public TimeInstant getStartTime() {
        return this.startTime;
    }

    /**
     * The start time for the requested data, if queried in overlapping fashion. Return capsules that end at or after
     * this time, and start at or before {@link #getEndTime()}.
     *
     * In general querying by overlapping starts should only be used for conditions where capsules never overlap.
     * This is because capsules must still be returned completely in order; that is, if a capsule overlaps the query
     * interval, and another capsule starts after it but ends sooner and thus does not overlap the query interval,
     * BOTH must be included. This is unlikely to result in a more efficient query than {@link #getStartTime()} and
     * so should not be used unless capsules are strictly non-overlapping.
     *
     * @return start time for the requested capsules queried by their overlap
     * @see #getStartTime()
     */
    public TimeInstant getOverlappingStartTime() {
        return new TimeInstant(this.startTime.getTimestamp() + this.maxDuration);
    }

    /**
     * The end time for the requested data. Return capsules that start at or before this time, and start at or after
     * {@link #getStartTime()}.
     *
     * @return end time for the requested data
     */
    public TimeInstant getEndTime() {
        return this.endTime;
    }

    /**
     * The end time for the requested data plus the "maximum duration" for the condition. This value
     * represents the latest possible time that a capsule may be relevant for a given request.
     *
     * @return expanded end time for the requested data
     */
    public TimeInstant getExpandedEndTime() {
        return new TimeInstant(this.endTime.getTimestamp() + this.maxDuration);
    }

    /**
     * The maximum capsules that can be returned as part of this request. Note that the connection need not
     * enforce this limit, as no more capsules than the limit will be read from the stream that is
     * returned from this method. This limit is provided for informational purposes to enable optimization
     * where possible.
     *
     * @return capsule limit for this request
     */
    public int getCapsuleLimit() {
        // If the capsule limit isn't max value, increase it by 1 before sending it to the connection so we can
        // accurately assess if the connection has more data, but it got truncated by the capsule limit.
        return this.capsuleLimit == Integer.MAX_VALUE ?
                Integer.MAX_VALUE : this.capsuleLimit + 1;
    }

    /**
     * The maximum duration of capsules in the series in nanoseconds.
     *
     * @return maximum duration
     */
    public long getMaxDuration() {
        return this.maxDuration;
    }

    /**
     * If true, you can specify a time instant to SetLastCertainKey to indicate to Seeq that any conditions that start
     * after a time instant are uncertain. If false, you cannot make such a method call (and should not try to calculate
     * a time)
     * because Seeq will not use the value.
     */
    public boolean isLastCertainKeyRequested() {
        return this.cursorCallback != null;
    }

    /**
     * If {@link #isLastCertainKeyRequested()} returns true, you can (and should) invoke this method to indicate to
     * Seeq that any capsules that start after a time instant are uncertain. If {@link #isLastCertainKeyRequested()}
     * is true, this method must be invoked at least once or all capsules returned will appear uncertain in Seeq.
     *
     * If all capsules are certain and new capsules always start after existing capsules, a good value for the the last
     * certain key is the start of the most recent capsule in the condition. (If new capsules start at or after
     * existing capsules, then decrement the key (one nanosecond earlier). In general, new capsules should always start
     * after the last certain key.)
     *
     * If some capsules are uncertain, but new capsules come in by order of their starts, a good value for the last
     * certain key is the start of the earliest uncertain capsule, decremented (i.e. one nanosecond earlier).
     *
     * If new capsules are not added in order of their start, it is recommended to use a last certain key far enough
     * in the past that no new capsules will start before that key. This requires more knowledge of the capsules'
     * meaning and should be a last resort, as it will lead to the condition and derived data appearing less certain
     * in Seeq.
     *
     * You can invoke this method anytime after
     * {@link ConditionPullDatasourceConnection#getCapsules(GetCapsulesParameters)}
     * is entered, and you may invoke it multiple times. If you do, the last invocation takes precedence. If you supply
     * null for the key, it will clear any key set by a previous call.
     */
    public void setLastCertainKey(TimeInstant key) {
        if (this.cursorCallback == null) {
            throw new IllegalStateException("SetLastCertainKey cannot be called unless IsLastCertainKeyRequested is " +
                    "true");
        }

        this.cursorCallback.accept(key);
    }
}
