package com.seeq.link.sdk.interfaces;

/**
 * This interface defines the functions that must be implemented by the class that facilitates a connection to an
 * external system.
 *
 * Threading considerations:
 *
 * Initialize() and Destroy() are called by the connector host on its own thread.
 *
 * Connect(), Monitor() and Disconnect() are all called from a (different) single thread.
 *
 * Index() is called on its own thread. Index() calls are always on the same thread and therefore sequential.
 *
 * All other interface methods may be called from multiple threads simultaneously.
 */
public interface DatasourceConnectionV2 {

    /**
     * A unique identifier for the datasource. This is usually a GUID generated for the configuration file.
     *
     * @return the class of data for the external system
     */
    String getDatasourceClass();

    /**
     * The name of the datasource as it is intended to appear in Seeq.
     *
     * @return the name of the datasource as it is intended to appear in Seeq
     */
    String getDatasourceName();

    /**
     * The unique identifier of the datasource. This is typically a GUID, and must
     * not change once established.
     *
     * @return the unique identifier of the datasource
     */
    String getDatasourceId();

    /**
     * Initializes a connection, attempting to connect if enabled.
     *
     * @param connectionService
     *         Provides all required services for accomplishing tasks like indexing and responding to requests.
     */
    void initialize(DatasourceConnectionServiceV2 connectionService);

    /**
     * Destroys a connection, cleaning up any resources as necessary.
     */
    default void destroy() {}

    /**
     * Attempts a connection to the external system.
     */
    void connect();

    /**
     * Examines the state of the connection to determine if it has become disconnected.
     *
     * @return true if the connection is live, false if it has become disconnected. A return of false will cause the
     *         {@link #disconnect()} method to be invoked.
     */
    boolean monitor();

    /**
     * Disconnects from the external system.
     */
    void disconnect();

    /**
     * Called by an Agent when the agent has changed the connection's configuration (usually due to the indexing
     * schedule being updated).
     */
    void saveConfig();
}
