package com.seeq.link.sdk.interfaces;

import java.util.List;

import org.slf4j.Logger;

import com.seeq.ApiException;
import com.seeq.link.sdk.IndexingState;
import com.seeq.link.sdk.export.ExportConnectionConfigV1;
import com.seeq.link.sdk.export.ExportSamples;
import com.seeq.link.sdk.interfaces.Connection.ConnectionState;
import com.seeq.link.sdk.services.PropertyTransformer;
import com.seeq.model.AssetInputV1;
import com.seeq.model.AssetTreeSingleInputV1;
import com.seeq.model.ConditionUpdateInputV1;
import com.seeq.model.DatasourceOutputV1;
import com.seeq.model.ItemUpdateOutputV1;
import com.seeq.model.PutAssetInputV1;
import com.seeq.model.PutScalarInputV1;
import com.seeq.model.ScalarInputV1;
import com.seeq.model.ScalarPropertyV1;
import com.seeq.model.SignalWithIdInputV1;
import com.seeq.model.UserGroupWithIdInputV1;

import lombok.experimental.UtilityClass;

/**
 * Provides services to the class that manages a connection to an external system.
 */
public interface DatasourceConnectionServiceV2 {

    /**
     * The interpolation method to be used for a particular signal.
     */
    @UtilityClass
    class InterpolationMethod {

        /**
         * Linear-interpolated signals have a slope connecting each sample.
         */
        public static final String Linear = "linear";

        /**
         * Step-interpolated signals are constant from one sample until the next.
         */
        public static final String Step = "step";
    }

    /**
     * Retrieves the agent service that can be used to access the REST API directly, among other things.
     *
     * @return The agent service.
     */
    AgentService getAgentService();

    /**
     * Enables a connection, which causes a connect and monitor thread to be created.
     */
    void enable();

    /**
     * Disables a connection, which disconnects and shuts down the monitoring thread.
     */
    void disable();

    /**
     * Checks the Seeq License for the required feature, logs the result, and returns true if the license includes
     * the feature.
     *
     * @param featureName
     *         The feature for which the license is being checked
     * @return whether the License has been found to have the feature
     */
    boolean isLicensed(String featureName);

    /**
     * Provides the current state of the connection to the external datasource.
     *
     * @return the current state of the connection to the external datasource
     */
    ConnectionState getConnectionState();

    /**
     * Called by the connect/monitor/disconnect functions if the state of the connection changes.
     *
     * @param newState
     *         The new state to which the connection is being transitioned.
     */
    void setConnectionState(ConnectionState newState);

    /**
     * Called by the connect/monitor/disconnect functions to set a new connection status message.
     *
     * @param message
     *         The new status message.
     */
    void setConnectionStatusMessage(String message);

    /**
     * Returns the current state of indexing activity, or null if the connector does not support indexing.
     *
     * @return the current state of indexing activity
     */
    IndexingState getIndexingState();

    /**
     * Sets a list of transforms to apply to all indexed signals.
     *
     * @param transforms
     *         The list of transforms to apply to the signals.
     */
    void setTransforms(List<PropertyTransformer.Spec> transforms);

    /**
     * If the connector supports exporting from Seeq to the datasource, this method must be called
     * during each connection's Initialize() method.
     *
     * @param exportInterface
     *         The IExportSamples interface used to put samples in the external datasource.
     * @param exportConfig
     *         The configuration that dictates how exporting takes place and what signals need to
     *         be created in the external system to receive the exported data.
     */
    void initializeExport(ExportSamples exportInterface, ExportConnectionConfigV1 exportConfig);

    /**
     * Creates a new signal in the Seeq database or overwrites an existing item with the same
     * DatasourceClass/DatasourceId/DataId if it already exists.
     *
     * @param signalDefinition
     *         An Object describing the properties of the signal.
     * @return The list of items created in Seeq during this call. The return value will be null if the items are being
     *         queued into a batch.
     * @throws ApiException
     *         If there is an error communicating with Seeq server, or Seeq server returns an error code during
     *         communication.
     */
    List<ItemUpdateOutputV1> putSignal(SignalWithIdInputV1 signalDefinition) throws ApiException;

    /**
     * Writes the signal buffer to the Seeq application server. This function should be called if it is important that
     * the signals exist in the Seeq database before any other activity takes place.
     *
     * @return The list of items created in Seeq during this call.
     * @throws ApiException
     *         If there is an error communicating with Seeq server, or Seeq server returns an error code during
     *         communication.
     */
    List<ItemUpdateOutputV1> flushSignals() throws ApiException;

    /**
     * Creates a new user group in Seeq database or overwrites an existing user group with the same
     * DatasourceClass/DatasourceId/DataId if it already exists.
     *
     * @param userGroupDefinition
     *         An Object describing the properties of the user group
     * @return The list of items created in Seeq during this call. The return value will be null if the items are
     *         being queued into a batch.
     * @throws ApiException
     *         If there is an error communicating with Seeq server, or Seeq server returns an error code during
     *         communication.
     */
    List<ItemUpdateOutputV1> putUserGroup(UserGroupWithIdInputV1 userGroupDefinition) throws ApiException;

    /**
     * Writes the user group buffer to the Seeq application server. This function should be called if it is important
     * that the signals exist in the Seeq database before any other activity takes place.
     *
     * @return The list of items created in Seeq during this call.
     * @throws ApiException
     *         If there is an error communicating with Seeq server, or Seeq server returns an error code during
     *         communication.
     */
    List<ItemUpdateOutputV1> flushUserGroups() throws ApiException;

    /**
     * Creates a new condition in the Seeq database or overwrites an existing item with the same
     * DatasourceClass/DatasourceId/DataId if it already exists.
     *
     * @param conditionDefinition
     *         An Object describing the properties of the condition.
     * @return The list of items created in Seeq during this call. The return value will be null if the items are being
     *         queued into a batch.
     * @throws ApiException
     *         If there is an error communicating with Seeq server, or Seeq server returns an error code during
     *         communication.
     */
    List<ItemUpdateOutputV1> putCondition(ConditionUpdateInputV1 conditionDefinition) throws ApiException;

    /**
     * Writes the condition buffer to the Seeq application server. This function should be called if it is important
     * that the conditions exist in the Seeq database before any other activity takes place.
     *
     * @return The list of items created in Seeq during this call.
     * @throws ApiException
     *         If there is an error communicating with Seeq server, or Seeq server returns an error code during
     *         communication.
     */
    List<ItemUpdateOutputV1> flushConditions() throws ApiException;

    /**
     * Creates a new scalar in the Seeq database or overwrites an existing item with the same
     * DatasourceClass/DatasourceId/DataId if it already exists.
     *
     * Scalar values should be set in the Formula field of PutScalarInputV1, in a format that can be parsed by the Seeq
     * Formula language. Units should be set in the UnitOfMeasure field as they will be parsed more permissively and
     * applied to the Formula if they can be parsed to a Seeq standard unit. Unrecognized units in this field will
     * result in a  unitless Scalar, but unrecognized units in the Formula will lead to rejection of the Scalar.
     *
     * @param scalarDefinition
     *         An Object describing the properties of the scalar.
     * @return The list of items created in Seeq during this call. The return value will be null if the items are being
     *         queued into a batch.
     * @throws ApiException
     *         If there is an error communicating with Seeq server, or Seeq server returns an error code during
     *         communication.
     */
    List<ItemUpdateOutputV1> putScalar(PutScalarInputV1 scalarDefinition) throws ApiException;

    /**
     * Creates a new scalar in the Seeq database or overwrites an existing item with the same
     * DatasourceClass/DatasourceId/DataId if it already exists.  This method is deprecated - use PutScalarInputV1
     * as the argument instead, which enables use of previousDataId property.
     *
     * Scalar values should be set in the Formula field of PutScalarInputV1, in a format that can be parsed by the Seeq
     * Formula language. Units should be set in the UnitOfMeasure field as they will be parsed more permissively and
     * applied to the Formula if they can be parsed to a Seeq standard unit. Unrecognized units in this field will
     * result in a  unitless Scalar, but unrecognized units in the Formula will lead to rejection of the Scalar.
     *
     * @param scalarDefinition
     *         An Object describing the properties of the scalar.
     * @return The list of items created in Seeq during this call. The return value will be null if the items are being
     *         queued into a batch.
     * @throws ApiException
     *         If there is an error communicating with Seeq server, or Seeq server returns an error code during
     *         communication.
     */
    List<ItemUpdateOutputV1> putScalar(ScalarInputV1 scalarDefinition) throws ApiException;

    /**
     * Writes the scalar buffer to the Seeq application server. This function should be called if it is important
     * that the scalars exist in the Seeq database before any other activity takes place.
     *
     * @return The list of items created in Seeq during this call.
     * @throws ApiException
     *         If there is an error communicating with Seeq server, or Seeq server returns an error code during
     *         communication.
     */
    List<ItemUpdateOutputV1> flushScalars() throws ApiException;

    /**
     * Creates a new asset in the Seeq database.
     *
     * @param assetDefinition
     *         An Object describing the properties of the asset.
     * @return The list of items created in Seeq during this call. The return value will be null if the items are being
     *         queued into a batch.
     * @throws ApiException
     *         If there is an error communicating with Seeq server, or Seeq server returns an error code during
     *         communication.
     */
    List<ItemUpdateOutputV1> putAsset(PutAssetInputV1 assetDefinition) throws ApiException;

    /**
     * Creates a new asset in the Seeq database.  This method is deprecated - use PutAssetInputV1
     * as the argument instead, which enables use of previousDataId property.
     *
     * @param assetDefinition
     *         An Object describing the properties of the asset.
     * @return The list of items created in Seeq during this call. The return value will be null if the items are being
     *         queued into a batch.
     * @throws ApiException
     *         If there is an error communicating with Seeq server, or Seeq server returns an error code during
     *         communication.
     */
    @Deprecated
    List<ItemUpdateOutputV1> putAsset(AssetInputV1 assetDefinition) throws ApiException;

    /**
     * Writes the asset buffer to the Seeq application server. This function should be called if it is important that
     * the assets exist in the Seeq database before any other activity takes place.
     *
     * @return The list of items created in Seeq during this call.
     * @throws ApiException
     *         If there is an error communicating with Seeq server, or Seeq server returns an error code during
     *         communication.
     */
    List<ItemUpdateOutputV1> flushAssets() throws ApiException;

    /**
     * Creates a new asset in the Seeq database at the root of the asset tree.
     *
     * @param assetDefinition
     *         An Object describing the properties of the root asset.
     * @return The list of items created in Seeq during this call. The return value will be null if the items are being
     *         queued into a batch.
     * @throws ApiException
     *         If there is an error communicating with Seeq server, or Seeq server returns an error code during
     *         communication.
     */
    List<ItemUpdateOutputV1> putRootAsset(PutAssetInputV1 assetDefinition) throws ApiException;

    /**
     * Creates a new asset in the Seeq database at the root of the asset tree.  This method is deprecated - use
     * PutAssetInputV1 as the argument instead, which enables use of previousDataId property.
     *
     * @param assetDefinition
     *         An Object describing the properties of the root asset.
     * @return The list of items created in Seeq during this call. The return value will be null if the items are being
     *         queued into a batch.
     * @throws ApiException
     *         If there is an error communicating with Seeq server, or Seeq server returns an error code during
     *         communication.
     */
    List<ItemUpdateOutputV1> putRootAsset(AssetInputV1 assetDefinition) throws ApiException;

    /**
     * Writes the root asset buffer to the Seeq application server. This function should be called if it is important
     * that the root assets exist in the Seeq database before any other activity takes place.
     *
     * @return The list of items created in Seeq during this call.
     * @throws ApiException
     *         If there is an error communicating with Seeq server, or Seeq server returns an error code during
     *         communication.
     */
    List<ItemUpdateOutputV1> flushRootAssets() throws ApiException;

    /**
     * Establishes a relationship in the Seeq database.
     *
     * @param relationshipDefinition
     *         The relationship to create.
     * @return The list of items created in Seeq during this call. The return value will be null if the items are being
     *         queued into a batch.
     * @throws ApiException
     *         If there is an error communicating with Seeq server, or Seeq server returns an error code during
     *         communication.
     */
    List<ItemUpdateOutputV1> putRelationship(AssetTreeSingleInputV1 relationshipDefinition) throws ApiException;

    /**
     * Writes the buffer of relationships to the Seeq application server. This function should be called if it is
     * important that the relationships exist in the Seeq database before any other activity takes place.
     *
     * @return The list of items created in Seeq during this call.
     * @throws ApiException
     *         If there is an error communicating with Seeq server, or Seeq server returns an error code during
     *         communication.
     */
    List<ItemUpdateOutputV1> flushRelationships() throws ApiException;

    /**
     * Get information about the Seeq datasource this connection services.
     *
     * @return Information about the datasource this connection services
     */
    DatasourceOutputV1 getDatasource();

    /**
     * If called from the {@link IndexingDatasourceConnection#index(SyncMode)} method, archiving will be skipped
     * after indexing has finished. The default is that any items not present during a {@link SyncMode#FULL} sync
     * will be archived in Seeq; this skips that step.
     */
    void skipArchiving();

    /**
     * Store the specified additional properties and storedInSeeq flag on the datasource.
     *
     * @param storedInSeeq
     *         Whether this datasource's data will be stored in Seeq (true) or a remote datasource (false).
     * @param additionalProperties
     *         Additional properties that should be set on the datasource. The 'normal' datasource properties cannot be
     *         set with this parameter, only 'custom' properties.
     */
    void storeDatasourceProperties(boolean storedInSeeq, List<ScalarPropertyV1> additionalProperties);

    /**
     * Deletes the specified properties from the datasource
     *
     * @param additionalPropertiesToDelete
     *         The names of the properties to delete. The 'normal' datasource properties cannot be deleted with this
     *         method, only 'custom' properties.
     */
    void deleteDatasourceProperties(List<String> additionalPropertiesToDelete);

    /**
     * Prints diagnostic messages to the Seeq agent log file (and the console). The logging facility used is
     * <a href="https://logback.qos.ch/">Logback</a>. The log file is in the Seeq data folder:
     * "log/jvm-link/jvm-link.log". When using the Debugging Agent, logs are only sent to the debug console.
     *
     * @return logging interface
     */
    Logger log();

    /**
     * Sets the datasource item type cleanup filter.
     * By default, the filter is set to null (meaning no filter is applied, all items are considered in cleaning up
     * of stale items)
     *
     * @param itemTypeCleanupFilter
     *         List of item types to be considered in datasource cleanup process
     */
    void setDatasourceItemTypeCleanupFilter(List<String> itemTypeCleanupFilter);

    /**
     * Sets the datasource item DataId Regex filter. When set, only items having the DataId matching the RegEx will
     * be included in the cleanup process.
     * By default, the filter is set to null (meaning that all items are considered in cleaning up of stale items).
     *
     * @param regex
     *         the Regex to apply
     */
    void setDatasourceItemDataIdRegexFilter(String regex);

    /**
     * Sets the datasource item DataId exclude Regex filter. When set, the items having the DataID matching the RegEx
     * will be excluded from cleanup process.
     * By default, the filter is set to null (meaning that all items are considered in cleaning up of stale items).
     *
     * @param regex
     *         the Regex to apply
     */
    void setDatasourceItemDataIdExcludeRegexFilter(String regex);

    /**
     * Sets the datasource item name Regex filter. When set, only items having the name matching the RegEx will
     * be included in the cleanup process.
     * By default, the filter is set to null (meaning that all items are considered in cleaning up of stale items).
     *
     * @param regex
     *         the Regex to apply
     */
    void setDatasourceItemNameRegexFilter(String regex);

    /**
     * Sets the datasource item name exclude Regex filter. When set, the items having the name matching the RegEx will
     * be excluded from cleanup process.
     * By default, the filter is set to null (meaning that all items are considered in cleaning up of stale items).
     *
     * @param regex
     *         the Regex to apply
     */
    void setDatasourceItemNameExcludeRegexFilter(String regex);
}
