package com.seeq.link.sdk.interfaces;

import java.util.function.Consumer;

import com.seeq.link.messages.agent.AgentMessages;
import com.seeq.link.messages.agent.AgentMessages.DatasourceService;
import com.seeq.link.sdk.IndexingSchedule;
import com.seeq.link.sdk.IndexingState;
import com.seeq.utilities.exception.OperationCanceledException;

/**
 * This interface is a specialization of a general connection. The Seeq Server connection (exposed via ISeeqConnection)
 * is a general connection, while a datasource connection is only implemented by connections made from connectors.
 * Datasource connections have extra information like ConnectionID and DatasourceID.
 */
public interface DatasourceConnection extends Connection {

    /**
     * The id of the connection.
     *
     * @return the connection id
     */
    String getConnectionId();

    /**
     * A human readable name for the data source this connection is to.
     *
     * @return the datasource name
     */
    String getDatasourceName();

    /**
     * The unique identifier of the data source this connection is to.
     *
     * @return the datasource ID
     */
    String getDatasourceId();

    /**
     * The human readable name of the type of data source this connection is to.
     *
     * @return the datasource class
     */
    String getDatasourceClass();

    /**
     * The type(s) of data that this connection provides.
     *
     * @return the datasource service(s)
     */
    DatasourceService[] getServices();

    /**
     * Handles messages intended for this connector.
     *
     * @param data
     *         the data to be handled by the connection
     */
    void processMessage(AgentMessages.DataDocument data);

    /**
     * Spawn a thread and runs metadataSync() on it. This ensures the sync happens asynchronously and that there is only
     * one sync at a time, per connection.
     *
     * @param syncMode
     *         Indicates whether to perform a full or incremental sync
     * @param callback
     *         Callback that takes a {@link SyncResult} to indicate success/failure/cancellation
     */
    void spawnMetadataSync(SyncMode syncMode, Consumer<SyncResult> callback);

    /**
     * Sync metadata with the appserver.
     *
     * @param syncMode
     *         Indicates whether to perform a full or incremental sync
     * @throws OperationCanceledException
     *         Thrown when sync operation is interrupted by configuration reload or disabling.
     * @throws Exception
     *         Thrown when metadata sync fails for unexpected reasons and should be retried.
     */
    void metadataSync(SyncMode syncMode) throws OperationCanceledException, Exception;

    /**
     * Indicates whether pull requests are supported
     *
     * @return True, if pull requests are supported, false otherwise
     */
    boolean isPullDatasourceConnection();

    /**
     * Indicates whether metadata sync is supported
     *
     * @return True, if metadata sync is supported, false otherwise
     */
    boolean isIndexingDatasourceConnection();

    /**
     * Indicates whether metadata sync can be triggered from outside the connector (from scheduler, or requested on
     * demand from the UI)
     *
     * @return True, if metadata sync can be triggered from outside the connector, false otherwise
     */
    boolean isIndexingScheduleSupported();

    /**
     * The indexing schedule for this connection, including the time and period of delay.
     *
     * @return indexing schedule for this connection.
     */
    IndexingSchedule getIndexingSchedule();

    /**
     * The current state of indexing, including item counts and sync status.
     *
     * @return current state of indexing
     */
    IndexingState getIndexingState();

    /**
     * The current indexing requests's sync mode
     *
     * @return the sync mode of the current indexing request
     */
    SyncMode getCurrentIndexingRequestSyncMode();

    /**
     * Sets the current indexing requests's sync mode
     * @param syncMode the new sync mode of the current indexing request
     */
    void setCurrentIndexingRequestSyncMode(SyncMode syncMode);

    /**
     * Called by an Agent when the agent has changed the connection's configuration (usually due to the indexing
     * schedule being updated.
     */
    void saveConfig();
}
