package com.seeq.link.sdk.interfaces;

import java.io.IOException;

import org.slf4j.Logger;

import com.seeq.link.sdk.ConfigObject;

/**
 * Provides services to the class that manages connections to external systems.
 */
public interface ConnectorServiceV2 {

    /**
     * Reads the configuration object from disk from the expected location (configuration/link within the data folder).
     *
     * @param supportedObjects
     *         An array of object types to try loading. If the config file is not present, or none of the class types
     *         could successfully serialize the JSON, then the first item in the array is returned. This serves as
     *         mechanism for providing the "default" config object class.
     * @return The serialized config object, or the first item in {@code supportedObjects}.
     * @throws IOException
     *         Thrown if there is a problem reading the configuration file
     */
    ConfigObject loadConfig(ConfigObject[] supportedObjects) throws IOException;

    /**
     * Writes the configuration object to disk in the expected location (configuration/link within the data folder).
     *
     * @param configObject
     *         The configuration object to write
     */
    void saveConfig(ConfigObject configObject);

    /**
     * Must be used when connections are established to external systems. Note: This function is used even if
     * connections are disabled.
     *
     * @param connection
     *         The connection being established.
     */
    void addConnection(DatasourceConnectionV2 connection);

    /**
     * This should be called by Connectors in the beginning of initialize() to set the developer's name.
     * This name is displayed in the user interface so users know who to contact when they need support.
     */
    void setConnectorDeveloperName(String connectorDeveloperName);

    /**
     * This should be called by Connectors in the beginning of initialize() to set the developer's support URL.
     * This URL is displayed in the user interface so users know who to contact when they need support.
     */
    void setConnectorDeveloperSupportUrl(String connectorDeveloperSupportUrl);

    /**
     * @return the name of the connector's developer, as set by setConnectorDeveloperName().
     */
    String getConnectorDeveloperName();

    /**
     * @return the support URL of the connector's developer, as set by setConnectorDeveloperSupportUrl().
     */
    String getConnectorDeveloperSupportUrl();

    /**
     * Retrieves the agent service that can be used to access the REST API directly, among other things.
     *
     * @return The agent service.
     */
    AgentService getAgentService();

    /**
     * Prints diagnostic messages to the Seeq agent log file (and the console). The logging facility used is
     * <a href="https://logback.qos.ch/">Logback</a>. The log file is in the Seeq data folder:
     * "log/jvm-link/jvm-link.log". When using the Debugging Agent, logs are only sent to the debug console.
     *
     * @return logging facility
     */
    Logger log();
}
