package com.seeq.link.sdk.interfaces;

import java.util.List;

import com.seeq.link.messages.agent.AgentMessages;
import com.seeq.link.sdk.interfaces.Connection.StateChangedEventArgs;
import com.seeq.link.sdk.utilities.Event;

/**
 * <p>
 * A connector handles configuration and creation of connections.
 * </p>
 * <p>
 * Connectors are loaded via the {@link java.util.ServiceLoader} mechanism. They need to be registered
 * under {@code META-INF/services/com.seeq.link.sdk.interfaces.Connector}
 * </p>
 */
public interface Connector {

    /**
     * The name of the connector, as determined by the connector. This String may contain a unique identifier for the
     * external database that is being connected to.
     *
     * @return the name
     */
    String getName();

    /**
     * Initializes a connector with the agent service interface it will use to communicate to Seeq Server. The
     * implementation of this function can choose to automatically establish connections based on a global registry
     * (like the PI "Known Servers Table").
     *
     * @param agentService
     *         The agent that is hosting this connector.
     * @throws Exception
     *         Thrown if connector had an issue initializing.
     */
    void initialize(AgentService agentService) throws Exception;

    /**
     * Performs cleanup of a connector as necessary, removing all connections and releasing all resources.
     */
    void destroy();

    /**
     * Get all the connections that this connector has made.
     *
     * @return A Collection of connections.
     */
    List<DatasourceConnection> getConnections();

    /**
     * Event for state changes of the connector's connection(s).
     *
     * @return the state changed event
     */
    Event<StateChangedEventArgs> getConnectionStateChangedEvent();

    /**
     * Called by an Agent when a data document is received from Seeq Server. It is the connector's responsibility to
     * determine whether to handle the document.
     *
     * @param destinationConnectorInstanceId
     *         A connector-specific identifier used to route the message within the connector.
     * @param document
     *         The message payload.
     * @return true if the message was processed by the connector, false if it was ignored
     */
    boolean processMessage(String destinationConnectorInstanceId, AgentMessages.DataDocument document);

    /**
     * Called by a connection when it wants its configuration saved.
     */
    void saveConfig();
}
