package com.seeq.link.sdk.interfaces;

import java.time.Duration;

import com.seeq.link.sdk.utilities.Event;

/**
 * Provides methods for establishing, monitoring and disabling connections to external systems. The external system may
 * be a connector datasource (and therefore an {@link DatasourceConnection}) or the Seeq Server (SeeqConnection).
 */
public interface Connection {

    class StateChangedEventArgs {
        private final Connection sender;
        private final ConnectionState state;
        private final String message;

        public StateChangedEventArgs(Connection sender, ConnectionState state, String message) {
            this.sender = sender;
            this.state = state;
            this.message = message;
        }

        public Connection getSender() {
            return this.sender;
        }

        public ConnectionState getState() {
            return this.state;
        }

        public String getMessage() {
            return this.message;
        }
    }

    /**
     * The set of possible connection states to an external entity.
     */
    enum ConnectionState {

        /**
         * The connector will not attempt to contact the external entity.
         */
        DISABLED,

        /**
         * The connector has attempted to connect with the external entity, but has not yet been successful.
         */
        CONNECTING,

        /**
         * The connector has successfully connected to the external entity.
         */
        CONNECTED,

        /**
         * The connector has attempted to connect or has previously successfully connected to the external entity, but
         * is now disconnected. Re-connection will be attempted after the delay specified in BaseReconnectDelay.
         */
        DISCONNECTED
    }

    /**
     * Initializes a connection, attempting to connect if enabled.
     *
     */
    void initialize();

    /**
     * Disconnects from external systems and kills the monitoring thread.
     */
    void destroy();

    /**
     * A summary of the connection status, including any connection failure reason.
     *
     * @return summary of the connection status, including any connection failure reason.
     */
    String getConnectionMessage();

    /**
     * Time to wait before reconnecting. Default is 5 seconds. There is an "exponential backoff" algorithm at play that
     * will increase the reconnect delay as repeated attempts fail. That's why this is the minimum delay.
     *
     * @return minimum reconnect delay
     */
    Duration getMinReconnectDelay();

    /**
     * Time to wait before reconnecting. Default is 5 seconds. There is an "exponential backoff" algorithm at play that
     * will increase the reconnect delay as repeated attempts fail. That's why this is the minimum delay.
     *
     * @param value
     *         minimum reconnect delay
     */
    void setMinReconnectDelay(Duration value);

    /**
     * Maximum time to delay a reconnect attempt. Default is 5 minutes. There is an "exponential backoff" algorithm at
     * play that will increase the reconnect delay as repeated attempts fail.
     *
     * @return maximum reconnect delay
     */
    Duration getMaxReconnectDelay();

    /**
     * Maximum time to delay a reconnect attempt. Default is 5 minutes. There is an "exponential backoff" algorithm at
     * play that will increase the reconnect delay as repeated attempts fail.
     *
     * @param value
     *         maximum reconnect delay
     */
    void setMaxReconnectDelay(Duration value);

    /**
     * The current state of the connection.
     *
     * @return state of the connection
     */
    ConnectionState getState();

    /**
     * Event for state changes of the connection.
     *
     * @return the state changed event
     */
    Event<StateChangedEventArgs> getStateChangedEvent();

    /**
     * Enable connection and call the connect() method. The connection will pass through a Connecting state first.
     */
    void enable();

    /**
     * Disable connections to the server. This calls the disconnect() method in the derived class. Reconnections will
     * not occur if/until a Connect() call is made.
     */
    void disable();

    /**
     * Time between testing of the connection vitality. Default should be 5 seconds.
     *
     * @return the monitor period
     */
    Duration getMonitorPeriod();

    /**
     * Time between testing of the connection vitality. Default should be 5 seconds.
     *
     * @param value
     *         - the new monitor period
     */
    void setMonitorPeriod(Duration value);
}
