package com.seeq.link.sdk.interfaces;

import java.io.IOException;
import java.util.function.Consumer;

import com.seeq.link.sdk.ConfigObject;

/**
 * Provides methods to load and save configuration objects. The location of the data store is abstracted from the caller
 * at this level: Behind the scenes, the ConfigService may be retrieving configuration from Appserver and/or from the
 * local file system using {@link ConfigObjectProvider}. But that is not exposed to the agents/connectors.
 *
 * This is the interface that agents/connectors will use to retrieve configuration data.
 */
public interface ConfigService {

    /**
     * Initializes the service with its {@link FileConfigObjectProvider} (used to retrieve configuration from disk)
     * and its {@link RemoteConfigObjectProvider} (used to retrieve the remote configuration).
     *
     * @param fileConfigObjectProvider
     *         Interface used to retrieve configuration from disk.
     * @param remoteConfigObjectProvider
     *         Interface used to retrieve remote configuration.
     */
    void initialize(
            FileConfigObjectProvider fileConfigObjectProvider,
            RemoteConfigObjectProvider remoteConfigObjectProvider);

    /**
     * Loads a configuration Object from the config data store. This function requires an array of config objects to use
     * as a means of discovering what Object type is encoded in the store.
     *
     * @param name
     *         The name of configuration store, which will normally be the connector name.
     * @param supportedConfigObjects
     *         An array of ConfigObject instances that represent the possible Object types that can be successfully
     *         retrieved from the store. If the persisted Object type does not match anything in the list, then the
     *         first item in the array is passed back (being effectively a means to have a 'default' config).
     * @return The configuration Object, which must be cast to a ConfigObject-derived class.
     * @throws IOException
     *         Thrown if JSON file could not be successfully deserialized.
     */
    ConfigObject loadConfigObject(String name, ConfigObject[] supportedConfigObjects) throws IOException;

    /**
     * Saves a configuration Object to the config data store. The config object's type will be encoded in 'Version'
     * field in the data store.
     *
     * @param name
     *         The name of configuration store, which will normally be the connector name.
     * @param configObject
     *         The configuration Object to serialize to the store.
     */
    void saveConfigObject(String name, Object configObject);

    /**
     * Deprecated: All configuration should now come from configuration files.
     *
     * Retrieves an argument that was passed in on the process's command line. These can be used to override saved
     * configuration data, usually for testing purposes.
     *
     * @param argumentName
     *         Name of the command line argument to retrieve.
     * @return Value of the command line argument.
     */
    @Deprecated
    String getGlobalArgument(String argumentName);

    /**
     * Registers a callback for when the configuration has changed.
     *
     * @param name
     *         Name of the configuration object you're interested in.
     * @param callback
     *         Callback function that takes the name as its only parameter.
     */
    void registerChangeCallback(String name, Consumer<String> callback);

    /**
     * Unregisters a callback previously registered.
     *
     * @param name
     *         Name of the configuration object you're interested in.
     */
    void unregisterChangeCallback(String name);
}
