package com.seeq.link.sdk.interfaces;

import java.io.IOException;

import com.seeq.link.sdk.ConfigObject;
import com.seeq.link.sdk.ConfigObjectWrapper;

/**
 * Provides methods to load and save configuration objects. This interface is implemented by a particular type of
 * datastore. For example, one may be a file-based Object provider and one may be an AppServer-based Object provider.
 */
public interface ConfigObjectProvider {

    /**
     * Loads a configuration Object from the config data store and retrieves its metadata. This function requires an
     * array of config objects to use as a means of discovering what Object type is encoded in the store.
     *
     * @param name
     *         The name of configuration store, which will normally be the connector name.
     * @param supportedConfigObjects
     *         An array of ConfigObject instances that represent the possible Object types that can be successfully
     *         retrieved from the store. If the persisted Object type does not match anything in the list, then the
     *         first item in the array is passed back (being effectively a means to have a 'default' config).
     * @return The configuration Object, which must be cast to a ConfigObject-derived class wrapped into a
     *         {@link ConfigObjectWrapper} together with its metadata.
     * @throws IOException
     *         Thrown if JSON file could not be successfully deserialized.
     */
    ConfigObjectWrapper loadConfigObject(String name, ConfigObject[] supportedConfigObjects) throws IOException;

    /**
     * Saves a configuration Object to the config data store. The config object's type will be encoded in 'Version'
     * field in the data store.
     *
     * @param name
     *         The name of configuration store, which will normally be the connector name.
     * @param configObject
     *         The configuration Object to serialize to the store.
     */
    void saveConfigObject(String name, Object configObject);
}
