package com.seeq.link.sdk.interfaces;

import com.seeq.link.sdk.utilities.ThreadCollection;
import com.seeq.utilities.ManualResetEvent;

/**
 * Interface for concurrent requests handler used to limit the maximum requests which can run in parallel.
 * Depending on usage, we could limit the number of concurrent requests per connection or per connector.
 */
public interface ConcurrentRequestsHandler {

    /**
     * Ensures that only a specific number of requests will be executed in parallel.
     *
     * @param request
     *         Concurrent task to be executed
     * @param threadCollection
     *         Thread collection used to spawn a new thread for the request
     * @param timeoutMillis
     *         Amount of time to wait for request completion
     * @param requestId
     *         Request identifier
     * @param requestThreadStartedEvent
     *         Manual reset event which will be set after the thread of the request is started. The thread may be
     *         waiting for permission to run its job, but we are sure the thread is started when the event is set.
     * @return the thread of the executed request
     */
    Thread runWhenPermitted(Runnable request, ThreadCollection threadCollection,
            long timeoutMillis, long requestId, ManualResetEvent requestThreadStartedEvent);

    /**
     * @return the maximum number of concurrent requests. When this limit is exceeded, any call of
     *         {@code runWhenPermitted} will block the execution of the request until an execution slot is available.
     */
    int getMaxConcurrentRequests();

    /**
     * @return the number of registered requests. May be different than the number of requests currently executed.
     */
    int getRegisteredRequestsCount();
}