package com.seeq.link.sdk.interfaces;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.Nullable;

import lombok.Data;
import lombok.NonNull;

/**
 * The authentication result as communicated back by the connector
 */
@Data
public class AuthResult {
    private AuthResult(
            boolean authenticated,
            @Nullable String userId,
            @Nullable String errorMessageOrNull,
            @Nullable String continuationTokenOrNull) {
        this.authenticated = authenticated;
        this.userId = userId;
        this.errorMessage = errorMessageOrNull;
        this.continuation = continuationTokenOrNull;
    }

    /**
     * Create an {@link AuthResult} representing a <em>successful</em> authentication.
     *
     * <ul>
     *  <li>{@link #authenticated} will be {@code true}</li>
     *  <li>{@link #userId} will be non-{@code null}</li>
     *  <li>{@link #errorMessage} will be {@code null}</li>
     *  <li>{@link #continuation} will be non-{@code null}</li>
     * </ul>
     *
     * @param userId
     *         the user ID, which is connector-specific
     * @return a freshly constructed {@link AuthResult} which <em>will</em> result in valid credentials to use Seeq,
     *         unless later {@link #unauthorized(String)}.
     */
    public static AuthResult success(@NonNull String userId) {
        return new AuthResult(true, userId, null, null);
    }

    /**
     * Create an {@link AuthResult} representing the need to additional round-trips.
     *
     * <ul>
     *  <li>{@link #authenticated} will be {@code false}</li>
     *  <li>{@link #userId} will be {@code null}</li>
     *  <li>{@link #errorMessage} will be {@code null}</li>
     *  <li>{@link #continuation} will be non-{@code null}</li>
     * </ul>
     * <br>
     * See: {@code com.seeq.link.connectors.windowsauth}
     *
     * @param continuationToken
     *         the continuation token
     * @return a freshly constructed {@link AuthResult} which must undergo negotiation before (possibly) continuing
     *         to authenticate.
     */
    public static AuthResult negotiate(@NonNull String continuationToken) {
        return new AuthResult(false, null, null, continuationToken);
    }

    /**
     * Create an {@link AuthResult} representing a failure to authenticate or authorize.
     *
     * @param errorMessage
     *         the error message indicating what failed.
     * @return a freshly constructed {@link AuthResult} which will not result in valid credentials to use Seeq.
     */
    public static AuthResult failed(@NonNull String errorMessage) {
        return new AuthResult(false, null, errorMessage, null);
    }

    /**
     * Create an {@link AuthResult} representing a failure to <em>authorize</em>, despite authenticating.
     * The additional parameters allow the system to perform group sync.
     *
     * @param errorMessage
     *         the error message indicating why the user is not able to use Seeq.
     * @param userId
     *         the user ID
     * @param groups
     *         the groups to sync, if any
     * @return a freshly constructed {@link AuthResult} which will not result in valid credentials to use Seeq.
     */
    public static AuthResult unauthorized(@NonNull String errorMessage,
            @NonNull String userId,
            @Nullable List<GroupInfo> groups) {
        AuthResult result = new AuthResult(false, userId, errorMessage, null);
        result.groups = groups;
        return result;
    }

    /**
     * Create an {@link AuthResult} representing a failure to <em>authorize</em>, despite authenticating.
     * All additional data (name, groups, etc.) will be copied from {@code this} instance.
     *
     * @param errorMessage
     *         the error message indicating why the user is not able to use Seeq.
     * @return a freshly constructed {@link AuthResult} which will not result in valid credentials to use Seeq,
     *         yet which carries a copy of all of the user information present on {@code this} instance.
     */
    public AuthResult unauthorized(@NonNull String errorMessage) {
        AuthResult result = new AuthResult(false, this.userId, errorMessage, this.continuation);
        result.name = this.name;
        result.firstName = this.firstName;
        result.lastName = this.lastName;
        result.emailAddress = this.emailAddress;
        result.securityId = this.securityId;
        result.groups = this.groups == null ? null : new ArrayList<>(this.groups);
        return result;
    }

    /**
     * Represents authentication and authorization success.
     * <p>If {@code true}, will always include a {@link #userId}, and will never have an {@link #errorMessage} or
     * {@link #continuation} token.</p>
     * <p>If {@code false}, one of the {@link #continuation} token or {@link #errorMessage} will be set.</p>
     *
     * <pre>
     *   {@code authenticated} &rarr; {@link #userId}
     * &not; {@code authenticated} &rarr; {@link #continuation} \u22bb {@link #errorMessage}
     * </pre>
     */
    private final boolean authenticated;

    /**
     * Will be non-null if authentication was successful.
     * <p>Never present in the case of a {@link #continuation} token.</p>
     * <p><em>May</em> be non-null if authorization {@link #failed(String)} after authentication.</p>
     *
     * <pre>
     * &not; {@code userId} &rarr; &not; {@link #authenticated}
     *   {@code userId} &rarr; &not; {@link #continuation}
     * </pre>
     */
    @Nullable
    private final String userId;

    /**
     * Will be non-null if authentication or authorization failed.
     * Mutually exclusive with {@link #continuation} token.
     *
     * <pre>
     * {@code errorMessage} &rarr; &not; {@link #authenticated}
     * {@code errorMessage} &rarr; &not; {@link #continuation}
     * </pre>
     */
    @Nullable
    private final String errorMessage;

    /**
     * Represents a continuation token, requiring an additional round-trip before authentication can complete.
     * Mutually exclusive with {@link #errorMessage} and {@link #userId}.
     *
     * <pre>
     * {@code continuation} &rarr; &not; {@link #authenticated}
     * {@code continuation} &rarr; &not; {@link #userId}
     * {@code continuation} &rarr; &not; {@link #errorMessage}
     * </pre>
     */
    @Nullable
    private final String continuation;

    // Retrieved if possible from Auth System. Existence depends on configuration.
    @Nullable
    private String name;

    // Retrieved if possible from Auth System. Existence depends on configuration.
    @Nullable
    private String firstName;

    // Retrieved if possible from Auth System. Existence depends on configuration.
    @Nullable
    private String lastName;

    // Retrieved if possible from Auth System. Existence depends on configuration.
    @Nullable
    private String emailAddress;

    /**
     * Unique Security identifier of the user.
     * For Windows users this is the group identifier in S- format (like {@code S-1-34-2345}).
     */
    @Nullable
    private String securityId;

    @Nullable
    private List<GroupInfo> groups;
}
