package com.seeq.link.sdk.interfaces;

import java.nio.file.Path;
import java.time.Duration;
import java.util.Collection;

import com.seeq.link.messages.agent.AgentMessages;
import com.seeq.link.sdk.export.ExportTaskScheduler;
import com.seeq.link.sdk.utilities.BatchSizeHelper;

/**
 * This is the connector's "catch all" interface for interacting with the agent, and includes facilities for
 * communicating with the Seeq Server (via the Agent).
 */
public interface AgentService {

    /**
     * Returns true if a full connection to Seeq Server has been established.
     *
     * @return true if a full connection to Seeq Server has been established
     */
    boolean isSeeqServerConnected();

    /**
     * The service used to load/save configuration and retrieve global arguments.
     *
     * @return the config service
     */
    ConfigService getConfigService();

    /**
     * Provides instances of API objects.
     *
     * @return API provider instance
     */
    SeeqApiProvider getApiProvider();

    /**
     * Provides instances of API objects that are used during indexing.
     * Note: Indexing Api Provider has a different retryTimeout than the default Api Provider. Retry timeout of
     * Indexing Api Provider can be configured in the Agent configuration
     *
     * @return API provider instance
     */
    SeeqApiProvider getIndexingApiProvider();

    /**
     * The display name of the Agent.
     *
     * @return display name of the Agent
     */
    String getDisplayName();

    /**
     * The identification of the Agent.
     *
     * @return the identification of the Agent
     */
    String getAgentIdentification();

    /**
     * The path of the data folder for configuration, logs, and Seeq data
     *
     * @return the folder path
     */
    Path getDataFolder();

    /**
     * The path of the Seeq global folder
     */
    Path getGlobalFolder();

    /**
     * @return True if the agent is remote, false if it is local to the Seeq Server
     */
    boolean isRemoteAgent();


    /**
     * External URL of the Seeq Server to which this agent is connected. Note that this URL is the "external" URL
     * that is accessed by users. It should not be used to programmatically create connections to the Seeq Server
     * because the agent may not be able to resolve the external hostname. Use the ApiProvider members of this
     * interface to obtain Seeq SDK interfaces configured with the correct "private" URL (if applicable).
     */
    String getSeeqServerURL();

    /**
     * Instantiates a new BatchSizeHelper object with default values.
     *
     * This exists for mocking purposes.
     *
     * @return New BatchSizeHelper object.
     */
    BatchSizeHelper createBatchSizeHelper();

    /**
     * Instantiates a new BatchSizeHelper object.
     *
     * This exists for mocking purposes.
     *
     * @param initialSize
     *         The initial value for the recommended batch size.
     * @param maximumDuration
     *         The maximum duration for processing that the caller wants a single batch to take.
     * @return New BatchSizeHelper object.
     */
    BatchSizeHelper createBatchSizeHelper(int initialSize, Duration maximumDuration);

    /**
     * As part of the {@link AgentService} interface, this function allows Connections to send data to Seeq Server by
     * wrapping DataDocuments in an AgentMessage.
     *
     * @param datasourceConnection
     *         The connection that is sending the message.
     * @param dataDocument
     *         The data of the message to be sent.
     */
    void sendMessage(DatasourceConnection datasourceConnection, AgentMessages.DataDocument dataDocument);

    /**
     * Update the Seeq Server with status information. This happens automatically on a regular basis and should only
     * be called when it's desirable for the Seeq Server to get the update immediately.
     */
    void sendAgentInfoToServer();

    /**
     * Requests a restart of the Agent. This should be used only as a last resort for bugs in the
     * third-party datasource libraries that cannot otherwise be handled by more surgical reset logic
     * that is connection-specific. Since this command will restart the Agent itself, all existing
     * requests across all connectors hosted by the Agent will be unceremoniously aborted and all
     * connections dropped, with a short outage as the Agent is restarted by the Seeq "Supervisor"
     * process.
     */
    void requestRestart();

    /**
     * Requests a FULL re-indexing of the given Connection.
     */
    void requestIndex(DatasourceConnection connection);

    /**
     * Requests a re-indexing of the given Connection.
     */
    void requestIndex(DatasourceConnection connection, SyncMode syncMode);

    /**
     * Marks the connections as reinitialized when sending the agent status
     */
    void markConnectionsAsReinitialized(Collection<String> connectionIds);

    /**
     * The TaskScheduler object that is used for all export jobs. This is shared at the agent level
     * so that export jobs across all connectors and connections are routed through a single
     * scheduler and the admin has complete control over how much load this agent puts on the Seeq
     * Server due to export activity.
     */
    ExportTaskScheduler getExportTaskScheduler();
}
