package com.seeq.link.sdk.interfaces;

import java.util.List;
import java.util.function.BiFunction;
import java.util.stream.Stream;

import org.jetbrains.annotations.Nullable;

import com.seeq.link.sdk.utilities.Sample;

/**
 * This interface defines the functions that must be implemented by the class that checks if a script is respecting
 * the established contract.
 */
public interface AddOnCalculationValidator {

    /**
     * Sets the stream of list of aligned samples into the detector.
     * The list of samples is complete and one can take any number of them to be used in validation.
     * For speed considerations it is recommended to only take few (e.g. 100)
     *
     * @param alignedSignalsStreamComplete
     *         complete stream of aligned list of samples
     * @return the instance of the detector
     */
    AddOnCalculationValidator setAlignedSignalsStreamComplete(Stream<List<Sample>> alignedSignalsStreamComplete);

    /**
     * During validation the detector will need to call the script for a subset of the samples provided in the
     * complete stream
     *
     * @param signalFragmentScriptCaller
     *         function providing the possibility to call the script for a subset of samples
     * @return the instance of the detector
     */
    AddOnCalculationValidator setSignalFragmentScriptCaller(
            BiFunction<Integer, Stream<List<Sample>>, Stream<Sample>> signalFragmentScriptCaller
    );

    /**
     * Set the name of the script into the detector. Script name may be used by the detector in error reporting.
     *
     * @param scriptName
     *         name of the script
     * @return the instance of detector
     */
    AddOnCalculationValidator setScriptName(String scriptName);

    /**
     * Set the window size for windowed calculations
     *
     * @param windowSize
     *         windowSize for windowed calculations, otherwise null
     * @return the instance of detector
     */
    AddOnCalculationValidator setWindowSize(@Nullable Long windowSize);

    /**
     * Return the minimum number of samples required for validation
     *
     * @return minimum number of samples required for validation
     */
    int getMinimumNumberOfSamples();

    /**
     * Does the validation that checks if the script is correctly made. In case the contract break is detected, a
     * RuntimeException is thrown. In case no problem is detected in the script, a subsequent call to
     * isContractRespected should return true.
     */
    void doValidate();

    /**
     * Gets the value of contract respected flag.
     *
     * @return true if the script is respecting the contract
     */
    boolean isValid();
}
