package com.seeq.link.sdk.interfaces;

import java.util.List;
import java.util.concurrent.locks.Lock;
import java.util.stream.Stream;

import javax.annotation.Nullable;

import com.seeq.link.messages.connector.extcalc.ExternalCalculationMessages.ExternalCalculationRequestMessage;
import com.seeq.link.sdk.utilities.Sample;

/**
 * Implemented by connections that offers external calculation services
 */
public interface AddOnCalculationDatasourceConnection {
    /**
     * Delegates calculation to the external script and obtain the result back from it.
     *
     * @param script
     *         Name of the script
     * @param alignedSignalsStream
     *         A stream of aligned samples given to the script for computing the result. Note: this parameter uses
     *         the same structure as AlignedIterator.
     * @param signalDataTypes
     *         Data type for value of samples of each signal provided in the input
     * @param numberOfSignals
     *         Number of signals the input stream contains
     * @param numberOfSamplesPerSignal
     *         Number of samples per signal
     * @param windowSize
     *         Number of sample in a window, null means the calculation does not use a window
     * @return A stream used to retrieve the samples. A stream is used so the connection author can avoid loading
     *         all samples into memory, if the underlying datasource interface allows it.
     */
    Stream<Sample> getCalculatedResults(String script, Stream<List<Sample>> alignedSignalsStream,
            AddOnCalcSignalDataType[] signalDataTypes,
            int numberOfSignals, int numberOfSamplesPerSignal, @Nullable Long windowSize);

    /**
     * Validates the calculation request, checking that all optional parameter is set according to the script input mode
     *
     * @param request
     *         request to validate
     */
    void validateRequest(ExternalCalculationRequestMessage request);

    /**
     * In case a script has been changed we need to check again if the script breaks the contract. Contract break
     * detection is done only one time.
     *
     * @return true if the script needs to be rechecked for breaking the established contract
     */
    boolean needsValidation();

    /**
     * Returns a new instance of a validator that is used to validate the script.
     *
     * @return a new validator
     */
    AddOnCalculationValidator createValidator();

    /**
     * Marks (into the connection) the script as breaking no contract. Once marked the detection is suppressed until
     * {@link #needsValidation()} returns again true. This should be reset to true when the script is
     * changed.
     */
    void markAsValid();

    /**
     * @return a lock which is used to ensure that the same script is not validated more than once even if more
     *         requests are triggered at the same time
     */
    Lock getValidationLock();
}
