package com.seeq.link.sdk.export;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.time.Duration;
import java.time.ZonedDateTime;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.commons.lang.NotImplementedException;
import org.apache.commons.lang3.StringUtils;

import com.seeq.api.ItemsApi;
import com.seeq.link.sdk.utilities.RequestTimings;
import com.seeq.link.sdk.utilities.TimeInstant;
import com.seeq.model.ItemOutputV1;
import com.seeq.model.PropertyOutputV1;
import com.seeq.model.ScalarPropertyV1;

import lombok.extern.slf4j.Slf4j;

/**
 * Represents the status of the export, which is serialized to properties for the item
 * in Seeq (for example, "Export - Status" and "Export - Message").
 */
@Slf4j
public class ExportStatus {
    public static final String Success = "SUCCESS";
    public static final String Failed = "FAILED";

    // These fields are serialized/de-serialized to/from Seeq item properties by reflection,
    // which is why Last_Write_Time is in Title_Snake_Case... the underscores become spaces.

    /**
     * Either SUCCESS or FAILED based on whether the last attempt to export was successful.
     */
    public String Status;

    /**
     * A helpful message, especially for a failure, for the user to understand the status of
     * export activity.
     */
    public String Message;

    /**
     * The timestamp of the latest sample exported to the datasource.
     */
    public TimeInstant Cursor;

    /**
     * The wall-clock time of the most recent export (successful or not).
     */
    public ZonedDateTime Last_Write_Time;

    /**
     * A measure of how "expensive" the export operation is from the standpoint of retrieving
     * the data and performing the calculation in Seeq. The units of this toll are "seconds per
     * day", meaning that it takes so many seconds to do the calculation per day of time range.
     */
    public double Toll;

    /**
     * Reads the export status from an item's properties, if they exist.
     *
     * @param itemsApi
     *         The ItemsApi interface to use to communicate with Seeq Server
     * @param itemID
     *         The ID of the item from which to read
     */
    public static ExportStatus read(ItemsApi itemsApi, String itemID) {
        ExportStatus status = new ExportStatus();

        ItemOutputV1 item = itemsApi.getItemAndAllProperties(itemID);

        for (PropertyOutputV1 property : item.getProperties()) {
            if (!property.getName().startsWith("Export - ")) {
                continue;
            }

            String fieldName = property.getName()
                    .replace("Export - ", "")
                    .replace(" ", "_");

            Field fieldInfo;
            try {
                fieldInfo = status.getClass().getField(fieldName);
            } catch (NoSuchFieldException e) {
                // Ignore this property, it's not a field of this Object (or maybe it's an old field)
                continue;
            }

            if (StringUtils.isBlank(property.getValue())) {
                // Leave the value as the default for the class
                continue;
            }

            try {
                if (fieldInfo.getType() == boolean.class) {
                    fieldInfo.setBoolean(status, Boolean.parseBoolean(property.getValue()));
                } else if (fieldInfo.getType() == double.class) {
                    fieldInfo.setDouble(status, Double.parseDouble(property.getValue()));
                } else if (fieldInfo.getType() == String.class) {
                    fieldInfo.set(status, property.getValue());
                } else if (fieldInfo.getType() == TimeInstant.class) {
                    fieldInfo.set(status, TimeInstant.parseIso(property.getValue()));
                } else if (fieldInfo.getType() == ZonedDateTime.class) {
                    fieldInfo.set(status, ZonedDateTime.parse(property.getValue()));
                } else {
                    throw new NotImplementedException(
                            String.format("Unrecognized field type for %s: %s", fieldName, fieldInfo.getType()));
                }
            } catch (Exception e) {
                LOG.error(String.format("Couldn't parse value of '%s' property for item %s ('%s')", fieldName, itemID,
                        property.getValue()), e);
            }
        }

        return status;
    }

    /**
     * Writes the export status to an item's properties.
     *
     * @param itemsApi
     *         The ItemsApi interface to use to communicate with Seeq Server
     * @param itemID
     *         The ID of the item from which to read
     */
    public void write(ItemsApi itemsApi, String itemID) {
        List<ScalarPropertyV1> properties = Stream.of(this.getClass().getFields())
                .filter(f -> Modifier.isPublic(f.getModifiers()) && !Modifier.isStatic(f.getModifiers()))
                .map(f -> {
                    String val = "";
                    try {
                        Object obj = f.get(this);
                        if (obj != null) {
                            if (f.getType().toString().contains("DateTime")) {
                                val = ((ZonedDateTime) obj).toString();
                            } else if (f.getType().toString().endsWith("TimeInstant")) {
                                val = obj.toString();
                            } else if (f.getType().toString().equalsIgnoreCase("double")) {
                                double d = (double) obj;
                                val = String.format("%.3f", d);
                            } else {
                                val = obj.toString();
                            }
                        }
                    } catch (IllegalAccessException e) {
                        // Fall through
                    }

                    ScalarPropertyV1 scalarProperty = new ScalarPropertyV1();
                    scalarProperty.setName("Export - " + f.getName().replace("_", " "));
                    scalarProperty.setValue(val);
                    return scalarProperty;
                })
                .collect(Collectors.toList());

        itemsApi.setProperties(itemID, properties);
    }

    public static final double S_PER_MS = 1 / 1000.0;
    public static final double DAYS_PER_MS = 1 / 1000.0 / 60.0 / 60.0 / 24.0;

    public void calculateAndSetToll(RequestTimings requestTimings, TimeInstant startTime, TimeInstant endTime) {
        if (requestTimings.getDatasourceSamplesCount() == 0 && requestTimings.getDatasourceCapsulesCount() == 0) {
            // No samples returned so it's tough to infer any toll
            return;
        }

        double timeTakenInSeconds =
                requestTimings.getCalcEngineProcessingDuration().plus(requestTimings.getDatasourceDuration())
                        .toMillis() * S_PER_MS;

        double rangeInDays = Duration.between(startTime.toDateTime(), endTime.toDateTime()).toMillis() * DAYS_PER_MS;

        this.Toll = timeTakenInSeconds / rangeInDays;
    }

    public double calculateExpectedDuration(TimeInstant startTime, TimeInstant endTime) {
        Duration timeRange = Duration.between(startTime.toDateTime(), endTime.toDateTime());
        return this.Toll * timeRange.toMillis() * ExportStatus.DAYS_PER_MS;
    }
}
