package com.seeq.link.sdk.export;

import java.time.Duration;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.seeq.ApiException;
import com.seeq.api.FormulasApi;
import com.seeq.api.ItemsApi;
import com.seeq.link.sdk.interfaces.Connection;
import com.seeq.link.sdk.interfaces.DatasourceConnectionServiceV2;
import com.seeq.link.sdk.utilities.RequestTimings;
import com.seeq.link.sdk.utilities.Sample;
import com.seeq.link.sdk.utilities.SeeqApiRequestHelper;
import com.seeq.link.sdk.utilities.SeeqSDKHelper;
import com.seeq.link.sdk.utilities.SeeqSdkApiResponse;
import com.seeq.link.sdk.utilities.Stopwatch;
import com.seeq.link.sdk.utilities.TimeInstant;
import com.seeq.model.FormulaRunOutputV1;
import com.seeq.utilities.ManualResetEvent;
import com.seeq.utilities.SeeqNames;
import com.seeq.utilities.exception.OperationCanceledException;

import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;

/**
 * Facilitates all aspects of exporting data to the datasource.
 */
@Slf4j
public class ExportOrchestrator {
    private static final Duration MinimumStandardLatency = Duration.ofMinutes(15);
    private final ExportSamples exportSamplesInterface;
    private final DatasourceConnectionServiceV2 connectionService;
    private final ExportConnectionConfigV1 config;
    private final int putSamplesPageSize;
    private final Runnable callback;

    ConcurrentHashMap<String, ExportJob> runningJobs;
    ConcurrentLinkedQueue<ExportJob> completedJobs;

    Thread mainLoopThread;
    ManualResetEvent stopMainLoop = new ManualResetEvent(false);

    /**
     * Instantiates a new ExportOrchestrator. Does not initialize it -- call Initialize() after construction.
     *
     * @param name
     *         The name of the exporter, usually just Datasource Name
     * @param exportSamplesInterface
     *         The ExportSamples interface as implemented by the connection
     * @param connectionService
     *         The DatasourceConnectionServiceV2 for the connection
     * @param config
     *         The ExportConnectionConfigV1 configuration for the export activity
     * @param putSamplesPageSize
     *         The number of samples to supply to the datasource per write call
     * @param callback
     *         Function for use in unit tests to avoid waiting too long before doing checks
     * @see ExportSamples
     * @see DatasourceConnectionServiceV2
     * @see ExportConnectionConfigV1
     */
    public ExportOrchestrator(String name, ExportSamples exportSamplesInterface,
            DatasourceConnectionServiceV2 connectionService,
            ExportConnectionConfigV1 config, int putSamplesPageSize, Runnable callback) {
        this.name = name;
        this.exportSamplesInterface = exportSamplesInterface;
        this.connectionService = connectionService;
        this.config = config;
        this.putSamplesPageSize = putSamplesPageSize;
        this.callback = callback;

        this.runningJobs = new ConcurrentHashMap<>();
        this.completedJobs = new ConcurrentLinkedQueue<>();
    }

    @Getter
    public String name;

    /**
     * Kicks off the main exporting loop that performs the exporting.
     */
    public void initialize() {
        if (this.mainLoopThread != null) {
            throw new IllegalStateException("ExportOrchestrator has already been initialized");
        }

        this.stopMainLoop.reset();

        this.mainLoopThread = new Thread(this::mainLoop);
        this.mainLoopThread.setName(String.format("Export Orchestrator for '%s'", this.name));
        this.mainLoopThread.start();

        LOG.debug("Export Orchestrator initialized");
    }

    /**
     * Cancels all jobs and shuts down the main exporting loop.
     */
    public void destroy() {
        this.stopMainLoop.set();

        while (true) {
            try {
                this.mainLoopThread.join(Duration.ofSeconds(5).toMillis());

                if (!this.mainLoopThread.isAlive()) {
                    break;
                }
            } catch (InterruptedException e) {
                break;
            }
        }

        LOG.debug("Export Orchestrator for '{}' waiting for main loop to shut down...",
                this.name);

        this.mainLoopThread = null;

        for (ExportJob job : this.runningJobs.values()) {
            job.getFuture().cancel(true);
            try {
                job.getFuture().get();
            } catch (Exception e) {
                // Do nothing, this is expected
            }
        }

        LOG.debug("Export Orchestrator destroyed");
    }

    void mainLoop() {
        ItemsApi itemsApi = this.connectionService.getAgentService().getApiProvider().createItemsApi();

        Stopwatch metricsLoggingStopwatch = new Stopwatch();
        Stopwatch directiveRefreshStopwatch = new Stopwatch();

        Map<String, ExportDirective> directives = null;
        Map<String, ExportStatus> statuses = null;

        this.completedJobs = new ConcurrentLinkedQueue<>();
        metricsLoggingStopwatch.start();

        int timeoutWithBackoff = 500;
        try {
            while (!this.stopMainLoop.waitOne(timeoutWithBackoff)) {
                try {
                    if (metricsLoggingStopwatch.elapsed(TimeUnit.MILLISECONDS) > Duration.ofMinutes(5).toMillis()) {
                        String jobStatisticsMessage = this.dequeueCompletedJobs();
                        LOG.debug(jobStatisticsMessage);
                        metricsLoggingStopwatch.restart();
                    }

                    if (this.connectionService.getConnectionState() != Connection.ConnectionState.CONNECTED) {
                        continue;
                    }

                    if (!this.connectionService.getAgentService().isSeeqServerConnected()) {
                        continue;
                    }

                    if (directives == null || statuses == null ||
                            directiveRefreshStopwatch.elapsed(TimeUnit.MILLISECONDS) >
                                    this.config.getDirectiveRefreshFrequencyAsDuration().toMillis()) {
                        RefreshDirectives retVal = this.refreshDirectives(itemsApi);
                        directives = retVal.getDirectives();
                        statuses = retVal.getStatuses();
                        this.logIfVerboseRefreshedDirectives(retVal);
                        directiveRefreshStopwatch.restart();
                    }

                    for (ExportDirective directive : directives.values()) {
                        ExportStatus status = statuses.get(directive.getItemName());
                        this.executeDirective(directive, status);
                    }
                    timeoutWithBackoff = 500;
                } catch (Exception e) {
                    timeoutWithBackoff = Math.min(timeoutWithBackoff * 2, 30000);
                    LOG.error("Error encountered in Export Orchestrator main loop:", e);
                } finally {
                    try {
                        if (this.callback != null) {
                            this.callback.run();
                        }
                    } catch (Exception e) {
                        // The backoff keeps this from being too noisy, and if we don't
                        // swallow it, the agent gets restarted repeatedly with no explanation.
                        LOG.error("Failed to invoke export main loop callback, retrying...");
                    }
                }
            }
        } catch (InterruptedException e) {
            LOG.error("Export Orchestrator mainLoop interrupted");
        }
    }

    private void logIfVerboseRefreshedDirectives(RefreshDirectives r) {
        if (r.getDirectives().entrySet().stream().anyMatch(d -> d.getValue().isVerbose())) {
            String directives =
                    r.getDirectives().entrySet().stream().sorted(Map.Entry.comparingByKey())
                            .map(d -> String.format(
                                    "- %s:%n" +
                                            "    Directive: %s%n" +
                                            "    Status: %s - %s",
                                    d.getKey(),
                                    d.getValue(),
                                    r.getStatuses().get(d.getKey()).Status,
                                    r.getStatuses().get(d.getKey()).Message
                            )).collect(Collectors.joining("%n"));
            LOG.debug("Refreshed directives (frequency: {}):%n{}",
                    this.config.getDirectiveRefreshFrequency(),
                    directives);
        }
    }

    private Duration getStandardLatency() {
        Duration minimumLatency = this.config.getMinimumLatencyAsDuration();
        return minimumLatency.toMillis() < MinimumStandardLatency.toMillis() ? MinimumStandardLatency : minimumLatency;
    }

    void executeDirective(ExportDirective directive, ExportStatus status) throws Exception {
        Function<String, Void> logIfVerbose = message -> {
            if (directive.isVerbose()) {
                LOG.debug(message);
            }
            return null;
        };

        logIfVerbose.apply(String.format("Executing directive %s", directive));

        ZonedDateTime lastWriteTime = status.Last_Write_Time != null ? status.Last_Write_Time :
                ZonedDateTime.parse("1980-01-01T00:00:00Z");
        Duration timeSinceLastWrite =
                Duration.between(lastWriteTime.withZoneSameInstant(ZoneId.of("UTC")),
                        ZonedDateTime.now(ZoneId.of("UTC")));

        Duration latency = directive.getLatencyOrDefault(this.getStandardLatency());
        if (timeSinceLastWrite.toMillis() < latency.toMillis()) {
            logIfVerbose.apply(String.format("Last write occurred too recently - a time span of %s has passed, " +
                    "which is less than the latency: %s", timeSinceLastWrite, latency));
            return;
        }

        TimeInstant cursor = status.Cursor;

        TimeInstant startTime = directive.getBackfillDateOrDefault();
        if (!directive.isClean() && cursor != null && cursor.getTimestamp() > startTime.getTimestamp()) {
            // We increment the export cursor timestamp to get the start time
            // so we don't rewrite the last sample every read/write cycle.
            startTime = cursor.increment();
        }
        TimeInstant endTime = this.exportSamplesInterface.getLatestAllowedWriteTime(directive.getItemName());
        if (endTime == null) {
            endTime = new TimeInstant(ZonedDateTime.now(ZoneId.of("UTC")));
        }

        if (endTime.getTimestamp() < startTime.getTimestamp()) {
            // This can happen for data that is forecasted into the future
            logIfVerbose.apply(String.format("Aborting execution because end time %s is before start time %s",
                    endTime, startTime));
            return;
        }

        ExportJob job = new ExportJob(directive, status, startTime, endTime);
        if (this.runningJobs.put(directive.getItemName(), job) != null) {
            // The item is already in the list of running jobs, which means it didn't finish in the timeframe
            // of the previous cycle
            logIfVerbose.apply(String.format("Job for item %s already running, no new job will be added to queue",
                    directive.getItemName()));
            return;
        } else {
            logIfVerbose.apply(String.format("Job for directive %s added to runningJobs.  " +
                    "There are %s other jobs currently running or queued", directive, this.runningJobs.size()));
        }

        try {
            job.setTask(new ExportTask(() -> {
                try {
                    this.doJob(job, new SeeqApiRequestHelper<>());
                } catch (OperationCanceledException e) {
                    // Do nothing
                } finally {
                    this.runningJobs.remove(directive.getItemName());
                }
            }));

            ExportTaskScheduler taskScheduler = this.connectionService.getAgentService().getExportTaskScheduler();

            double expectedDurationInSeconds = status.calculateExpectedDuration(startTime, endTime);
            if (expectedDurationInSeconds == 0.000d) {
                logIfVerbose.apply(String.format("Toll in status was 0, adding job for directive %s to normal queue",
                        directive));
                taskScheduler.queueTask(job.getTask(), ExportTaskScheduler.Queue.Normal);
            } else {
                if (expectedDurationInSeconds < 1) {
                    logIfVerbose.apply(String.format("Adding job for directive %s to fast queue", directive));
                    job.setFuture(taskScheduler.queueTask(job.getTask(), ExportTaskScheduler.Queue.Fast));
                } else if (expectedDurationInSeconds > 60) {
                    logIfVerbose.apply(String.format("Adding job for directive %s to slow queue", directive));
                    job.setFuture(taskScheduler.queueTask(job.getTask(), ExportTaskScheduler.Queue.Slow));
                } else {
                    logIfVerbose.apply(String.format("Adding job for directive %s to normal queue", directive));
                    job.setFuture(taskScheduler.queueTask(job.getTask(), ExportTaskScheduler.Queue.Normal));
                }
            }
        } catch (Exception e) {
            this.runningJobs.remove(directive.getItemName());
            throw new Exception(String.format("Exception while queuing %s", directive.getItemName()), e);
        }
    }

    @Data
    @AllArgsConstructor
    static class RefreshDirectives {
        private Map<String, ExportDirective> directives;
        private Map<String, ExportStatus> statuses;
    }

    @SuppressFBWarnings("WMI_WRONG_MAP_ITERATOR")
    RefreshDirectives refreshDirectives(ItemsApi itemsApi) {
        Duration minimumLatency = this.config.getMinimumLatencyAsDuration();
        Iterable<ExportDirective> newDirectives = ExportDirectives.read(this.getName(), itemsApi, minimumLatency);

        HashMap<String, ExportDirective> directives = new HashMap<>();
        HashMap<String, ExportStatus> statuses = new HashMap<>();

        HashMap<String, ArrayList<ExportDirective>> conflicts = new HashMap<>();

        HashSet<String> allNames = new HashSet<>();
        for (ExportDirective directive : newDirectives) {
            allNames.add(directive.getItemName());

            ExportStatus status = ExportStatus.read(itemsApi, directive.getSeeqItemID());

            if (directives.containsKey(directive.getItemName())) {
                if (!conflicts.containsKey(directive.getItemName())) {
                    conflicts.put(directive.getItemName(), new ArrayList<>());
                }
                conflicts.get(directive.getItemName()).add(directive);
                writeConflictingDirectivesError(itemsApi, directive, status);
                continue;
            }

            directives.put(directive.getItemName(), directive);
            statuses.put(directive.getItemName(), status);
        }

        if (this.config instanceof ApprovalExportConnectionConfigV1) {
            ((ApprovalExportConnectionConfigV1) this.config).removeNewOrChangedFromConfigIfNotPresent(allNames);
        }

        for (String signalName : conflicts.keySet()) {
            // We never write to a tag that has conflicting directives because it otherwise will
            // have "indeterminate" data being written to it that would be confusing/undesirable
            writeConflictingDirectivesError(itemsApi, directives.get(signalName), statuses.get(signalName));

            ArrayList<ExportDirective> conflictingDirectives = conflicts.get(signalName);
            conflictingDirectives.add(directives.get(signalName));
            String conflictingDirectivesString = conflictingDirectives.stream().map(
                    d -> ExportDirective.getExtendedDescriptor(d.getSeeqItemID(), this.connectionService)).collect(
                    Collectors.joining("\n"));
            LOG.error("Duplicate/conflicting export directives for {}:\n{}", signalName, conflictingDirectivesString);

            directives.remove(signalName);
            statuses.remove(signalName);
        }

        return new RefreshDirectives(directives, statuses);
    }

    /**
     * Removes all completed jobs from the queue and returns a message to log
     */
    String dequeueCompletedJobs() {
        Duration maximumTardiness = Duration.ZERO;
        ExportJob maximumTardinessJob = null;
        Duration writeDurationTotal = Duration.ZERO;
        int samplesWrittenTotal = 0;
        int jobsCompleted = 0;

        ExportJob job = this.completedJobs.poll();
        while (job != null) {
            if (job.getTardiness() != null && job.getTardiness().compareTo(maximumTardiness) > 0) {
                maximumTardiness = job.getTardiness();
                maximumTardinessJob = job;
            }

            jobsCompleted += 1;
            samplesWrittenTotal += job.getWriteCount();
            writeDurationTotal = writeDurationTotal.plus(job.getWriteDuration());
            job = this.completedJobs.poll();
        }

        int writeSpeed =
                writeDurationTotal.toMillis() > 0 ?
                        (int) (samplesWrittenTotal / (writeDurationTotal.toMillis() * ExportStatus.S_PER_MS)) : 0;

        String message = String.format("Export statistics: %s jobs completed", jobsCompleted);

        if (jobsCompleted > 0) {
            message +=
                    String.format(", %s samples written (avg speed: %s per second)", samplesWrittenTotal, writeSpeed);
        }

        if (maximumTardinessJob != null) {
            message += String.format(
                    ", maximum tardiness: %.1f sec (tag: \"%s\")", maximumTardiness.toMillis() * ExportStatus.S_PER_MS,
                    maximumTardinessJob.getDirective().getItemName());
        }

        return message;
    }

    private static void writeConflictingDirectivesError(ItemsApi itemsApi, ExportDirective directive,
            ExportStatus status) {
        status.Status = ExportStatus.Failed;
        status.Message = "ERROR: Multiple items trying to write to the same signal. " +
                String.format(
                        "Check net-link logs for \"Duplicate/conflicting export directives for %s:\" for more " +
                                "information.", directive.getItemName());
        status.write(itemsApi, directive.getSeeqItemID());
    }

    void doJob(ExportJob job, SeeqApiRequestHelper<FormulaRunOutputV1> requestHelper) {
        ZonedDateTime utcNow = ZonedDateTime.now(ZoneId.of("UTC"));
        if (job.getStatus().Last_Write_Time != null) {
            ZonedDateTime lastWrite = job.getStatus().Last_Write_Time.withZoneSameInstant(ZoneId.of("UTC"));
            Duration latency = job.getDirective().getLatencyOrDefault(this.config.getMinimumLatencyAsDuration());
            job.setTardiness(Duration.between(lastWrite, utcNow).minus(latency));
        }

        job.getStatus().Last_Write_Time = utcNow;

        ItemsApi itemsApi = this.connectionService.getAgentService().getApiProvider().createItemsApi();

        try {
            // Note that the status Object will be modified by readThenWrite()
            this.readThenWrite(job, job.getStatus(), job.getStartTime(), job.getEndTime(), requestHelper);
            job.getStatus().Status = ExportStatus.Success;
        } catch (ApiException e) {
            job.getStatus().Status = ExportStatus.Failed;
            job.getStatus().Message = SeeqSDKHelper.formatApiException(e);
        } catch (RuntimeException e) {
            job.getStatus().Status = ExportStatus.Failed;
            job.getStatus().Message = e.getMessage();
        } catch (Exception e) {
            job.getStatus().Status = ExportStatus.Failed;
            job.getStatus().Message = e.toString();
        }

        job.throwIfCancellationRequested();

        try {
            job.getStatus().write(itemsApi, job.getDirective().getSeeqItemID());
        } catch (Exception e) {
            LOG.error("ERROR: Could not write export status to item {}", job.getDirective().getSeeqItemID(), e);
        }
    }

    @Data
    @AllArgsConstructor
    private static class QueryTimeRange {
        private TimeInstant start;
        private TimeInstant end;
    }

    private void readThenWrite(ExportJob job, ExportStatus status, TimeInstant startTime, TimeInstant endTime,
            SeeqApiRequestHelper<FormulaRunOutputV1> requestHelper) {
        // This is necessary due to its use in a lambda (Java only)
        final QueryTimeRange queryTimeRange = new QueryTimeRange(startTime, endTime);

        ItemsApi itemsApi = this.connectionService.getAgentService().getApiProvider().createItemsApi();
        FormulasApi formulasApi = this.connectionService.getAgentService().getApiProvider().createFormulasApi();

        ExportDirective directive = job.getDirective();

        Function<String, Void> logIfVerbose = message -> {
            if (directive.isVerbose()) {
                LOG.debug(message);
            }
            return null;
        };

        Stopwatch writeStopwatch = new Stopwatch();

        while (true) {
            // We check for cancellation after any code that involves I/O (and therefore could be slow)
            job.throwIfCancellationRequested();

            logIfVerbose.apply(String.format(
                    "Running formula request for directive %s using Seeq ID %s over time range %s - %s",
                    directive, directive.getSeeqItemID(), startTime, endTime
            ));

            requestHelper.setRequest(() ->
                    formulasApi.runFormulaWithHttpInfo(
                            String.format("%d ns", queryTimeRange.getStart().getTimestamp()),
                            String.format("%d ns", queryTimeRange.getEnd().getTimestamp()),
                            "$item",
                            null,
                            Collections.singletonList(String.format("item=%s", directive.getSeeqItemID())),
                            null,
                            null,
                            null,
                            0,
                            this.putSamplesPageSize,
                            null)
            );
            SeeqSdkApiResponse<FormulaRunOutputV1> response = requestHelper.makeRequest();

            logIfVerbose.apply(String.format(
                    "Formula request for directive %s returned %s samples",
                    directive, response.getData().getSamples().getSamples().size()
            ));

            job.throwIfCancellationRequested();

            RequestTimings requestTimings = RequestTimings.fromApiResponseHeaders(response.getHeaders());

            status.calculateAndSetToll(requestTimings, startTime, endTime);

            // Filter out uncertain samples first.
            // Remove boundary values because they may be in the future and PI can't always handle that
            // https://livelibrary.osisoft.com/LiveLibrary/content/en/server-v14/GUID-64F01EE4-8C23-4B0B-919D-287D54D59141
            List<Sample> samplesToWrite = response.getData().getSamples().getSamples().stream()
                    .filter(ss -> !(ss.getIsUncertain() != null ? ss.getIsUncertain() : false))
                    .filter(ss -> ((long) ss.getKey()) >= queryTimeRange.getStart().getTimestamp() &&
                            ((long) ss.getKey()) <= queryTimeRange.getEnd().getTimestamp())
                    .map(ss -> new Sample(new TimeInstant((long) ss.getKey()), ss.getValue()))
                    .collect(Collectors.toList());

            logIfVerbose.apply(String.format("After filtering, %s samples will be put", samplesToWrite.size()));

            writeStopwatch.restart();
            this.exportSamplesInterface.putSamples(new PutSamplesParameters(
                    directive.getItemName(),
                    directive.getSeeqItemID(),
                    response.getData().getMetadata().get(SeeqNames.Properties.ValueUom),
                    response.getData().getMetadata().get(SeeqNames.Properties.InterpolationMethod).equals("Step"),
                    startTime, endTime,
                    samplesToWrite,
                    directive.isClean(),
                    directive.isVerbose())
            );
            writeStopwatch.stop();

            if (job.getDirective().isClean()) {
                job.getDirective().setClean(false);
                ExportDirectives.write(job.getDirective(), itemsApi);
            }

            job.setWriteCount(job.getWriteCount() + samplesToWrite.size());
            job.setWriteDuration(job.getWriteDuration().plusMillis(writeStopwatch.elapsed(TimeUnit.MILLISECONDS)));

            if (samplesToWrite.size() > 0) {
                // This must be set here before possibly breaking out of the loop
                status.Cursor = samplesToWrite.get(samplesToWrite.size() - 1).getKey();
            }

            if (response.getData().getSamples().getSamples().size() < this.putSamplesPageSize ||
                    samplesToWrite.size() == 0) {
                logIfVerbose.apply(String.format(
                        "Sample count for directive %s was less than page size (%s) or count " +
                                "of samples to be put was zero; exiting loop",
                        directive, this.putSamplesPageSize
                ));
                break;
            }

            queryTimeRange.setStart(samplesToWrite.get(samplesToWrite.size() - 1).getKey());
        }

        if (job.getWriteCount() > 0) {
            ArrayList<String> metricStrings = new ArrayList<>();
            int writeSpeed =
                    job.getWriteDuration().toMillis() > 0 ?
                            (int) (job.getWriteCount() / (job.getWriteDuration().toMillis() * ExportStatus.S_PER_MS)) :
                            0;

            metricStrings.add(String.format("%s samples", job.getWriteCount()));
            if (writeSpeed > 0) {
                metricStrings.add(String.format("at %s per second", writeSpeed));
            }

            if (job.getTardiness() != null) {
                metricStrings.add(
                        String.format("tardy by %.1f seconds", job.getTardiness().toMillis() * ExportStatus.S_PER_MS));
            }

            String metricsFullString = String.join(", ", metricStrings);
            status.Message = String.format("Wrote %s", metricsFullString);
        } else {
            status.Message = "No samples written in last cycle.";
        }

        logIfVerbose.apply(String.format("Adding completed job to completedJobs queue for directive %s", directive));
        this.completedJobs.add(job);
    }
}
