package com.seeq.link.sdk.export;

import java.time.Duration;
import java.util.concurrent.Future;

import com.seeq.link.sdk.utilities.TimeInstant;
import com.seeq.utilities.exception.OperationCanceledException;

import lombok.Data;

/**
 * An internal data structure that captures the state around a particular
 * export job. There is a one-to-one mapping between jobs and signals to
 * be exported. A job lives until the data is written to the datasource,
 * then a new job is spawned according to the latency on the directive.
 */
@Data
public class ExportJob {

    /**
     * Creates an export job for the supplied directive.
     *
     * @param directive
     *         The export directive for this job
     */
    public ExportJob(ExportDirective directive) {
        this.directive = directive;
    }

    public ExportJob(ExportDirective directive, ExportStatus status, TimeInstant startTime, TimeInstant endTime) {
        this.directive = directive;
        this.status = status;
        this.startTime = startTime;
        this.endTime = endTime;
        this.future = null;
    }

    void throwIfCancellationRequested() {
        if (this.future != null && this.future.isCancelled()) {
            throw new OperationCanceledException(String.format("Job for %s canceled", this.directive));
        }
    }

    /**
     * The start time for the job task
     */
    private TimeInstant startTime;

    /**
     * The end time for the job task
     */
    private TimeInstant endTime;

    /**
     * The status of the job
     */
    private ExportStatus status;

    /**
     * The export directive for this job.
     */
    private final ExportDirective directive;

    /**
     * The asynchronous Task associated with doing this job.
     */
    private ExportTask task;

    /**
     * Used by ExportOrchestrator when it is destroyed and there are outstanding jobs. (Java only)
     */
    private Future<?> future;

    /**
     * The tardiness of a job, which is the amount of time between when a
     * job should have run (according to the export directive latency) and
     * when it actually did.
     */
    private Duration tardiness = null;

    /**
     * The number of samples/capsules written to the datasource by the job.
     */
    private int writeCount = 0;

    /**
     * The timen taken to write the samples/capsules (see WriteCount) to
     * the datasource. Does not include the time taken to read the data
     * from Seeq.
     */
    private Duration writeDuration = Duration.ZERO;
}
