package com.seeq.link.sdk.export;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.NoSuchElementException;

import org.jetbrains.annotations.NotNull;

import com.seeq.api.ItemsApi;
import com.seeq.model.ItemSearchPreviewPaginatedListV1;
import com.seeq.model.ItemSearchPreviewV1;
import com.seeq.model.PropertyInputV1;
import com.seeq.model.ScalarPropertyV1;
import com.seeq.utilities.SeeqNames;

/**
 * Provides a Read function to enumerate all export directives on items in Seeq, and
 * a Write function to overwrite the export directive for a particular item.
 */
public final class ExportDirectives {

    private static final ArrayList<String> SEARCH_FILTER = new ArrayList<>(
            Collections.singletonList(String.format("%s==/.*\\[EXPORT.*/", SeeqNames.Properties.Description)));

    private static final ArrayList<String> ALLOWED_TYPES =
            new ArrayList<>(Arrays.asList(SeeqNames.Types.StoredSignal, SeeqNames.Types.CalculatedSignal));

    private ExportDirectives() {}

    /**
     * Reads all export directives for all items. The "Everyone" group must be given
     * read access to the items in order for them to be discovered, since by default
     * they will not be accessible to the Agent API Key user.
     *
     * @param connectionName
     *         The name of the connection for which to filter the export directives
     * @param itemsApi
     *         The ItemsApi to use to retrieve the directives.
     * @param minimumLatency
     *         Minimum latency from export configuration
     * @return An Iterator of ExportDirectives. Note that this is a lazy iterator,
     *         and the ItemsApi won't be called until the first item is accessed.
     */
    public static Iterable<ExportDirective> read(
            String connectionName,
            ItemsApi itemsApi,
            Duration minimumLatency
    ) {
        class ExportDirectiveReadIterator implements Iterator<ExportDirective> {
            final String connectionName;
            final ItemsApi itemsApi;
            final Duration minimumLatency;
            ExportDirective nextDirective = null;

            static final int limit = 1000;
            int offset = 0;
            ItemSearchPreviewPaginatedListV1 itemSearch = null;
            int itemSearchIndex;

            public ExportDirectiveReadIterator(String connectionName, ItemsApi itemsApi, Duration minimumLatency) {
                this.connectionName = connectionName;
                this.itemsApi = itemsApi;
                this.minimumLatency = minimumLatency;
            }

            @Override
            public boolean hasNext() {
                if (this.nextDirective != null) {
                    return true;
                }

                while (true) {
                    if (this.itemSearch == null) {
                        this.itemSearch = this.itemsApi.searchItems(
                                SEARCH_FILTER, null, null, ALLOWED_TYPES,
                                this.offset, limit,
                                new ArrayList<>(Collections.singletonList(SeeqNames.Properties.Id)));

                        this.itemSearchIndex = 0;
                    }

                    if (this.itemSearch != null && this.itemSearchIndex >= this.itemSearch.getItems().size()) {
                        if (this.itemSearch.getItems().size() < limit) {
                            return false;
                        }

                        this.itemSearch = null;
                        this.offset += limit;
                    }

                    if (this.itemSearch == null) {
                        continue;
                    }

                    ItemSearchPreviewV1 itemPreview = this.itemSearch.getItems().get(this.itemSearchIndex);

                    try {
                        ExportDirective directive;
                        try {
                            directive = ExportDirective.parse(itemPreview.getDescription(), this.minimumLatency);
                        } catch (Exception e) {
                            ScalarPropertyV1 statusProperty = new ScalarPropertyV1();
                            statusProperty.setName("Export - Status");
                            statusProperty.setValue(ExportStatus.Failed);
                            ScalarPropertyV1 messageProperty = new ScalarPropertyV1();
                            messageProperty.setName("Export - Message");
                            messageProperty.setValue(e.getMessage());
                            this.itemsApi.setProperties(itemPreview.getId(), new ArrayList<>(Arrays.asList(
                                    statusProperty, messageProperty)));
                            continue;
                        }

                        if (!this.connectionName.equals(directive.getConnectionName())) {
                            continue;
                        }

                        directive.setSeeqItemID(itemPreview.getId());
                        directive.setSeeqItemType(itemPreview.getType());

                        this.nextDirective = directive;
                    } finally {
                        this.itemSearchIndex++;
                    }

                    return true;
                }
            }

            @Override
            public ExportDirective next() {
                if (this.nextDirective == null && !this.hasNext()) {
                    throw new NoSuchElementException("No more export directives");
                }

                ExportDirective returnedDirective = this.nextDirective;
                this.nextDirective = null;
                return returnedDirective;
            }
        }

        class ExportDirectiveReadIterable implements Iterable<ExportDirective> {
            final String connectionName;
            final ItemsApi itemsApi;
            final Duration minimumLatency;

            public ExportDirectiveReadIterable(String connectionName, ItemsApi itemsApi, Duration minimumLatency) {
                this.connectionName = connectionName;
                this.itemsApi = itemsApi;
                this.minimumLatency = minimumLatency;
            }

            @NotNull
            @Override
            public Iterator<ExportDirective> iterator() {
                return new ExportDirectiveReadIterator(this.connectionName, this.itemsApi, this.minimumLatency);
            }
        }

        return new ExportDirectiveReadIterable(connectionName, itemsApi, minimumLatency);
    }

    public static void write(ExportDirective directive, ItemsApi itemsApi) {
        String oldDescription =
                itemsApi.getProperty(directive.getSeeqItemID(), SeeqNames.Properties.Description).getValue();
        String newDescription =
                oldDescription != null
                        ? oldDescription.replaceAll("\\[EXPORT.*]\\s*$", directive.toString())
                        : directive.toString();
        PropertyInputV1 descriptionProperty = new PropertyInputV1();
        descriptionProperty.setValue(newDescription);
        itemsApi.setProperty(directive.getSeeqItemID(), SeeqNames.Properties.Description, descriptionProperty);
    }
}
