package com.seeq.link.sdk.export;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.time.Duration;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.format.DateTimeParseException;
import java.time.format.ResolverStyle;
import java.time.temporal.ChronoField;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.commons.lang3.StringUtils;

import com.google.code.regexp.Matcher;
import com.google.code.regexp.Pattern;
import com.seeq.api.ItemsApi;
import com.seeq.link.sdk.interfaces.DatasourceConnectionServiceV2;
import com.seeq.link.sdk.utilities.TimeInstant;
import com.seeq.link.sdk.utilities.TimeInterval;
import com.seeq.model.ItemOutputV1;

import lombok.AllArgsConstructor;
import lombok.Data;

/**
 * Specifies the parameters for the exporting an item, being able to serialize to/de-serialize from
 * a human readable String of the form [EXPORT TO "connection" AS "name"]
 */
@Data
public class ExportDirective {
    private String seeqItemID;
    private String seeqItemType;
    private String connectionName;
    private String itemName;
    private Duration latency;
    private TimeInstant backfillDate;
    private boolean clean;
    private boolean verbose;

    private enum TokenType {
        To,
        As,
        Every,
        BackfillDate,
        Clean,
        Verbose
    }

    @Data
    @AllArgsConstructor
    private static class TokenDefinition {
        private TokenType tokenType;
        private String syntax;
    }

    private static TokenDefinition token(TokenType t, String r) {
        return new TokenDefinition(t, r);
    }

    private static ArrayList<TokenDefinition> TokenDefinitions = new ArrayList<>(Arrays.asList(
            token(TokenType.To, "TO \"<ConnectionName>\""),

            token(TokenType.As, "AS \"<ItemName>\""),

            token(TokenType.Every, "EVERY \"<Latency>\""),

            token(TokenType.BackfillDate, "BACKFILL TO \"<BackfillDate>\""),

            token(TokenType.Clean, "CLEAN"),

            token(TokenType.Verbose, "VERBOSE")
    ));

    private static String getPossibleDirectives() {
        return TokenDefinitions.stream().map(TokenDefinition::getSyntax)
                .collect(Collectors.joining(String.format("%n")));
    }

    /**
     * Returns the latency, which is how often the item is exported. It is specified in the
     * EVERY "5min" part of the directive.
     *
     * @param defaultLatency
     *         A Duration that provides the latency to use if not configured
     *         in the ExportDirective
     * @return A Duration representing the specified latency, or the provided default
     */
    public Duration getLatencyOrDefault(Duration defaultLatency) {
        return this.latency != null ? this.latency : defaultLatency;
    }

    /**
     * Returns the backfill date, which is the date to which data is written when the CLEAN
     * clause is supplied or data is being written for the first time.
     *
     * @return A TimeInstant representing the backfill date, or the default of "5 hours before now".
     */
    public TimeInstant getBackfillDateOrDefault() {
        return this.backfillDate != null ? this.backfillDate :
                new TimeInstant(ZonedDateTime.now(ZoneId.of("UTC")).minusHours(5));
    }

    /**
     * Returns a String describing the syntax of the export directive.
     */
    public static String getSyntax() {
        return String.format("[EXPORT %s]", getPossibleDirectives());
    }

    /**
     * Parses a (serialized) String representation of the export directive and returns a
     * populated ExportDirective object.
     *
     * @param input
     *         An export directive in String form. See Syntax property for format.
     * @param minimumLatency
     *         Minimum latency from export configuration
     * @return A populated Export Directive.
     * @throws IllegalArgumentException
     *         When the String could not be parsed as an export directive.
     */
    public static ExportDirective parse(String input, Duration minimumLatency) {
        Pattern pattern = Pattern.compile("^.*\\[\\s*EXPORT (.*)\\]\\s*$", Pattern.MULTILINE);
        Matcher wholeDirectiveMatch = pattern.matcher(input);

        if (!wholeDirectiveMatch.matches()) {
            throw new IllegalArgumentException(
                    String.format("Input '%s' not recognized as a valid export directive%n" +
                            "Export syntax:%n%s", input, getSyntax()));
        }

        ExportDirective exportDirective = new ExportDirective();
        Method[] methods = exportDirective.getClass().getMethods();
        String remaining = wholeDirectiveMatch.group(1);

        while (remaining.trim().length() > 0) {
            boolean recognized = false;
            for (TokenDefinition tokenDefinition : TokenDefinitions) {
                String regexString = "^" + tokenDefinition.getSyntax();
                regexString = regexString.replace("\"<", "\"(?<");
                regexString = regexString.replace(">\"", ">[^\"]+)\"");
                regexString = regexString.replace(" ", "\\s+");
                Pattern regex = Pattern.compile(regexString);
                Matcher match = regex.matcher(remaining);
                if (!match.find()) {
                    continue;
                }

                if (tokenDefinition.getTokenType() == TokenType.Clean) {
                    exportDirective.setClean(true);
                }

                if (tokenDefinition.getTokenType() == TokenType.Verbose) {
                    exportDirective.setVerbose(true);
                }

                try {
                    for (String groupName : regex.groupNames()) {
                        if (groupName.equals("0")) {
                            continue;
                        }
                        String group = match.group(groupName);
                        String methodName = "set" + groupName;
                        Method method =
                                Stream.of(methods).filter(m -> m.getName().equals(methodName)).findFirst().orElseThrow(
                                        () -> new NoSuchMethodException("No method on ExportDirective found that " +
                                                "matches syntax: " + methodName));
                        if (method.getParameterTypes()[0] == String.class) {
                            method.invoke(exportDirective, group);
                        } else if (method.getParameterTypes()[0] == TimeInstant.class) {
                            ZonedDateTime dateTime;
                            try {
                                try {
                                    dateTime = ZonedDateTime.parse(group);
                                } catch (DateTimeParseException e) {
                                    dateTime = ZonedDateTime.parse(group, new DateTimeFormatterBuilder()
                                            .append(DateTimeFormatter.ISO_DATE)
                                            .parseDefaulting(ChronoField.HOUR_OF_DAY, 0)
                                            .parseDefaulting(ChronoField.HOUR_OF_AMPM, 0)
                                            .parseDefaulting(ChronoField.MINUTE_OF_HOUR, 0)
                                            .parseDefaulting(ChronoField.SECOND_OF_MINUTE, 0)
                                            .toFormatter()
                                            .withZone(ZoneId.of("UTC"))
                                            .withResolverStyle(ResolverStyle.LENIENT));
                                }
                            } catch (Exception e) {
                                throw new Exception(String.format(
                                        "Error parsing date/time \"%s\". Format should be \"2020-04-21T14:34:52Z\"",
                                        group));
                            }

                            method.invoke(exportDirective, new TimeInstant(dateTime));
                        } else if (method.getParameterTypes()[0] == Duration.class) {
                            method.invoke(exportDirective, TimeInterval.parseFriendlyDuration(group));
                        }
                    }
                } catch (Exception e) {
                    throw new IllegalArgumentException(
                            String.format("Error parsing directive fragment '%s': %s",
                                    match, e.getMessage()), e);
                }

                recognized = true;

                remaining = match.end() < remaining.length() - 1 ? remaining.substring(match.end() + 1) : "";
                break;
            }

            if (!recognized) {
                throw new IllegalArgumentException(
                        String.format("Unrecognized export directive fragment:%n%s%n%nExport syntax:%n%s", remaining,
                                getSyntax()));
            }
        }

        ArrayList<String> problems = new ArrayList<>();

        if (StringUtils.isBlank(exportDirective.getConnectionName())) {
            problems.add("Export directive must specify valid ConnectionName");
        }

        if (StringUtils.isBlank(exportDirective.getItemName())) {
            problems.add("Export directive must specify valid ItemName");
        }

        if (exportDirective.getLatency() != null && exportDirective.getLatency().compareTo(minimumLatency) < 0) {
            problems.add(String.format("Export directive Latency %s " +
                            "is less than the MinimumLatency %s specified by your Seeq administrator",
                    TimeInterval.toFriendlyStringMatchDotNetTimeSpan(exportDirective.getLatency()),
                    TimeInterval.toFriendlyStringMatchDotNetTimeSpan(minimumLatency)));
        }

        if (problems.size() > 0) {
            throw new IllegalArgumentException(
                    String.format("Encountered the following issues:%n%s%nExport syntax:%n%s",
                            String.join("\n", problems), getSyntax())
            );
        }

        return exportDirective;
    }

    /**
     * Returns a helpful "descriptor" that can be optionally written to the target datasource
     * that refers to the name and workbook within Seeq of the exported item. This is useful
     * for admins to be able to track down an item and potentially rename it or adjust its
     * directive in other ways.
     *
     * @param itemID
     *         The Seeq ID of the item
     * @param connectionService
     *         The IDatasourceConnectionServiceV2 for the connection
     * @return Descriptor to write to target datasource
     */
    public static String getExtendedDescriptor(String itemID, DatasourceConnectionServiceV2 connectionService) {
        ItemsApi itemsApi = connectionService.getAgentService().getApiProvider().createItemsApi();

        ItemOutputV1 itemOutput = itemsApi.getItemAndAllProperties(itemID);

        // Remove quotes here because it gets really confusing to have them escaped in the JSON and
        // then manually transferred to a PI Point with the escapes in place. Yes, that means that
        // the names won't perfectly match when the person is looking for the point in Seeq, but
        // chances are they'll find out. This is the "least problematic" of solutions for this rare
        // edge case.
        String cleansedItemName = itemOutput.getName().replace("\"", "");

        if (itemOutput.getScopedTo() == null) {
            // Note that we use apostrophes around the name here instead of quotes because we don't want
            // anything to be escaped in the JSON, it would make it harder to copy/paste to PI System Management Tools
            return String.format("Named '%s' at global scope on %s", cleansedItemName,
                    connectionService.getAgentService().getSeeqServerURL());
        }

        return String.format("Named '%s' within %s/workbook/%s", cleansedItemName,
                connectionService.getAgentService().getSeeqServerURL(), itemOutput.getScopedTo());
    }

    /**
     * Serializes the export directive to its String representation.
     *
     * @return Export directive as a String
     */
    @Override
    public String toString() {
        StringBuilder str = new StringBuilder("[EXPORT");
        for (TokenDefinition definition : TokenDefinitions) {
            String token = definition.getSyntax();

            if (definition.getTokenType() == TokenType.Clean && this.isClean()) {
                str.append(" ").append(token);
                continue;
            }

            if (definition.getTokenType() == TokenType.Verbose && this.isVerbose()) {
                str.append(" ").append(token);
                continue;
            }

            Matcher match = Pattern.compile("<([^>]+)>").matcher(token);
            while (match.find()) {
                try {
                    Object valObj = this.getClass().getMethod("get" + match.group(1)).invoke(this);
                    if (valObj != null) {
                        String valString;
                        if (valObj instanceof Duration) {
                            valString = TimeInterval.toFriendlyString((Duration) valObj);
                        } else {
                            valString = valObj.toString();
                        }
                        str.append(" ").append(token.replace(match.group(0), valString));
                    }
                } catch (NoSuchMethodException | InvocationTargetException | IllegalAccessException e) {
                    // Keep going
                }
            }
        }

        str.append("]");
        return str.toString();
    }
}
