package com.seeq.link.sdk.export;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import lombok.AccessLevel;
import lombok.Data;
import lombok.Getter;
import lombok.Setter;

@Data
public class ApprovalExportConnectionConfigV1 extends ExportConnectionConfigV1 {
    private boolean autoCreate;
    private boolean autoUpdate;
    private boolean requireApproval = true;

    @Getter(AccessLevel.NONE)
    @Setter(AccessLevel.NONE)
    private ArrayList<ExportDefinitionV1> newOrChanged = new ArrayList<>();

    @Getter(AccessLevel.NONE)
    @Setter(AccessLevel.NONE)
    private Runnable notifyAddedOrChangedCallback = null;

    public ArrayList<ExportDefinitionV1> getNewOrChanged() {
        synchronized (this) {
            return this.newOrChanged;
        }
    }

    void setNewOrChanged(ArrayList<ExportDefinitionV1> value) {
        synchronized (this) {
            this.newOrChanged = value;
        }
    }

    public void setNotifyAddedOrChangedCallback(Runnable callback) {
        this.notifyAddedOrChangedCallback = callback;
    }

    private ExportDefinitionV1 getExistingDefinition(String name) {
        return this.newOrChanged.stream().filter(d -> d.getName().equals(name)).findFirst().orElse(null);
    }

    /**
     * Will return true if AutoCreate is true and RequireApproval is false, or if change is approved (independent of
     * AutoCreate setting).
     *
     * @param name
     *         Name of the exported signal
     * @return true if creation is approved
     */
    public boolean isCreationApproved(String name) {
        if (this.autoCreate && !this.requireApproval) {
            return true;
        }

        return this.isChangeApproved(name);
    }

    /**
     * Will return true if AutoUpdate is true and RequireApproval is false, or if change is approved (independent of
     * AutoUpdate setting).
     *
     * @param name
     *         Name of the exported signal
     * @return true if update is approved
     */
    public boolean isUpdateApproved(String name) {
        if (this.autoUpdate && !this.requireApproval) {
            return true;
        }

        return this.isChangeApproved(name);
    }

    private boolean isChangeApproved(String name) {
        synchronized (this) {
            ExportDefinitionV1 existingDefinition = this.getExistingDefinition(name);
            if (existingDefinition == null) {
                return false;
            }

            return existingDefinition.isApproved();
        }
    }

    /**
     * Adds a signal to the NewOrChanged section of the configuration, if it doesn't already exist.
     *
     * @param name
     *         Name of the exported signal
     * @param properties
     *         The properties for the exported signal
     * @return true if there were any actual changes to the config, false if everything remained identical
     */
    public boolean addNewOrChangedToConfig(String name, HashMap<String, Object> properties) {
        boolean added = false;
        boolean changed = false;
        synchronized (this) {
            HashMap<String, Object> nullSafeProperties = properties != null ? properties : new HashMap<>();
            ExportDefinitionV1 existingDefinition = this.getExistingDefinition(name);
            if (existingDefinition == null) {
                existingDefinition = new ExportDefinitionV1(name, false, nullSafeProperties);
                added = true;
            } else if (existingDefinition.getProperties().size() != nullSafeProperties.size()) {
                changed = true;
            } else {
                for (Map.Entry<String, Object> property : nullSafeProperties.entrySet()) {
                    if (!existingDefinition.getProperties().containsKey(property.getKey()) ||
                            !existingDefinition.getProperties().get(property.getKey()).toString().equals(
                                    property.getValue().toString())) {
                        changed = true;
                        break;
                    }
                }
            }

            if (added) {
                this.newOrChanged.add(existingDefinition);
            } else if (changed) {
                existingDefinition.setProperties(properties);
            }
        }

        if ((added || changed) && this.notifyAddedOrChangedCallback != null) {
            this.notifyAddedOrChangedCallback.run();
        }

        return (added || changed);
    }

    /**
     * Removes a signal from the NewOrChanged section of the configuration, if it exists.
     *
     * @param name
     *         Name of the exported signal
     * @return true if the signal existed and was removed, otherwise false
     */
    public boolean removeNewOrChangedFromConfig(String name) {
        synchronized (this) {
            ExportDefinitionV1 existingDefinition = this.getExistingDefinition(name);
            if (existingDefinition == null) {
                return false;
            }

            this.newOrChanged.remove(existingDefinition);
        }

        if (this.notifyAddedOrChangedCallback != null) {
            this.notifyAddedOrChangedCallback.run();
        }

        return true;
    }

    /**
     * Removes any signals present in the config if they are not part of the supplied set.
     *
     * @param names
     *         A set of names to keep
     * @return true if any signal existed and was removed, otherwise false
     */
    public boolean removeNewOrChangedFromConfigIfNotPresent(Set<String> names) {
        boolean atLeastOneRemoved = false;
        synchronized (this) {
            // We can't modify the list in place, so make a copy of it first
            ArrayList<ExportDefinitionV1> copyOfNewOrChanged = new ArrayList<>(this.newOrChanged);

            for (ExportDefinitionV1 definition : copyOfNewOrChanged) {
                if (names.contains(definition.getName())) {
                    continue;
                }

                this.newOrChanged.remove(definition);

                atLeastOneRemoved = true;
            }
        }

        if (atLeastOneRemoved && this.notifyAddedOrChangedCallback != null) {
            this.notifyAddedOrChangedCallback.run();
        }

        return atLeastOneRemoved;
    }
}
