package com.seeq.link.sdk;

import java.time.ZonedDateTime;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.seeq.link.sdk.interfaces.SyncMode;

import lombok.Getter;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;

/**
 * Captures the intended schedule for a connection's indexing activity. getNext() specifies the next scheduled
 * indexing time, with timezone information. getFrequency() indicates how frequent indexing should occur. This class
 * honors timezone peculiarities, so that a frequency of 1d is ensured to be at the same time of day regardless of
 * daylight savings time and whatnot. isOnStartupAndConfigChange() indicates whether indexing will occur when the
 * agent starts and when the connection's configuration changes.
 *
 * All scheduling captured by this class is for {@link SyncMode#FULL} indexing. Connections are responsible for
 * triggering {@link SyncMode#INCREMENTAL} indexing activity according to their own requirements. (The Stress Test
 * Connector is an example of how a connection might do it.) Connections will call requestNow(SyncMode) when
 * they want to trigger an index themselves.
 */
@Slf4j
public class IndexingSchedule {

    @Getter
    @Setter
    private String frequency;

    @Getter
    @Setter
    private boolean onStartupAndConfigChange;

    private boolean onStartupAndConfigChangeComplete;

    // This variable can be modified by the connector and by the agent on different threads
    private volatile SyncMode indexingRequested;

    public IndexingSchedule() {
        this.next = ZonedDateTime.now();
        this.frequency = "1w";
        this.onStartupAndConfigChange = true;
        this.onStartupAndConfigChangeComplete = false;
        this.indexingRequested = SyncMode.NONE;
    }

    public IndexingSchedule(ZonedDateTime next, String frequency, boolean onStartupAndConfigChange) {
        this();

        this.next = next;
        this.frequency = frequency;
        this.onStartupAndConfigChange = onStartupAndConfigChange;
    }

    private ZonedDateTime next;

    /**
     * Returns the time of the next scheduled indexing activity as a fully-qualified ISO 8601 string with time zone
     * info.
     *
     * @return the time of the next scheduled indexing activity
     */
    public String getNext() {
        return this.next.toString();
    }

    /**
     * Called by the JSON deserializer when reading in the schedule from disk.
     *
     * @param next
     *         the time of the next scheduled indexing activity
     */
    public void setNext(String next) {
        if (next == null || next.trim().length() == 0) {
            this.next = ZonedDateTime.now();
            return;
        }

        try {
            this.next = ZonedDateTime.parse(next);
        } catch (Exception e) {
            // If we can't parse it, then just set it to now
            LOG.error("Could not parse index schedule next '{}', setting to now", next);
            this.next = ZonedDateTime.now();
        }
    }

    /**
     * Returns the date/time of the next scheduled index. Only used by tests.
     *
     * @return the date/time of the next scheduled index
     */
    @JsonIgnore
    ZonedDateTime getNextDateTime() {
        return this.next;
    }

    /**
     * Called by the agent to determine if this connection should index. An index may be triggered due to
     * scheduling, startup/config change, or as a result of a request from the connection.
     *
     * @param connectionName
     *         The name of this connection. Used solely for logging purposes.
     * @return The {@link SyncMode} desired. This will always be {@link SyncMode#FULL} unless the connection has
     *         requested an {@link SyncMode#INCREMENTAL}.
     */
    public SyncMode shouldIndex(String connectionName) {
        if (this.onStartupAndConfigChange && !this.onStartupAndConfigChangeComplete) {
            LOG.debug("Connection {} should index now because IndexingSchedule.OnStartupAndConfigChange is true",
                    connectionName);
            return SyncMode.FULL;
        }

        if (this.indexingRequested != SyncMode.NONE) {
            LOG.debug("Connection {} should index now because the connector requested {} index", connectionName,
                    this.indexingRequested.toString());
        }

        return this.indexingRequested;
    }

    /**
     * Called by a connection to request an index right now. This is the method by which an
     * {@link SyncMode#INCREMENTAL} index can happen.
     *
     * @param syncMode
     *         The indexing mode requested
     */
    public void requestNow(SyncMode syncMode) {
        this.indexingRequested = syncMode;
    }

    /**
     * Called by the agent when indexing has finished.
     */
    public void markIndexingFinished() {
        this.onStartupAndConfigChangeComplete = true;
        this.indexingRequested = SyncMode.NONE;
    }
}
