package com.seeq.link.sdk;

import static com.seeq.link.messages.connector.auth.AuthConnectionMessages.AuthResponseMessage;
import static com.seeq.link.messages.connector.extcalc.ExternalCalculationMessages.ExternalCalculationResponseMessage;
import static org.apache.commons.lang3.StringUtils.isBlank;

import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.locks.Lock;
import java.util.function.BiFunction;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.seeq.ApiException;
import com.seeq.api.AssetsApi;
import com.seeq.api.ConditionsApi;
import com.seeq.api.ItemsApi;
import com.seeq.api.ScalarsApi;
import com.seeq.api.SignalsApi;
import com.seeq.api.SystemApi;
import com.seeq.api.TreesApi;
import com.seeq.api.UserGroupsApi;
import com.seeq.link.messages.agent.AgentMessages.DatasourceService;
import com.seeq.link.messages.connector.auth.AuthConnectionMessages.AuthRequestMessage;
import com.seeq.link.messages.connector.condition.ConditionConnectionMessages.ConditionRequestMessage;
import com.seeq.link.messages.connector.condition.ConditionConnectionMessages.ConditionResponseMessage;
import com.seeq.link.messages.connector.extcalc.ExternalCalculationMessages;
import com.seeq.link.messages.connector.extcalc.ExternalCalculationMessages.ExternalCalculationRequestMessage;
import com.seeq.link.messages.connector.extcalc.ExternalCalculationMessages.SignalDataType;
import com.seeq.link.messages.connector.oauth2.OAuth2ConnectionMessages;
import com.seeq.link.messages.connector.signal.SignalConnectionMessages.DataStatus;
import com.seeq.link.messages.connector.signal.SignalConnectionMessages.SignalRequestMessage;
import com.seeq.link.messages.connector.signal.SignalConnectionMessages.SignalResponseMessage;
import com.seeq.link.sdk.export.ExportConnectionConfigV1;
import com.seeq.link.sdk.export.ExportOrchestrator;
import com.seeq.link.sdk.export.ExportSamples;
import com.seeq.link.sdk.interfaces.AddOnCalcSignalDataType;
import com.seeq.link.sdk.interfaces.AddOnCalculationDatasourceConnection;
import com.seeq.link.sdk.interfaces.AddOnCalculationValidator;
import com.seeq.link.sdk.interfaces.AgentService;
import com.seeq.link.sdk.interfaces.AuthDatasourceConnection;
import com.seeq.link.sdk.interfaces.AuthParameters;
import com.seeq.link.sdk.interfaces.AuthResult;
import com.seeq.link.sdk.interfaces.ConditionPullDatasourceConnection;
import com.seeq.link.sdk.interfaces.DatasourceConnection;
import com.seeq.link.sdk.interfaces.DatasourceConnectionServiceV2;
import com.seeq.link.sdk.interfaces.DatasourceConnectionV2;
import com.seeq.link.sdk.interfaces.GetCapsulesParameters;
import com.seeq.link.sdk.interfaces.GetSamplesParameters;
import com.seeq.link.sdk.interfaces.GroupInfo;
import com.seeq.link.sdk.interfaces.IndexingDatasourceConnection;
import com.seeq.link.sdk.interfaces.NonSchedulableIndexingConnection;
import com.seeq.link.sdk.interfaces.OAuth2AuthParameters;
import com.seeq.link.sdk.interfaces.OAuth2AuthResult;
import com.seeq.link.sdk.interfaces.OAuth2DatasourceConnection;
import com.seeq.link.sdk.interfaces.OAuth2PreAuthParameters;
import com.seeq.link.sdk.interfaces.OAuth2PreAuthResult;
import com.seeq.link.sdk.interfaces.PullDatasourceConnection;
import com.seeq.link.sdk.interfaces.SignalPullDatasourceConnection;
import com.seeq.link.sdk.interfaces.SyncMode;
import com.seeq.link.sdk.interfaces.SyncStatus;
import com.seeq.link.sdk.services.PropertyTransformer;
import com.seeq.link.sdk.utilities.ApiModelHelper;
import com.seeq.link.sdk.utilities.BatchSizeHelper;
import com.seeq.link.sdk.utilities.Capsule;
import com.seeq.link.sdk.utilities.ExceptionHelper;
import com.seeq.link.sdk.utilities.RequestCancellation;
import com.seeq.link.sdk.utilities.Sample;
import com.seeq.link.sdk.utilities.TimeInstant;
import com.seeq.model.AssetBatchInputV1;
import com.seeq.model.AssetInputV1;
import com.seeq.model.AssetTreeBatchInputV1;
import com.seeq.model.AssetTreeSingleInputV1;
import com.seeq.model.ConditionBatchInputV1;
import com.seeq.model.ConditionUpdateInputV1;
import com.seeq.model.DatasourceOutputV1;
import com.seeq.model.ItemIdListInputV1;
import com.seeq.model.ItemUpdateOutputV1;
import com.seeq.model.LicenseStatusOutputV1;
import com.seeq.model.LicensedFeatureStatusOutputV1;
import com.seeq.model.PutAssetInputV1;
import com.seeq.model.PutScalarInputV1;
import com.seeq.model.PutScalarsInputV1;
import com.seeq.model.PutSignalsInputV1;
import com.seeq.model.PutUserGroupsInputV1;
import com.seeq.model.ScalarInputV1;
import com.seeq.model.ScalarPropertyV1;
import com.seeq.model.SignalWithIdInputV1;
import com.seeq.model.UserGroupWithIdInputV1;
import com.seeq.utilities.SeeqNames;
import com.seeq.utilities.exception.OperationCanceledException;

import lombok.Data;
import lombok.extern.slf4j.Slf4j;

/**
 * Hosts an {@link DatasourceConnectionV2} connection and acts as a bridge to the older {@link DatasourceConnection}
 * interface via the {@link DatasourceConnectionServiceV2} interface.
 */
@Slf4j
public class DatasourceConnectionV2Host extends BaseDatasourceConnection<ConnectorV2Host>
        implements DatasourceConnectionServiceV2 {
    private final Logger guestLogger;
    private final DatasourceConnectionV2 connection;
    private DatasourceOutputV1 datasource;
    private List<PropertyTransformer.Spec> transforms;

    private static final String EXCLUDE_FROM_INDEXING = "Exclude From Indexing";

    // Datasource item cleanup filter parameters
    private List<String> datasourceItemTypeCleanupFilter = null;
    private String datasourceItemDataIdRegexFilter = null;
    private String datasourceItemDataIdExcludeRegexFilter = null;
    private String datasourceItemNameRegexFilter = null;
    private String datasourceItemNameExcludeRegexFilter = null;

    private ExportOrchestrator exportOrchestrator = null;

    DatasourceConnectionV2Host(AgentService agentService, ConnectorV2Host connectorHost,
            DatasourceConnectionV2 connection, DatasourceService[] services) {
        super(agentService, connectorHost, connection, services);
        this.connection = connection;

        // We have to prefix with com.seeq so that all of our logging configuration is applied to the guest
        this.guestLogger = LoggerFactory.getLogger("com.seeq.link.plugin." +
                this.connection.getClass().getSimpleName());

        this.transforms = null;

        this.initSignals();
        this.initConditions();
        this.initScalars();
        this.initAssets();
        this.initRootAssets();
        this.initRelationships();
        this.initUserGroups();

        this.signalBatchSizeHelper = this.getAgentService().createBatchSizeHelper();
        this.conditionBatchSizeHelper = this.getAgentService().createBatchSizeHelper();
        this.scalarBatchSizeHelper = this.getAgentService().createBatchSizeHelper();
        this.assetBatchSizeHelper = this.getAgentService().createBatchSizeHelper();
        this.rootAssetBatchSizeHelper = this.getAgentService().createBatchSizeHelper();
        this.relationshipBatchSizeHelper = this.getAgentService().createBatchSizeHelper();
        this.userGroupBatchSizeHelper = this.getAgentService().createBatchSizeHelper();
    }

    @Override
    public void initialize() {
        this.connection.initialize(this);
    }

    @Override
    public boolean isLicensed(String featureName) {
        if (!this.getAgentService().isSeeqServerConnected()) {
            this.log().info("Waiting for Seeq Server connection to verify license");
            return false;
        }

        SystemApi systemApi = this.getAgentService().getApiProvider().createSystemApi();

        LicenseStatusOutputV1 licenseStatusOutput = null;
        try {
            licenseStatusOutput = systemApi.getLicense();
        } catch (ApiException apex) {
            this.log().error("Failed to get license from Seeq Server due to exception:", apex);
            return false;
        }

        if (licenseStatusOutput.getAdditionalFeatures().stream()
                .noneMatch(o -> o.getName().equals(featureName))) {
            this.log().error("Seeq Server license does not include the feature {}", featureName);
            return false;
        }

        Optional<LicensedFeatureStatusOutputV1> licensedFeatureStatusOutputV1 =
                licenseStatusOutput.getAdditionalFeatures().stream()
                        .filter(o -> o.getName().equals(featureName))
                        .findFirst();
        if (licensedFeatureStatusOutputV1.isPresent() &&
                licensedFeatureStatusOutputV1
                        .get().getValidity() != LicensedFeatureStatusOutputV1.ValidityEnum.VALID
        ) {
            this.log().error("License for feature {} is not valid", featureName);
            return false;
        }

        this.log().info("License validated for feature {}", featureName);
        return true;
    }

    @Override
    public void destroy() {
        super.destroy();
        this.connection.destroy();

        if (this.exportOrchestrator != null) {
            this.exportOrchestrator.destroy();
        }
    }

    @Override
    public void setConnectionState(ConnectionState newState) {
        if (newState == ConnectionState.CONNECTED) {
            // When the new state is CONNECTED we want to clear the message
            this.setState(newState, "");
        } else {
            // Otherwise we keep the the old message
            this.setState(newState, this.getConnectionMessage());
        }
    }

    @Override
    public void setConnectionStatusMessage(String message) {
        this.setState(this.getState(), message);
    }

    @Override
    protected void handleConnectionMonitorException(String methodName, Exception exception) {
        super.handleConnectionMonitorException(methodName, exception);
        this.setState(this.getState(), ExceptionHelper.toExceptionMessage(exception));
    }

    @Override
    public ConnectionState getConnectionState() {
        return this.getState();
    }

    @Override
    public DatasourceOutputV1 getDatasource() {
        return this.datasource;
    }

    private boolean skipArchiving = false;

    @Override
    public void skipArchiving() {
        this.skipArchiving = true;
    }

    @Override
    public void storeDatasourceProperties(boolean storedInSeeq, List<ScalarPropertyV1> additionalProperties) {
        this.updateExistingDatasource(this.getDatasource(), storedInSeeq, additionalProperties);
    }

    @Override
    public void deleteDatasourceProperties(List<String> additionalPropertiesToDelete) {
        Map<String, ScalarPropertyV1> existingAdditionalPropertiesByName =
                this.getDatasource().getAdditionalProperties().stream()
                        .collect(Collectors.toMap(ScalarPropertyV1::getName, p -> p));

        ItemsApi itemsApi = this.getAgentService().getIndexingApiProvider().createItemsApi();

        for (String propertyToRemove : additionalPropertiesToDelete) {
            if (existingAdditionalPropertiesByName.containsKey(propertyToRemove)) {
                itemsApi.deleteProperty(this.getDatasource().getId(), propertyToRemove);
            } else {
                LOG.debug("The property {} is not one of the additionalProperties of the datasource. Skipping delete.",
                        propertyToRemove);
            }
        }

        this.datasource = this.getDatasourceAndCreationInfo().datasource;
    }


    /**
     * Returns a DatasourceAndCreationInfo object for the current datasource.
     *
     * @return A DatasourceAndCreationInfo object for the current datasource.
     */
    private DatasourceAndCreationInfo getDatasourceAndCreationInfo() {
        boolean storedInSeeq = !this.isPullDatasourceConnection();

        return this.getOrCreateDatasource(storedInSeeq);
    }

    @Override
    public boolean isPullDatasourceConnection() {
        return this.connection instanceof PullDatasourceConnection;
    }

    @Override
    public boolean isIndexingDatasourceConnection() {
        return this.connection instanceof IndexingDatasourceConnection;
    }

    @Override
    public boolean isIndexingScheduleSupported() {
        if (this.connection instanceof IndexingDatasourceConnection) {
            return !(this.connection instanceof NonSchedulableIndexingConnection);
        } else {
            return false;
        }
    }

    /**
     * If a local copy hasn't been cached, gets or creates this connection's datasource.
     *
     * @return True if the datasource was newly created, false if it already existed.
     */
    private boolean ensureDatasourceExists() {
        if (this.datasource != null) {
            return false;
        }

        DatasourceAndCreationInfo datasourceAndCreationInfo = this.getDatasourceAndCreationInfo();
        this.datasource = datasourceAndCreationInfo.datasource;
        return datasourceAndCreationInfo.newlyCreated;
    }

    @Override
    public void metadataSync(SyncMode syncMode) throws Exception {
        IndexingDatasourceConnection indexingConnection = (IndexingDatasourceConnection) this.connection;

        boolean newlyCreated = this.ensureDatasourceExists();

        this.datasource = this.getDatasourceAndCreationInfo().datasource;

        String requestId = "Cleaning up datasource " + this.datasource.getId();
        this.cancelRequest(requestId);

        this.itemsWithErrors = 0;
        this.initSignals();
        this.initConditions();
        this.initScalars();
        this.initAssets();
        this.initRootAssets();
        this.initRelationships();
        this.initUserGroups();
        this.signalBatchSizeHelper = this.getAgentService().createBatchSizeHelper();
        this.conditionBatchSizeHelper = this.getAgentService().createBatchSizeHelper();
        this.scalarBatchSizeHelper = this.getAgentService().createBatchSizeHelper();
        this.assetBatchSizeHelper = this.getAgentService().createBatchSizeHelper();
        this.rootAssetBatchSizeHelper = this.getAgentService().createBatchSizeHelper();
        this.relationshipBatchSizeHelper = this.getAgentService().createBatchSizeHelper();
        this.userGroupBatchSizeHelper = this.getAgentService().createBatchSizeHelper();
        this.syncToken = ZonedDateTime.now().format(DateTimeFormatter.ISO_INSTANT);

        this.setSyncStatus(SyncStatus.SYNC_IN_PROGRESS);
        this.sendSyncToken(this.datasource.getId(), this.syncToken, syncMode.name());

        // Ensure we can see if the connection requests that we skip archiving
        this.skipArchiving = false;

        this.setTransforms(indexingConnection.getConfiguration().getTransforms());

        indexingConnection.index(syncMode);

        // This will flush everything one last time
        this.flushRelationships();

        if (syncMode == SyncMode.FULL && !newlyCreated && !this.skipArchiving) {
            // Archive any stale items that do not have the current sync token
            this.cleanUpStaleItems(this.datasource.getId(), this.syncToken, this.datasourceItemTypeCleanupFilter,
                    this.datasourceItemDataIdRegexFilter, this.datasourceItemDataIdExcludeRegexFilter,
                    this.datasourceItemNameRegexFilter, this.datasourceItemNameExcludeRegexFilter,
                    requestId);
        } else {
            if (syncMode != SyncMode.FULL) {
                LOG.debug("Skipping datasource archiving because SyncMode is not FULL");
            } else if (newlyCreated) {
                LOG.debug("Skipping datasource archiving because datasource was newly created");
            } else {
                LOG.debug("Skipping datasource archiving because skipArchiving() was called by connection");
            }
        }

        // Set the sync status to SYNC_COMPLETE
        this.setSyncStatus(SyncStatus.SYNC_COMPLETE);

        this.syncToken = null;
    }

    @Data
    private static class RelationshipKey {
        private final String childId;
        private final String parentId;
    }

    private BatchSizeHelper signalBatchSizeHelper;
    private BatchSizeHelper conditionBatchSizeHelper;
    private BatchSizeHelper scalarBatchSizeHelper;
    private BatchSizeHelper assetBatchSizeHelper;
    private BatchSizeHelper rootAssetBatchSizeHelper;
    private BatchSizeHelper relationshipBatchSizeHelper;
    private BatchSizeHelper userGroupBatchSizeHelper;

    // We store the accumulation of inputs for a batch in a map because connectors will often call a PutXxxxx API
    // multiple times with the same item. They do that because it can be a pain to accurately keep track of what they
    // have already sent, especially if they are constructing an asset tree. A map alleviates them from having to
    // worry about it and eliminates duplicate items (within a batch) sent to appserver.
    private Map<String, SignalWithIdInputV1> signalBatchInput = null;
    private Map<String, ConditionUpdateInputV1> conditionBatchInput = null;
    private Map<String, PutScalarInputV1> scalarBatchInput = null;
    private Map<String, PutAssetInputV1> assetBatchInput = null;
    private Map<String, PutAssetInputV1> rootAssetBatchInput = null;
    private Map<RelationshipKey, AssetTreeSingleInputV1> relationshipBatchInput = null;
    private Map<String, UserGroupWithIdInputV1> userGroupBatchInput = null;

    private String syncToken = null;

    private static boolean shouldExcludeFromIndexing(List<ScalarPropertyV1> properties) {
        ScalarPropertyV1 excludeFromIndexingProperty = excludeFromIndexingProperty(properties);
        return excludeFromIndexingProperty != null &&
                excludeFromIndexingProperty.getValue() instanceof Boolean &&
                (boolean) excludeFromIndexingProperty.getValue();
    }

    private static ScalarPropertyV1 excludeFromIndexingProperty(List<ScalarPropertyV1> properties) {
        return properties.stream().filter(p -> p.getName().equals(EXCLUDE_FROM_INDEXING))
                .findFirst().orElse(null);
    }

    @VisibleForTesting
    int getItemsWithErrors() {
        return this.itemsWithErrors;
    }

    @Override
    public void setTransforms(List<PropertyTransformer.Spec> transforms) {
        this.transforms = transforms;
    }

    private static void checkNameAndDataId(String name, String dataId) {
        Preconditions.checkArgument(!isBlank(dataId), "DataId cannot be null. It must be a unique"
                + " identifier for the item within the data source.");
        Preconditions.checkArgument(!isBlank(name), "Name cannot be null. It is the text that will "
                + "be used as the primary way to reference this item in Seeq.");
    }

    @Override
    public List<ItemUpdateOutputV1> putSignal(SignalWithIdInputV1 signalDefinition) throws ApiException {
        checkNameAndDataId(signalDefinition.getName(), signalDefinition.getDataId());

        signalDefinition.setDatasourceId(this.getDatasourceId());
        signalDefinition.setDatasourceClass(this.getDatasourceClass());

        if (this.transforms != null) {
            signalDefinition = PropertyTransformer.transform(signalDefinition, this.transforms);

            if (shouldExcludeFromIndexing(signalDefinition.getAdditionalProperties())) {
                return null;
            } else {
                // This Signal has something other than true for the value of its Exclude From Indexing property;
                // remove the property
                signalDefinition.getAdditionalProperties()
                        .remove(excludeFromIndexingProperty(signalDefinition.getAdditionalProperties()));
            }
        }

        signalDefinition.setSyncToken(this.syncToken);

        List<ItemUpdateOutputV1> itemsFlushed = new ArrayList<>();

        this.signalBatchInput.put(signalDefinition.getDataId(), signalDefinition);

        if (this.signalBatchInput.size() >= this.signalBatchSizeHelper.getBatchSize()) {
            itemsFlushed = this.flushSignals();
        }

        return itemsFlushed;
    }

    private void initSignals() {
        this.signalBatchInput = new HashMap<>();
    }

    @Override
    public List<ItemUpdateOutputV1> flushSignals() throws ApiException {
        this.setSyncStatus(SyncStatus.SYNC_IN_PROGRESS);

        List<ItemUpdateOutputV1> itemsFlushed = new ArrayList<>();

        try {
            if (this.signalBatchInput.size() != 0) {
                PutSignalsInputV1 batchInput = new PutSignalsInputV1();
                batchInput.setSignals(ImmutableList.copyOf(this.signalBatchInput.values()));

                SignalsApi signalsApi = this.getAgentService().getIndexingApiProvider().createSignalsApi();

                this.signalBatchSizeHelper.start();
                itemsFlushed = signalsApi.putSignals(batchInput).getItemUpdates();
                this.signalBatchSizeHelper.stop(this.signalBatchInput.size());
                this.logAndCountBadItems(itemsFlushed);

                StringBuilder logStr = new StringBuilder(
                        String.format("Sync Batch Stats (Signals): %d items took %.2f seconds",
                                this.signalBatchInput.size(),
                                this.signalBatchSizeHelper.getLastDuration().toMillis() / 1000d));
                double ips = this.signalBatchSizeHelper.getLastItemsPerSecond();
                if (Double.isFinite(ips)) {
                    logStr.append(String.format(" at %.1f per second", ips));
                }

                LOG.debug(logStr.toString());
            }
        } finally {
            this.initSignals();
        }
        return itemsFlushed;
    }

    private void initUserGroups() {
        this.userGroupBatchInput = new HashMap<>();
    }

    @Override
    public List<ItemUpdateOutputV1> putUserGroup(UserGroupWithIdInputV1 userGroupDefinition) throws ApiException {
        checkNameAndDataId(userGroupDefinition.getName(), userGroupDefinition.getDataId());

        // We expect connectors to index UserGroups that don’t match their datasourceId/class because we decided that
        // some connectors will have the possibility to create user group stubs with a
        // different datasource (e.g PI Connector creates LDAP/Win Auth user group stubs)
        // see https://seeq.atlassian.net/wiki/spaces/SQ/pages/511643741/CRAB-15096 (section PI AD Synchronization)
        if (userGroupDefinition.getDatasourceId() == null) {
            userGroupDefinition.setDatasourceId(this.getDatasourceId());
        }
        if (userGroupDefinition.getDatasourceClass() == null) {
            userGroupDefinition.setDatasourceClass(this.getDatasourceClass());
        }

        if (this.transforms != null) {
            userGroupDefinition = PropertyTransformer.transform(userGroupDefinition, this.transforms);
        }

        userGroupDefinition.setSyncToken(this.syncToken);

        List<ItemUpdateOutputV1> itemsFlushed = new ArrayList<>();

        this.userGroupBatchInput.put(userGroupDefinition.getDataId(), userGroupDefinition);

        if (this.userGroupBatchInput.size() >= this.userGroupBatchSizeHelper.getBatchSize()) {
            itemsFlushed = this.flushUserGroups();
        }

        return itemsFlushed;
    }

    @Override
    public List<ItemUpdateOutputV1> flushUserGroups() throws ApiException {
        this.setSyncStatus(SyncStatus.SYNC_IN_PROGRESS);

        List<ItemUpdateOutputV1> itemsFlushed = new ArrayList<>();

        try {
            if (this.userGroupBatchInput.size() != 0) {
                PutUserGroupsInputV1 batchInput = new PutUserGroupsInputV1();
                batchInput.setUserGroups(ImmutableList.copyOf(this.userGroupBatchInput.values()));

                UserGroupsApi userGroupsApi = this.getAgentService().getIndexingApiProvider().createUserGroupsApi();

                this.userGroupBatchSizeHelper.start();
                itemsFlushed = userGroupsApi.putUserGroups(batchInput).getItemUpdates();
                this.userGroupBatchSizeHelper.stop(this.userGroupBatchInput.size());
                this.logAndCountBadItems(itemsFlushed);

                StringBuilder logStr = new StringBuilder(
                        String.format("Sync Batch Stats (User Groups): %d items took %.2f seconds",
                                this.userGroupBatchInput.size(),
                                this.userGroupBatchSizeHelper.getLastDuration().toMillis() / 1000d));
                double ips = this.userGroupBatchSizeHelper.getLastItemsPerSecond();
                if (Double.isFinite(ips)) {
                    logStr.append(String.format(" at %.1f per second", ips));
                }

                LOG.debug(logStr.toString());
            }
        } finally {
            this.initUserGroups();
        }
        return itemsFlushed;
    }

    @Override
    public List<ItemUpdateOutputV1> putCondition(ConditionUpdateInputV1 conditionDefinition) throws ApiException {
        checkNameAndDataId(conditionDefinition.getName(), conditionDefinition.getDataId());

        List<ItemUpdateOutputV1> itemsFlushed = new ArrayList<>();

        conditionDefinition.setDatasourceId(this.getDatasourceId());
        conditionDefinition.setDatasourceClass(this.getDatasourceClass());

        if (this.transforms != null) {
            conditionDefinition = PropertyTransformer.transform(conditionDefinition, this.transforms);

            if (shouldExcludeFromIndexing(conditionDefinition.getProperties())) {
                return null;
            } else {
                // This Condition has something other than true for the value of its Exclude From Indexing property;
                // remove the property
                conditionDefinition.getProperties()
                        .remove(excludeFromIndexingProperty(conditionDefinition.getProperties()));
            }
        }

        conditionDefinition.setSyncToken(this.syncToken);

        this.conditionBatchInput.put(conditionDefinition.getDataId(), conditionDefinition);

        if (this.conditionBatchInput.size() >= this.conditionBatchSizeHelper.getBatchSize()) {
            itemsFlushed = this.flushConditions();
        }

        return itemsFlushed;
    }

    private void initConditions() {
        this.conditionBatchInput = new HashMap<>();
    }

    @Override
    public List<ItemUpdateOutputV1> flushConditions() throws ApiException {
        this.setSyncStatus(SyncStatus.SYNC_IN_PROGRESS);

        List<ItemUpdateOutputV1> itemsFlushed = new ArrayList<>();

        try {
            if (this.conditionBatchInput.size() != 0) {
                ConditionBatchInputV1 batchInput = new ConditionBatchInputV1();
                batchInput.setConditions(new ArrayList<ConditionUpdateInputV1>(this.conditionBatchInput.values()));

                ConditionsApi conditionsApi = this.getAgentService().getIndexingApiProvider().createConditionsApi();

                this.conditionBatchSizeHelper.start();
                itemsFlushed = conditionsApi.putConditions(batchInput).getItemUpdates();
                this.conditionBatchSizeHelper.stop(this.conditionBatchInput.size());
                this.logAndCountBadItems(itemsFlushed);

                StringBuilder logStr = new StringBuilder(
                        String.format("Sync Batch Stats (Conditions): %d items took %.2f seconds",
                                this.conditionBatchInput.size(),
                                this.conditionBatchSizeHelper.getLastDuration().toMillis() / 1000d));
                double ips = this.conditionBatchSizeHelper.getLastItemsPerSecond();
                if (Double.isFinite(ips)) {
                    logStr.append(String.format(" at %.1f per second", ips));
                }

                LOG.debug(logStr.toString());
            }
        } finally {
            this.initConditions();
        }

        return itemsFlushed;
    }

    @Override
    public List<ItemUpdateOutputV1> putScalar(PutScalarInputV1 scalarDefinition) throws ApiException {
        return this.putScalarInternal(scalarDefinition);
    }

    @Override
    public List<ItemUpdateOutputV1> putScalar(ScalarInputV1 scalarDefinition) throws ApiException {
        PutScalarInputV1 putScalarInput = new PutScalarInputV1();
        ApiModelHelper.copySchemaProperties(scalarDefinition, putScalarInput);
        return this.putScalarInternal(putScalarInput);
    }

    private List<ItemUpdateOutputV1> putScalarInternal(PutScalarInputV1 scalarDefinition) throws ApiException {
        checkNameAndDataId(scalarDefinition.getName(), scalarDefinition.getDataId());
        if (isBlank(scalarDefinition.getFormula())) {
            boolean stringOrNullUnits =
                    Optional.ofNullable(scalarDefinition.getUnitOfMeasure()).orElse("string")
                            .equalsIgnoreCase("string");
            String safeFormula = stringOrNullUnits ?
                    "\"" + Optional.ofNullable(scalarDefinition.getFormula()).orElse("") + "\"" :
                    "Scalar.Invalid";
            scalarDefinition.setFormula(safeFormula);
        }

        List<ItemUpdateOutputV1> itemsFlushed = new ArrayList<>();

        scalarDefinition.setDatasourceId(this.getDatasourceId());
        scalarDefinition.setDatasourceClass(this.getDatasourceClass());

        if (this.transforms != null) {
            scalarDefinition = PropertyTransformer.transform(scalarDefinition, this.transforms);

            if (shouldExcludeFromIndexing(scalarDefinition.getProperties())) {
                return null;
            } else {
                // This Scalar has something other than true for the value of its Exclude From Indexing property;
                // remove the property
                scalarDefinition.getProperties()
                        .remove(excludeFromIndexingProperty(scalarDefinition.getProperties()));
            }
        }

        scalarDefinition.setSyncToken(this.syncToken);

        this.scalarBatchInput.put(scalarDefinition.getDataId(), scalarDefinition);

        if (this.scalarBatchInput.size() >= this.scalarBatchSizeHelper.getBatchSize()) {
            itemsFlushed = this.flushScalars();
        }

        return itemsFlushed;
    }

    private void initScalars() {
        this.scalarBatchInput = new HashMap<>();
    }

    @Override
    public List<ItemUpdateOutputV1> flushScalars() throws ApiException {
        this.setSyncStatus(SyncStatus.SYNC_IN_PROGRESS);

        List<ItemUpdateOutputV1> itemsFlushed = new ArrayList<>();

        try {
            if (this.scalarBatchInput.size() != 0) {
                PutScalarsInputV1 batchInput = new PutScalarsInputV1();
                batchInput.setScalars(new ArrayList<>(this.scalarBatchInput.values()));
                batchInput.setDatasourceId(this.getDatasourceId());
                batchInput.setDatasourceClass(this.getDatasourceClass());

                ScalarsApi scalarsApi = this.getAgentService().getIndexingApiProvider().createScalarsApi();

                this.scalarBatchSizeHelper.start();
                itemsFlushed = scalarsApi.putScalars(batchInput).getItemUpdates();
                this.scalarBatchSizeHelper.stop(this.scalarBatchInput.size());
                this.logAndCountBadItems(itemsFlushed);

                StringBuilder logStr = new StringBuilder(
                        String.format("Sync Batch Stats (Scalars): %d items took %.2f seconds",
                                this.scalarBatchInput.size(),
                                this.scalarBatchSizeHelper.getLastDuration().toMillis() / 1000d));
                double ips = this.scalarBatchSizeHelper.getLastItemsPerSecond();
                if (Double.isFinite(ips)) {
                    logStr.append(String.format(" at %.1f per second", ips));
                }

                LOG.debug(logStr.toString());
            }
        } finally {
            this.initScalars();
        }

        return itemsFlushed;
    }

    private void initAssets() {
        this.assetBatchInput = new HashMap<>();
    }

    private void initRootAssets() {
        this.rootAssetBatchInput = new HashMap<>();
    }

    @Override
    public List<ItemUpdateOutputV1> putAsset(PutAssetInputV1 assetDefinition) throws ApiException {
        return this.putAssetInternal(assetDefinition, this.syncToken, this.assetBatchInput, this.assetBatchSizeHelper,
                this::flushAssets);
    }

    @Override
    public List<ItemUpdateOutputV1> putAsset(AssetInputV1 assetDefinition) throws ApiException {
        PutAssetInputV1 putAssetInput = new PutAssetInputV1();
        ApiModelHelper.copySchemaProperties(assetDefinition, putAssetInput);
        return this.putAssetInternal(putAssetInput, this.syncToken,
                this.assetBatchInput,
                this.assetBatchSizeHelper,
                this::flushAssets);
    }

    @Override
    public List<ItemUpdateOutputV1> putRootAsset(PutAssetInputV1 assetDefinition) throws ApiException {
        return this.putAssetInternal(assetDefinition, this.syncToken, this.rootAssetBatchInput,
                this.rootAssetBatchSizeHelper, this::flushRootAssets);
    }

    @Override
    public List<ItemUpdateOutputV1> putRootAsset(AssetInputV1 assetDefinition) throws ApiException {
        PutAssetInputV1 putAssetInput = new PutAssetInputV1();
        ApiModelHelper.copySchemaProperties(assetDefinition, putAssetInput);
        return this.putAssetInternal(putAssetInput, this.syncToken, this.rootAssetBatchInput,
                this.rootAssetBatchSizeHelper, this::flushRootAssets);
    }

    private List<ItemUpdateOutputV1> putAssetInternal(PutAssetInputV1 assetDefinition, String syncToken,
            Map<String, PutAssetInputV1> batchInput, BatchSizeHelper batchSizeHelper,
            CheckedSupplier<List<ItemUpdateOutputV1>> flushFunction) throws ApiException {
        checkNameAndDataId(assetDefinition.getName(), assetDefinition.getDataId());

        if (this.transforms != null) {
            assetDefinition = PropertyTransformer.transform(assetDefinition, this.transforms);

            if (shouldExcludeFromIndexing(assetDefinition.getProperties())) {
                return null;
            } else {
                // This Asset has something other than true for the value of its Exclude From Indexing property;
                // remove the property
                assetDefinition.getProperties()
                        .remove(excludeFromIndexingProperty(assetDefinition.getProperties()));
            }
        }

        assetDefinition.setSyncToken(syncToken);

        List<ItemUpdateOutputV1> itemsFlushed = new ArrayList<>();

        this.ensureDatasourceExists();
        assetDefinition.setHostId(this.datasource.getId());
        batchInput.put(assetDefinition.getDataId(), assetDefinition);

        if (batchInput.size() >= batchSizeHelper.getBatchSize()) {
            itemsFlushed = flushFunction.get();
        }

        return itemsFlushed;
    }

    @Override
    public List<ItemUpdateOutputV1> flushAssets() throws ApiException {
        List<ItemUpdateOutputV1> itemsFlushed;
        try {
            AssetBatchInputV1 batchInput = new AssetBatchInputV1();
            batchInput.setAssets(new ArrayList<>(this.assetBatchInput.values()));
            itemsFlushed = this.flushAssetsInternal(batchInput, this.assetBatchSizeHelper, r -> {});
        } finally {
            this.initAssets();
        }
        return itemsFlushed;
    }

    @Override
    public List<ItemUpdateOutputV1> flushRootAssets() throws ApiException {
        CheckedConsumer<List<ItemUpdateOutputV1>> makeItemsRootAssets = assetList -> {
            TreesApi treesApi = this.getAgentService().getIndexingApiProvider().createTreesApi();

            // Move nodes to root of asset tree so it will appear in Workbench
            ItemIdListInputV1 itemIdList = new ItemIdListInputV1();
            itemIdList.setItems(assetList.stream()
                    .filter(o -> o.getItem() != null)
                    .map(o -> o.getItem().getId())
                    .collect(Collectors.toList()));

            treesApi.moveNodesToRootOfTree(itemIdList);
        };

        List<ItemUpdateOutputV1> itemsFlushed;
        try {
            AssetBatchInputV1 batchInput = new AssetBatchInputV1();
            batchInput.setAssets(new ArrayList<>(this.rootAssetBatchInput.values()));
            itemsFlushed = this.flushAssetsInternal(batchInput, this.rootAssetBatchSizeHelper, makeItemsRootAssets);
        } finally {
            this.initRootAssets();
        }

        return itemsFlushed;
    }

    private List<ItemUpdateOutputV1> flushAssetsInternal(AssetBatchInputV1 batchInput, BatchSizeHelper batchSizeHelper,
            CheckedConsumer<List<ItemUpdateOutputV1>> responseConsumer) throws ApiException {
        this.setSyncStatus(SyncStatus.SYNC_IN_PROGRESS);

        List<ItemUpdateOutputV1> itemsFlushed = new ArrayList<>();

        if (batchInput.getAssets().size() != 0) {
            AssetsApi assetsApi = this.getAgentService().getIndexingApiProvider().createAssetsApi();

            batchInput.setHostId(batchInput.getAssets().get(0).getHostId());

            batchSizeHelper.start();
            itemsFlushed = assetsApi.batchCreateAssets(batchInput).getItemUpdates();

            batchSizeHelper.stop(batchInput.getAssets().size());
            responseConsumer.accept(itemsFlushed);
            this.logAndCountBadItems(itemsFlushed);

            StringBuilder logStr = new StringBuilder(
                    String.format("Sync Batch Stats (Assets): %d items took %.2f seconds",
                            batchInput.getAssets().size(),
                            batchSizeHelper.getLastDuration().toMillis() / 1000d));

            double ips = batchSizeHelper.getLastItemsPerSecond();
            if (Double.isFinite(ips)) {
                logStr.append(String.format(" at %.1f per second", ips));
            }
            LOG.debug(logStr.toString());
        }

        return itemsFlushed;
    }

    private void initRelationships() {
        this.relationshipBatchInput = new HashMap<>();
    }

    @Override
    public List<ItemUpdateOutputV1> putRelationship(AssetTreeSingleInputV1 relationshipDefinition) throws ApiException {
        Preconditions.checkArgument(relationshipDefinition.getChildDataId() != null,
                "Child data ID cannot be null.");
        Preconditions.checkArgument(relationshipDefinition.getParentDataId() != null,
                "Parent data ID cannot be null.");

        this.ensureDatasourceExists();
        this.relationshipBatchInput.put(new RelationshipKey(relationshipDefinition.getChildDataId(),
                relationshipDefinition.getParentDataId()), relationshipDefinition);

        List<ItemUpdateOutputV1> itemsFlushed = new ArrayList<>();

        if (this.relationshipBatchInput.size() >= this.relationshipBatchSizeHelper.getBatchSize()) {
            itemsFlushed = this.flushRelationships();
        }

        return itemsFlushed;
    }

    @Override
    public List<ItemUpdateOutputV1> flushRelationships() throws ApiException {
        // If we're flushing relationships, we have to make sure all the signals and assets have been flushed too
        this.flushUserGroups();
        this.flushSignals();
        this.flushConditions();
        this.flushAssets();
        this.flushScalars();
        this.flushRootAssets();

        List<ItemUpdateOutputV1> itemsFlushed = new ArrayList<>();

        try {
            if (this.relationshipBatchInput.size() != 0) {
                AssetTreeBatchInputV1 batchInput = new AssetTreeBatchInputV1();
                batchInput.setChildHostId(this.datasource.getId());
                batchInput.setParentHostId(this.datasource.getId());
                batchInput.setRelationships(new ArrayList<>(this.relationshipBatchInput.values()));
                // Don't keep hierarchy up-to-date during full sync, but instead update it afterwards, to speed up sync
                batchInput.setDisableAssetTreeIndexUpdateDuringSync(
                        this.getCurrentIndexingRequestSyncMode() == SyncMode.FULL);

                TreesApi treesApi = this.getAgentService().getIndexingApiProvider().createTreesApi();

                this.relationshipBatchSizeHelper.start();

                itemsFlushed = treesApi.batchMoveNodesToParents(batchInput).getItemUpdates();
                this.relationshipBatchSizeHelper.stop(this.relationshipBatchInput.size());
                this.logAndCountBadItems(itemsFlushed);
                StringBuilder logStr = new StringBuilder(
                        String.format(
                                "Sync Batch Stats (Relationships): %d items took %.2f seconds",
                                this.relationshipBatchInput.size(),
                                this.relationshipBatchSizeHelper.getLastDuration().toMillis() / 1000d));

                double ips = this.relationshipBatchSizeHelper.getLastItemsPerSecond();
                if (Double.isFinite(ips)) {
                    logStr.append(String.format(" at %.1f per second", ips));
                }

                LOG.debug(logStr.toString());
            }
        } finally {
            this.initRelationships();
        }

        return itemsFlushed;
    }

    @Override
    protected void connect() {
        this.connection.connect();
    }

    @Override
    protected void monitor() {
        if (!this.connection.monitor()) {
            this.disconnect();
            this.setConnectionState(ConnectionState.DISCONNECTED);
        }
    }

    @Override
    protected void disconnect() {
        this.connection.disconnect();
        this.setConnectionState(ConnectionState.DISCONNECTED);
    }

    @Override
    public Logger log() {
        return this.getLog();
    }

    @Override
    public void setDatasourceItemTypeCleanupFilter(List<String> itemTypeCleanupFilter) {
        this.datasourceItemTypeCleanupFilter = itemTypeCleanupFilter;
    }

    @Override
    public void setDatasourceItemDataIdRegexFilter(String regex) {
        this.datasourceItemDataIdRegexFilter = regex;
    }

    @Override
    public void setDatasourceItemDataIdExcludeRegexFilter(String regex) {
        this.datasourceItemDataIdExcludeRegexFilter = regex;
    }

    @Override
    public void setDatasourceItemNameRegexFilter(String regex) {
        this.datasourceItemNameRegexFilter = regex;
    }

    @Override
    public void setDatasourceItemNameExcludeRegexFilter(String regex) {
        this.datasourceItemNameExcludeRegexFilter = regex;
    }

    @Override
    public Logger getLog() {
        return this.guestLogger;
    }

    @Override
    public SignalResponseMessage signalRequest(SignalRequestMessage request) throws Exception {
        Preconditions.checkArgument(this.connection instanceof SignalPullDatasourceConnection,
                "AgentMessages.DataDocument protobuf can contain a SignalRequestMessage " +
                        "only if the connection is an instance of SignalPullDatasourceConnection");

        SignalPullDatasourceConnection pullConnection = (SignalPullDatasourceConnection) this.connection;

        SignalResponseMessage.Builder response = SignalResponseMessage.newBuilder();

        try (Stream<Sample> sampleInputs = pullConnection.getSamples(
                new GetSamplesParameters(request,
                        cursor -> {
                            if (cursor != null) {
                                response.setCursor(cursor.getTimestamp());
                            } else {
                                response.clearCursor();
                            }
                        }))) {

            Sample sample = null;
            Iterator<Sample> sampleInputsIterator = sampleInputs.iterator();
            while (response.getSampleCount() < request.getSampleLimit() && sampleInputsIterator.hasNext()) {
                // Allow this thread to be interrupted
                RequestCancellation.check();

                sample = sampleInputsIterator.next();

                if (sample.getKey().getTimestamp() <= request.getStartTime()) {
                    // If we are processing values to the left of the request range, only keep the last one
                    // we see that is on or before the start time. Otherwise they will count toward our sample
                    // limit and we may hit that limit before we've gotten to anything that's actually relevant
                    // to this request.
                    response.clearSample();
                }

                response.addSample(signalValueBuilderFromSample(sample));

                if (sample.getKey().getTimestamp() >= request.getEndTime()) {
                    // We have our right-bounding value, so exit. Besides just being good defensive programming,
                    // this is needed due to Proficy and other connections that expand the query interval by
                    // MaxInterpolation.
                    break;
                }
            }

            // If the last sample seen is not a right-bounding value and we hit the sample limit, then make
            // an assumption that there are more samples to be had. This will cause Appserver to issue another
            // request. There is an edge case where this extra request is unnecessary and only bounding values
            // will be returned, but the simplicity of this logic makes that OK.
            response.setHasMoreSamples(sample != null && sample.getKey().getTimestamp() < request.getEndTime() &&
                    response.getSampleCount() == request.getSampleLimit());
        }

        return response.build();
    }

    @Override
    public ConditionResponseMessage conditionRequest(ConditionRequestMessage request) throws Exception {
        Preconditions.checkArgument(this.connection instanceof ConditionPullDatasourceConnection,
                "AgentMessages.DataDocument protobuf can contain a ConditionRequestMessage " +
                        "only if the connection is an instance of ConditionPullDatasourceConnection");

        ConditionPullDatasourceConnection pullConnection = (ConditionPullDatasourceConnection) this.connection;

        ConditionResponseMessage.Builder response = ConditionResponseMessage.newBuilder();

        // For performance reasons only property transforms targeting "Type == Capsule" are executed on Capsules
        List<PropertyTransformer.Spec> capsuleTransforms =
                PropertyTransformer.findTransformsWithInputType(SeeqNames.Types.Capsule, this.transforms);

        try (Stream<Capsule> capsules =
                pullConnection.getCapsules(new GetCapsulesParameters(request,
                        cursor -> {
                            if (cursor != null) {
                                response.setCursor(cursor.getTimestamp());
                            } else {
                                response.clearCursor();
                            }
                        }))) {
            Capsule capsule = null;

            Iterator<Capsule> capsuleInputsIterator = capsules.iterator();
            while (capsuleInputsIterator.hasNext() && response.getCapsuleCount() < request.getCapsuleLimit()) {
                // Allow this thread to be interrupted
                RequestCancellation.check();

                capsule = capsuleInputsIterator.next();

                if (capsuleTransforms != null) {
                    capsule = PropertyTransformer.transform(capsule, request.getConditionId(), capsuleTransforms);
                }

                if (capsule.getStart().getTimestamp() <= request.getStartTime()) {
                    // If we are processing values to the left of the request range, only keep the last one
                    // we see that is on or before the start time. Otherwise they will count toward our capsule
                    // limit and we may hit that limit before we've gotten to anything that's actually relevant
                    // to this request.
                    response.clearCapsule();
                }

                response.addCapsule(capsule.toMessage());

                // Accept a single right boundary value even though conditions don't have right boundary values.
                // This keeps the exit logic the same as for signals. Also pipelines can use a capsule exactly at the
                // end of the request range to fill an end-inclusive data block.
                if (capsule.getStart().getTimestamp() >= request.getEndTime()) {
                    break;
                }
            }

            response.setHasMoreCapsules(capsule != null && capsule.getStart().getTimestamp() < request.getEndTime() &&
                    response.getCapsuleCount() == request.getCapsuleLimit());
        }

        return response.build();
    }

    private static final Map<SignalDataType, AddOnCalcSignalDataType> SIGNAL_DATA_TYPE_MAP =
            ImmutableMap.of(
                    SignalDataType.NUMERIC, AddOnCalcSignalDataType.NUMERIC,
                    SignalDataType.STRING, AddOnCalcSignalDataType.STRING
            );

    @Override
    public ExternalCalculationResponseMessage calculationRequest(ExternalCalculationRequestMessage request) {
        Preconditions.checkArgument(this.connection instanceof AddOnCalculationDatasourceConnection,
                "AgentMessages.DataDocument protobuf can contain ExternalCalculationRequest " +
                        "only if the connection is an instance of ExternalCalculationDatasourceConnection");

        AddOnCalculationDatasourceConnection calculationConnection =
                (AddOnCalculationDatasourceConnection) this.connection;

        try {
            calculationConnection.validateRequest(request);
        } catch (Exception e) {
            throw new OperationCanceledException(e);
        }

        ExternalCalculationResponseMessage.Builder response = ExternalCalculationResponseMessage.newBuilder();

        int numberOfSignals = (int) request.getSignalsCount();
        int numberOfSamplesPerSignal = request.getSampleCount();
        Long windowSize = request.hasInputWindowSize() ? request.getInputWindowSize() : null;

        AddOnCalcSignalDataType[] signalDataType = request.getSignalDataTypeList().stream()
                .map(SIGNAL_DATA_TYPE_MAP::get)
                .toArray(AddOnCalcSignalDataType[]::new);

        AddOnCalculationValidator detector = calculationConnection.createValidator().setWindowSize(windowSize);
        Lock validationLock = calculationConnection.getValidationLock();
        try {
            validationLock.lockInterruptibly();
            boolean haveEnoughSamples = numberOfSamplesPerSignal >= detector.getMinimumNumberOfSamples();
            if (calculationConnection.needsValidation() && haveEnoughSamples) {
                // The validation of the script is only done one time per script after a script has been changed.
                // Once the script has been validated and proved to not break the contract, the check is not anymore
                // made - the script (in fact the connection) being marked as valid.
                // If the script is changed in the filesystem, the connection is invalidated again and validation will
                // occur on first try to use again the script.

                Stream<List<Sample>> completeAlignedSignalsStream = request.getSampleList().stream()
                        .map(DatasourceConnectionV2Host::toSampleList);

                BiFunction<Integer, Stream<List<Sample>>, Stream<Sample>> scriptCallbackBiFunction =
                        (numberOfSamplesInTestStream, testStream) -> calculationConnection.getCalculatedResults(
                                request.getScript(),
                                testStream,
                                signalDataType,
                                numberOfSignals,
                                numberOfSamplesInTestStream,
                                windowSize);

                detector.setAlignedSignalsStreamComplete(completeAlignedSignalsStream)
                        .setSignalFragmentScriptCaller(scriptCallbackBiFunction)
                        .setScriptName(request.getScript());

                detector.doValidate();

                if (detector.isValid()) {
                    calculationConnection.markAsValid();
                }
            }
        } catch (InterruptedException e) {
            throw new OperationCanceledException(e);
        } finally {
            validationLock.unlock();
        }

        Stream<List<Sample>> alignedSignalsStream = request.getSampleList().stream()
                .map(DatasourceConnectionV2Host::toSampleList);

        try (
                Stream<Sample> results = calculationConnection.getCalculatedResults(request.getScript(),
                        alignedSignalsStream, signalDataType, numberOfSignals, numberOfSamplesPerSignal, windowSize)) {

            Iterator<Sample> resultsIterator = results.iterator();
            while (resultsIterator.hasNext()) {
                // Allow this thread to be interrupted
                RequestCancellation.check();

                Sample sample = resultsIterator.next();

                response.addSample(externalCalculationValueBuilderFromSample(sample));
            }

        }

        return response.build();
    }

    static List<Sample> toSampleList(ExternalCalculationMessages.SampleMultivaluedData multiValueData) {
        final long timestamp = multiValueData.getTimestamp();
        return multiValueData.getValueList()
                .stream()
                .map(value -> DatasourceConnectionV2Host.toSample(timestamp, value))
                .collect(Collectors.toList());
    }

    static Sample toSample(long timestamp, ExternalCalculationMessages.ValueData value) {
        Sample sample = new Sample();
        sample.setKey(new TimeInstant(timestamp));

        if (value.hasDataStatus() && value.getDataStatus() == ExternalCalculationMessages.DataStatus.BAD) {
            // null value in sample is treated as a BAD value
            sample.setValue(null);
        } else {

            if (value.hasDoubleValue()) {
                sample.setValue(value.getDoubleValue());
            } else if (value.hasStringValue()) {
                sample.setValue(value.getStringValue());
            } else {
                sample.setValue(null);
            }
        }
        return sample;
    }

    static ExternalCalculationMessages.SampleData.Builder externalCalculationValueBuilderFromSample(Sample sample) {
        TimeInstant timeInstant = sample.getKey();
        long timestamp = timeInstant.getTimestamp();

        ExternalCalculationMessages.SampleData.Builder valueBuilder =
                ExternalCalculationMessages.SampleData.newBuilder();
        ExternalCalculationMessages.ValueData.Builder valueData = ExternalCalculationMessages.ValueData.newBuilder();

        valueBuilder.setTimestamp(timestamp);

        if (sample.getValue() != null) {
            if (sample.getValue().getClass().equals(Byte.class)
                    || sample.getValue().getClass().equals(Short.class)
                    || sample.getValue().getClass().equals(Integer.class)
                    || sample.getValue().getClass().equals(Long.class)
                    || sample.getValue().getClass().equals(Float.class)
                    || sample.getValue().getClass().equals(Double.class)) {
                valueData.setDoubleValue(((Number) sample.getValue()).doubleValue());
            } else {
                valueData.setStringValue(sample.getValue().toString());
            }
        } else {
            valueData.setDataStatus(ExternalCalculationMessages.DataStatus.BAD);
        }

        valueBuilder.setValue(valueData);

        return valueBuilder;
    }

    static SignalResponseMessage.SampleData.Builder signalValueBuilderFromSample(Sample sample) {
        TimeInstant timeInstant = sample.getKey();
        long timestamp = timeInstant.getTimestamp();

        SignalResponseMessage.SampleData.Builder valueBuilder =
                SignalResponseMessage.SampleData.newBuilder();

        valueBuilder.setTimestamp(timestamp);

        if (sample.getValue() != null) {
            if (sample.getValue().getClass().equals(Byte.class)
                    || sample.getValue().getClass().equals(Short.class)
                    || sample.getValue().getClass().equals(Integer.class)
                    || sample.getValue().getClass().equals(Long.class)) {
                valueBuilder.setLongValue(((Number) sample.getValue()).longValue());
            } else if (sample.getValue().getClass().equals(Float.class)
                    || sample.getValue().getClass().equals(Double.class)) {
                valueBuilder.setDoubleValue(((Number) sample.getValue()).doubleValue());
            } else if (sample.getValue().getClass().equals(Boolean.class)) {
                valueBuilder.setBooleanValue(((Boolean) sample.getValue()).booleanValue());
            } else {
                valueBuilder.setStringValue(sample.getValue().toString());
            }
        } else {
            valueBuilder.setDataStatus(DataStatus.BAD);
        }
        return valueBuilder;
    }

    @Override
    public AuthResponseMessage authRequest(AuthRequestMessage request) {
        Preconditions.checkArgument(this.connection instanceof AuthDatasourceConnection, "AgentMessages.DataDocument " +
                "protobuf can contain a AuthRequest only if the connection is an instance of AuthDatasourceConnection");

        AuthDatasourceConnection authConnection = (AuthDatasourceConnection) this.connection;
        AuthResult authResult = authConnection.authRequest(toAuthParameters(request));
        return toResponseMessage(authResult).build();
    }

    @Override
    public OAuth2ConnectionMessages.OAuth2AuthResponseMessage oAuth2AuthRequest(
            OAuth2ConnectionMessages.OAuth2AuthRequestMessage oAuth2AuthRequest) {
        Preconditions.checkArgument(this.connection instanceof OAuth2DatasourceConnection,
                "AgentMessages.DataDocument protobuf can contain an OAuth2 Request only if the connection " +
                        "is an instance of OAuth2DatasourceConnection");
        OAuth2AuthResult authResult = ((OAuth2DatasourceConnection) this.connection)
                .oAuth2AuthRequest(toOAuth2AuthParameters(oAuth2AuthRequest));
        return toResponseMessage(authResult).build();
    }

    @Override
    public OAuth2ConnectionMessages.OAuth2PreAuthResponseMessage oAuth2PreAuthRequest(
            OAuth2ConnectionMessages.OAuth2PreAuthRequestMessage oAuth2PreAuthRequest) {
        Preconditions.checkArgument(this.connection instanceof OAuth2DatasourceConnection,
                "AgentMessages.DataDocument protobuf can contain an OAuth2 Request only if the connection " +
                        "is an instance of OAuth2DatasourceConnection");
        OAuth2PreAuthResult preAuthResult = ((OAuth2DatasourceConnection) this.connection)
                .oAuth2PreAuthRequest(toOAuth2PreAuthParameters(oAuth2PreAuthRequest));
        return toResponseMessage(preAuthResult).build();
    }

    @VisibleForTesting
    static AuthResponseMessage.Builder toResponseMessage(AuthResult authResult) {
        AuthResponseMessage.Builder response = AuthResponseMessage.newBuilder();

        response.setAuthenticated(authResult.isAuthenticated());
        Optional.ofNullable(authResult.getUserId()).ifPresent(response::setUserId);
        Optional.ofNullable(authResult.getSecurityId()).ifPresent(response::setSecurityId);
        Optional.ofNullable(authResult.getName()).ifPresent(response::setName);
        Optional.ofNullable(authResult.getEmailAddress()).ifPresent(response::setEmailAddress);
        Optional.ofNullable(authResult.getContinuation()).ifPresent(response::setContinuation);
        Optional.ofNullable(authResult.getFirstName()).ifPresent(response::setFirstName);
        Optional.ofNullable(authResult.getLastName()).ifPresent(response::setLastName);

        // add UserGroups into response
        if (authResult.getGroups() != null) {
            authResult.getGroups().stream()
                    .map(DatasourceConnectionV2Host::toUserGroupInfo)
                    .forEach(response::addUserGroup);

            // communicate to Seeq that user groups were provided
            // this helps in differentiating the situations: groups were not provided because the connector is not
            // able to provide them and will be managed in seeq vs. groups were not provided because the user has no
            // groups in AD and therefore Seeq may decide to remove them from the user
            response.setUserGroupsProvided(true);
        }
        Optional.ofNullable(authResult.getErrorMessage()).ifPresent(response::setMessage);

        return response;
    }

    @VisibleForTesting
    static OAuth2ConnectionMessages.OAuth2AuthResponseMessage.Builder toResponseMessage(
            OAuth2AuthResult authResult) {
        OAuth2ConnectionMessages.OAuth2AuthResponseMessage.Builder response =
                OAuth2ConnectionMessages.OAuth2AuthResponseMessage.newBuilder();
        Optional.ofNullable(authResult.getAccessToken()).ifPresent(response::setAccessToken);
        Optional.ofNullable(authResult.getAuthenticated()).ifPresent(response::setAuthenticated);
        Optional.ofNullable(authResult.getEmailAddress()).ifPresent(response::setEmailAddress);
        Optional.ofNullable(authResult.getFirstName()).ifPresent(response::setFirstName);
        Optional.ofNullable(authResult.getLastName()).ifPresent(response::setLastName);
        Optional.ofNullable(authResult.getName()).ifPresent(response::setName);
        Optional.ofNullable(authResult.getSubject()).ifPresent(response::setSubject);

        // add UserGroups into response
        if (authResult.getUserGroups() != null) {
            authResult.getUserGroups().stream()
                    .map(DatasourceConnectionV2Host::toOAuth2UserGroupInfo)
                    .forEach(response::addUserGroup);
        }
        // we want to differentiate between:
        // - authResult.getUserGroups() isEmpty - means the groups must be cleared in Seeq
        // - authResult.getUserGroups() == null - means the groups remains untouched in Seeq
        response.setUserGroupsProvided(authResult.getUserGroups() != null);

        return response;
    }

    private static OAuth2ConnectionMessages.OAuth2PreAuthResponseMessage.Builder toResponseMessage(
            OAuth2PreAuthResult preAuthResult) {
        OAuth2ConnectionMessages.OAuth2PreAuthResponseMessage.Builder response =
                OAuth2ConnectionMessages.OAuth2PreAuthResponseMessage.newBuilder();

        Optional.ofNullable(preAuthResult.getAuthorizationRequestURI()).ifPresent(response::setAuthorizationRequestURI);

        return response;
    }

    private static AuthResponseMessage.UserGroupInfo toUserGroupInfo(GroupInfo group) {
        AuthResponseMessage.UserGroupInfo.Builder userGroupInfoBuilder = AuthResponseMessage.UserGroupInfo.newBuilder();
        Optional.ofNullable(group.getSecurityId()).ifPresent(userGroupInfoBuilder::setSecurityId);
        Optional.ofNullable(group.getName()).ifPresent(userGroupInfoBuilder::setName);

        return userGroupInfoBuilder.build();
    }

    private static OAuth2ConnectionMessages.OAuth2AuthResponseMessage.UserGroupInfo toOAuth2UserGroupInfo(
            GroupInfo group) {
        OAuth2ConnectionMessages.OAuth2AuthResponseMessage.UserGroupInfo.Builder userGroupInfoBuilder =
                OAuth2ConnectionMessages.OAuth2AuthResponseMessage.UserGroupInfo.newBuilder();
        Optional.ofNullable(group.getSecurityId()).ifPresent(userGroupInfoBuilder::setSecurityId);
        Optional.ofNullable(group.getName()).ifPresent(userGroupInfoBuilder::setName);

        return userGroupInfoBuilder.build();
    }

    private static AuthParameters toAuthParameters(AuthRequestMessage request) {
        AuthParameters authParameters = new AuthParameters();
        if (request.hasUsername()) {
            authParameters.setUsername(request.getUsername());
        }
        if (request.hasPassword()) {
            authParameters.setPassword(request.getPassword());
        }
        if (request.hasSequenceId()) {
            authParameters.setSequenceId(request.getSequenceId());
        }
        return authParameters;
    }

    private static OAuth2PreAuthParameters toOAuth2PreAuthParameters(
            OAuth2ConnectionMessages.OAuth2PreAuthRequestMessage request) {
        return new OAuth2PreAuthParameters();
    }

    private static OAuth2AuthParameters toOAuth2AuthParameters(
            OAuth2ConnectionMessages.OAuth2AuthRequestMessage request) {
        OAuth2AuthParameters parameters = new OAuth2AuthParameters();

        if (request.hasCode()) {
            parameters.setCode(request.getCode());
        }
        if (request.hasState()) {
            parameters.setState(request.getState());
        }

        return parameters;
    }


    @FunctionalInterface
    private interface CheckedSupplier<T> {
        T get() throws ApiException;
    }

    @FunctionalInterface
    private interface CheckedConsumer<T> {
        void accept(T t) throws ApiException;

    }

    @Override
    public void saveConfig() {
        this.connection.saveConfig();
    }

    @Override
    public void initializeExport(ExportSamples exportInterface, ExportConnectionConfigV1 exportConfig) {
        if (this.exportOrchestrator != null) {
            throw new IllegalStateException("InitializeExport has already been called for this connection");
        }

        if (exportConfig.isEnabled()) {
            this.exportOrchestrator = new ExportOrchestrator(
                    this.connection.getDatasourceName(), exportInterface, this, exportConfig, 100000, null);

            this.exportOrchestrator.initialize();
        }
    }
}
