package com.seeq.link.sdk;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.seeq.api.ItemsApi;
import com.seeq.link.messages.agent.AgentMessages;
import com.seeq.link.messages.agent.AgentMessages.DataDocument;
import com.seeq.link.sdk.interfaces.AddOnCalculationDatasourceConnection;
import com.seeq.link.sdk.interfaces.AgentService;
import com.seeq.link.sdk.interfaces.AuthDatasourceConnection;
import com.seeq.link.sdk.interfaces.ConditionPullDatasourceConnection;
import com.seeq.link.sdk.interfaces.Connection.StateChangedEventArgs;
import com.seeq.link.sdk.interfaces.Connector;
import com.seeq.link.sdk.interfaces.ConnectorServiceV2;
import com.seeq.link.sdk.interfaces.ConnectorV2;
import com.seeq.link.sdk.interfaces.DatasourceConnection;
import com.seeq.link.sdk.interfaces.DatasourceConnectionV2;
import com.seeq.link.sdk.interfaces.IndexingDatasourceConnection;
import com.seeq.link.sdk.interfaces.NonConfigurableConnector;
import com.seeq.link.sdk.interfaces.OAuth2DatasourceConnection;
import com.seeq.link.sdk.interfaces.SignalPullDatasourceConnection;
import com.seeq.link.sdk.utilities.Event;
import com.seeq.model.ConnectionOutputV1;
import com.seeq.model.PropertyInputV1;
import com.seeq.utilities.SeeqNames;

import lombok.extern.slf4j.Slf4j;

/**
 * Hosts an IConnectorV2-based connector and acts as a bridge to the older IConnector interface via the
 * IConnectorServiceV2 interface.
 */
@Slf4j
public class ConnectorV2Host implements Connector, ConnectorServiceV2 {

    private final Logger guestLogger;
    private final ConnectorV2 connector;
    private ConfigObject config;

    private AgentService agentService;
    private final List<DatasourceConnectionV2Host> connectionsUnsafe = new ArrayList<>();

    private boolean saveConfigCalled = false;
    private String connectorDeveloperName;
    private String connectorDeveloperSupportUrl;

    @VisibleForTesting
    boolean wasSaveConfigCalled() {
        return this.saveConfigCalled;
    }

    public ConnectorV2Host(ConnectorV2 connector) {
        this.connector = connector;
        this.guestLogger = LoggerFactory.getLogger("com.seeq.link.plugin." +
                this.connector.getClass().getSimpleName());
    }

    @Override
    public Logger log() {
        return this.guestLogger;
    }

    @Override
    public String getName() {
        return this.connector.getName();
    }

    @Override
    public List<DatasourceConnection> getConnections() {
        synchronized (this.connectionsUnsafe) {
            return ImmutableList.copyOf(this.connectionsUnsafe);
        }
    }

    private final Event<StateChangedEventArgs> connectionStateChangedEvent = new Event<>();

    @Override
    public Event<StateChangedEventArgs> getConnectionStateChangedEvent() {
        return this.connectionStateChangedEvent;
    }

    public static AgentMessages.DatasourceService[] determineServices(DatasourceConnectionV2 connection) {
        List<AgentMessages.DatasourceService> connectionServices = new ArrayList<>();
        if (connection instanceof SignalPullDatasourceConnection) {
            connectionServices.add(AgentMessages.DatasourceService.SIGNAL);
        }

        if (connection instanceof ConditionPullDatasourceConnection) {
            connectionServices.add(AgentMessages.DatasourceService.CONDITION);
        }

        if (connection instanceof AddOnCalculationDatasourceConnection) {
            connectionServices.add(AgentMessages.DatasourceService.EXTERNAL_CALC);
        }

        if (connection instanceof AuthDatasourceConnection) {
            connectionServices.add(AgentMessages.DatasourceService.AUTH);
        }

        if (connection instanceof OAuth2DatasourceConnection) {
            connectionServices.add(AgentMessages.DatasourceService.OAUTH2);
        }

        if (connection instanceof IndexingDatasourceConnection) {
            connectionServices.add(AgentMessages.DatasourceService.CONNECTION_INDEX);
        }

        if (connectionServices.size() == 0) {
            connectionServices.add(AgentMessages.DatasourceService.API_ONLY);
        }

        return connectionServices.toArray(new AgentMessages.DatasourceService[connectionServices.size()]);
    }

    @Override
    public synchronized void addConnection(DatasourceConnectionV2 connection) {
        synchronized (this.connectionsUnsafe) {
            Optional<DatasourceConnectionV2Host> maybeDuplicateConnectionHost = this.connectionsUnsafe.stream()
                    .filter(c -> c.getDatasourceId().equalsIgnoreCase(connection.getDatasourceId()) ||
                            c.getDatasourceName().equalsIgnoreCase(connection.getDatasourceName()))
                    .findFirst();

            if (maybeDuplicateConnectionHost.isPresent()) {
                throw new IllegalArgumentException(
                        String.format("Connections with duplicate name or ID detected:\n'%s' '%s'\n'%s' '%s'\n" +
                                        "To fix, edit the '%s.json' configuration file and provide unique names and " +
                                        "IDs.",
                                maybeDuplicateConnectionHost.get().getDatasourceId(),
                                maybeDuplicateConnectionHost.get().getDatasourceName(),
                                connection.getDatasourceId(),
                                connection.getDatasourceName(),
                                this.connector.getName()));
            }

            DatasourceConnectionV2Host connectionHost = new DatasourceConnectionV2Host(this.agentService, this,
                    connection, determineServices(connection));
            connectionHost.getStateChangedEvent().add((sender, args) ->
                    this.connectionStateChangedEvent.dispatch(args.getSender(), args)
            );
            connectionHost.initialize();
            this.connectionsUnsafe.add(connectionHost);
        }
    }

    @Override
    public AgentService getAgentService() {
        return this.agentService;
    }

    @Override
    public synchronized void initialize(AgentService agentService) throws Exception {
        this.agentService = agentService;

        this.agentService.getConfigService().registerChangeCallback(this.connector.getName(),
                this::onConfigChanged);

        this.saveConfigCalled = false;
        this.connector.initialize(this);

        if (!this.saveConfigCalled && !(this.connector instanceof NonConfigurableConnector)) {
            if (this.config != null) {
                // If `loadConfig()` was called `this.config` was set, so we can save it automatically
                this.saveConfig();
                this.log().info("The connector '{}' didn't save its configuration during the `initialize` method, " +
                        "so it was saved automatically afterwards.", this.getName());
            } else {
                throw new IllegalStateException(String.format(
                        "The connector '%s' didn't save its configuration during the `initialize` method. " +
                                "A connector must either implement the `%s` interface or " +
                                "save the configuration by calling `connectorService.saveConfig(connectorConfig)`.",
                        this.getName(), NonConfigurableConnector.class.getSimpleName()));
            }
        }

        this.agentService.markConnectionsAsReinitialized(
                this.getConnections().stream().map(DatasourceConnection::getConnectionId).collect(Collectors.toList()));
    }

    @Override
    public synchronized void destroy() {
        this.agentService.getConfigService().unregisterChangeCallback(this.connector.getName());

        // We want to avoid synchronizing on connectionsUnsafe during connection.destroy() because this can take long
        // time.
        for (DatasourceConnection connection : this.getConnections()) {
            try {
                connection.destroy();
            } catch (Exception e) {
                this.log().error("Error encountered destroying connection '{}'", connection.getConnectionId(), e);
            }
        }

        synchronized (this.connectionsUnsafe) {
            this.connectionsUnsafe.clear();
        }

        try {
            this.connector.destroy();
        } catch (Exception e) {
            this.log().error("Error encountered destroying connector '{}'", this.connector.getName(), e);
        }
    }

    @Override
    public boolean processMessage(String destinationConnectorInstanceId, DataDocument document) {
        Optional<DatasourceConnection> connection = this.getConnections().stream()
                .filter(c -> c.getConnectionId().equals(destinationConnectorInstanceId)).findFirst();

        if (connection.isPresent()) {
            connection.get().processMessage(document);
            return true;
        } else {
            return false;
        }
    }

    @Override
    public ConfigObject loadConfig(ConfigObject[] supportedObjects) throws IOException {
        this.config = this.agentService.getConfigService().loadConfigObject(this.getName(), supportedObjects);

        return this.config;
    }

    @Override
    public void saveConfig() {
        if (this.config == null) {
            return;
        }

        this.saveConfig(this.config);
    }

    @Override
    public void saveConfig(ConfigObject configObject) {
        if (this.connector instanceof NonConfigurableConnector) {
            throw new IllegalStateException("saveConfig cannot be called for a NonConfigurableConnector");
        }

        this.config = configObject;

        this.agentService.getConfigService().saveConfigObject(this.getName(), this.config);
        this.saveDatasourceClassOnConfigurationObjects();
        this.saveConfigCalled = true;
    }

    @Override
    public void setConnectorDeveloperName(String connectorDeveloperName) {
        Preconditions.checkArgument(!StringUtils.isBlank(connectorDeveloperName),
                "ConnectorDeveloperName must not be blank");
        this.connectorDeveloperName = connectorDeveloperName;
    }

    @Override
    public void setConnectorDeveloperSupportUrl(String connectorDeveloperSupportUrl) {
        Preconditions.checkArgument(!StringUtils.isBlank(connectorDeveloperSupportUrl) &&
                        connectorDeveloperSupportUrl.startsWith("https://"),
                "ConnectorDeveloperSupportUrl must not be blank and must start with 'https://'");
        this.connectorDeveloperSupportUrl = connectorDeveloperSupportUrl;
    }

    @Override
    public String getConnectorDeveloperName() {
        return this.connectorDeveloperName;
    }

    @Override
    public String getConnectorDeveloperSupportUrl() {
        return this.connectorDeveloperSupportUrl;
    }

    private void saveDatasourceClassOnConfigurationObjects() {
        Map<String, String> datasourceClassesByDatasourceId = new HashMap<>();
        synchronized (this.connectionsUnsafe) {
            this.connectionsUnsafe.forEach(connection ->
                    datasourceClassesByDatasourceId.put(connection.getDatasourceId(), connection.getDatasourceClass()));
        }

        final Predicate<ConnectionOutputV1> datasourceConnectionExists =
                (connectionOutput) -> datasourceClassesByDatasourceId.containsKey(connectionOutput.getDatasourceId());

        final Predicate<ConnectionOutputV1> datasourceClassNeedsUpdate =
                (connectionOutput) -> !datasourceClassesByDatasourceId.get(connectionOutput.getDatasourceId())
                        .equals(connectionOutput.getDatasourceClass());

        ItemsApi itemsApi = this.agentService.getApiProvider().createItemsApi();
        this.agentService.getApiProvider().createAgentsApi()
                .getConnections(this.agentService.getAgentIdentification(), this.getName()).getConnections().stream()
                .filter(datasourceConnectionExists)
                .filter(datasourceClassNeedsUpdate)
                .forEach(connectionOutput -> itemsApi
                        .setProperty(connectionOutput.getId(), SeeqNames.Properties.DatasourceClass,
                                new PropertyInputV1()
                                        .value(datasourceClassesByDatasourceId.get(connectionOutput.getDatasourceId()))
                        ));
    }

    private synchronized void onConfigChanged(String name) {
        LOG.info("{} configuration changed", this.getName());

        try {
            this.destroy();
        } catch (Exception e) {
            LOG.error("Error destroying connector", e);
        }

        try {
            this.initialize(this.agentService);
        } catch (Exception e) {
            LOG.error("Error initializing connector", e);
        }
    }
}
