package com.seeq.link.sdk;

import java.io.IOException;

import com.seeq.link.sdk.interfaces.ConfigService;

/**
 * Facilitates a consistent implementation of configuration aspects of a connector or agent.
 *
 * @param <TConfig>
 *         The class for the top-level configuration object.
 */
public abstract class Configurable<TConfig> {

    public abstract String getName();

    private ConfigService configService;

    public ConfigService getConfigService() {
        return this.configService;
    }

    private TConfig config;

    protected TConfig getConfig() {
        return this.config;
    }

    private ConfigObject[] supportedConfigObjects;

    /**
     * Initializes the Object with information necessary to load/save configuration objects.
     *
     * @param configService
     *         The config service, as it was passed in by a higher-level coordinator.
     * @param supportedConfigObjects
     *         An array of ConfigObject instances that represent the possible Object types that can be successfully
     *         retrieved from the store. If the persisted Object type does not match anything in the list, then the
     *         first item in the array is passed back (being effectively a means to have a 'default' config).
     * @throws IOException
     *         Thrown if configuration file could not be successfully deserialized.
     */
    protected void initialize(ConfigService configService, ConfigObject[] supportedConfigObjects) throws IOException {
        this.configService = configService;
        this.configService.registerChangeCallback(this.getName(), (String name) -> this.onConfigFileChanged(name));

        this.supportedConfigObjects = supportedConfigObjects;

        this.loadConfig(this.getName());
    }

    public void destroy() {
        if (this.configService != null) {
            // Only if initialize has been called it make sense to unregister the callback
            this.configService.unregisterChangeCallback(this.getName());
        }
    }

    protected TConfig migrateConfig(Object configObject) {
        throw new ClassCastException(
                String.format("No migration routine specified. Need to migrate %s. " +
                        "Please override Configurable.migrateConfig() in your connector.", configObject.getClass()));
    }

    /**
     * Saves the current in-memory configuration to the configuration store via the ConfigService.
     */
    public void saveConfig() {
        if (this.config == null) {
            return;
        }

        this.configService.saveConfigObject(this.getName(), this.config);
    }

    @SuppressWarnings("unchecked")
    protected void loadConfig(String name) throws IOException {
        Object configObject = this.configService.loadConfigObject(name, this.supportedConfigObjects);

        if (this.supportedConfigObjects[0].getClass() != configObject.getClass()) {
            this.config = this.migrateConfig(configObject);
        } else {
            this.config = (TConfig) configObject;
        }

        this.saveConfig();
    }

    private void onConfigFileChanged(String name) {
        this.onConfigChanged();
    }

    protected void onConfigChanged() {
        // Do nothing in base class
    }
}
