package com.seeq.link.sdk;

import static com.seeq.link.messages.connector.extcalc.ExternalCalculationMessages.ExternalCalculationRequestMessage;
import static com.seeq.link.messages.connector.extcalc.ExternalCalculationMessages.ExternalCalculationResponseMessage;

import java.io.UnsupportedEncodingException;
import java.net.InetAddress;
import java.net.URLEncoder;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import javax.ws.rs.ProcessingException;
import javax.ws.rs.core.GenericType;

import org.apache.commons.lang.NotImplementedException;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;

import com.google.common.base.Stopwatch;
import com.google.common.collect.Maps;
import com.seeq.ApiClient;
import com.seeq.ApiException;
import com.seeq.Pair;
import com.seeq.api.AssetsApi;
import com.seeq.api.DatasourcesApi;
import com.seeq.api.ItemsApi;
import com.seeq.api.RequestsApi;
import com.seeq.api.SignalsApi;
import com.seeq.api.TreesApi;
import com.seeq.api.UserGroupsApi;
import com.seeq.link.messages.ErrorMessage;
import com.seeq.link.messages.agent.AgentMessages;
import com.seeq.link.messages.agent.AgentMessages.DataDocument;
import com.seeq.link.messages.agent.AgentMessages.DatasourceService;
import com.seeq.link.messages.connector.auth.AuthConnectionMessages.AuthRequestMessage;
import com.seeq.link.messages.connector.auth.AuthConnectionMessages.AuthResponseMessage;
import com.seeq.link.messages.connector.command.ConnectionIndexMessages;
import com.seeq.link.messages.connector.condition.ConditionConnectionMessages.ConditionRequestMessage;
import com.seeq.link.messages.connector.condition.ConditionConnectionMessages.ConditionResponseMessage;
import com.seeq.link.messages.connector.oauth2.OAuth2ConnectionMessages.OAuth2AuthRequestMessage;
import com.seeq.link.messages.connector.oauth2.OAuth2ConnectionMessages.OAuth2AuthResponseMessage;
import com.seeq.link.messages.connector.oauth2.OAuth2ConnectionMessages.OAuth2PreAuthRequestMessage;
import com.seeq.link.messages.connector.oauth2.OAuth2ConnectionMessages.OAuth2PreAuthResponseMessage;
import com.seeq.link.messages.connector.request.RequestMessages;
import com.seeq.link.messages.connector.signal.SignalConnectionMessages.SignalRequestMessage;
import com.seeq.link.messages.connector.signal.SignalConnectionMessages.SignalResponseMessage;
import com.seeq.link.sdk.interfaces.AgentService;
import com.seeq.link.sdk.interfaces.ConcurrentRequestsHandler;
import com.seeq.link.sdk.interfaces.ConcurrentRequestsHandlerProvider;
import com.seeq.link.sdk.interfaces.Connector;
import com.seeq.link.sdk.interfaces.DatasourceConnection;
import com.seeq.link.sdk.interfaces.DatasourceConnectionV2;
import com.seeq.link.sdk.interfaces.IndexingDatasourceConnection;
import com.seeq.link.sdk.interfaces.PullDatasourceConnection;
import com.seeq.link.sdk.interfaces.SyncMode;
import com.seeq.link.sdk.interfaces.SyncResult;
import com.seeq.link.sdk.interfaces.SyncStatus;
import com.seeq.link.sdk.services.PropertyTransformer;
import com.seeq.link.sdk.utilities.BatchSizeHelper;
import com.seeq.link.sdk.utilities.ConnectorHelper;
import com.seeq.link.sdk.utilities.DefaultConcurrentRequestsHandler;
import com.seeq.link.sdk.utilities.RequestCancellation;
import com.seeq.link.sdk.utilities.ThreadCollection;
import com.seeq.link.sdk.utilities.TimeInstant;
import com.seeq.model.AssetBatchInputV1;
import com.seeq.model.PutAssetInputV1;
import com.seeq.model.AssetTreeBatchInputV1;
import com.seeq.model.AssetTreeSingleInputV1;
import com.seeq.model.DatasourceCleanUpInputV1;
import com.seeq.model.DatasourceCleanUpOutputV1;
import com.seeq.model.DatasourceInputV1;
import com.seeq.model.DatasourceOutputV1;
import com.seeq.model.ItemBatchOutputV1;
import com.seeq.model.ItemIdListInputV1;
import com.seeq.model.ItemUpdateOutputV1;
import com.seeq.model.PutSignalsInputV1;
import com.seeq.model.PutUserGroupsInputV1;
import com.seeq.model.ScalarPropertyV1;
import com.seeq.model.SignalWithIdInputV1;
import com.seeq.model.UserGroupWithIdInputV1;
import com.seeq.utilities.ManualResetEvent;
import com.seeq.utilities.SeeqNames;
import com.seeq.utilities.exception.OperationCanceledException;

import lombok.extern.slf4j.Slf4j;


/**
 * Base class to handle connections to remote data sources.
 *
 * @param <TConnector>
 *         The class of the connector that hosts these connections.
 */
@Slf4j
abstract class BaseDatasourceConnection<TConnector extends Connector>
        extends BaseConnection implements DatasourceConnection {

    private final AgentService agentService;

    public AgentService getAgentService() {
        return this.agentService;
    }

    private final TConnector connector;

    public TConnector getConnector() {
        return this.connector;
    }

    public BaseDatasourceConnection(
            AgentService agentService,
            TConnector connector,
            String datasourceClass,
            String datasourceName,
            String datasourceId,
            IndexingSchedule indexingSchedule,
            DatasourceService[] services,
            Integer maxConcurrentRequests,
            Integer maxResultsPerRequest) {
        this.agentService = agentService;
        this.connector = connector;
        this.datasourceClass = datasourceClass;
        this.datasourceName = datasourceName;
        this.datasourceId = datasourceId;
        this.indexingSchedule = indexingSchedule;
        this.services = services;

        this.connectionId = this.generateConnectionId();
        this.connectorDeveloperInfo = ConnectorHelper.generateConnectorDeveloperInfo(this.connector);

        maxConcurrentRequests = maxConcurrentRequests != null ? maxConcurrentRequests : Integer.MAX_VALUE;
        this.concurrentRequestsHandler = new DefaultConcurrentRequestsHandler(maxConcurrentRequests);

        if (maxResultsPerRequest != null) {
            this.maxResultsPerRequest = maxResultsPerRequest;
        }
    }

    BaseDatasourceConnection(
            AgentService agentService,
            TConnector connector,
            DatasourceConnectionV2 connectionV2,
            DatasourceService[] services) {
        this.agentService = agentService;
        this.connector = connector;
        this.datasourceClass = connectionV2.getDatasourceClass();
        this.datasourceName = connectionV2.getDatasourceName();
        this.datasourceId = connectionV2.getDatasourceId();

        if (connectionV2 instanceof IndexingDatasourceConnection) {
            this.indexingSchedule = ((IndexingDatasourceConnection) connectionV2).getConfiguration().getIndexing();
        } else {
            this.indexingSchedule = null;
        }

        this.services = services;

        this.connectionId = this.generateConnectionId();
        this.connectorDeveloperInfo = ConnectorHelper.generateConnectorDeveloperInfo(this.connector);

        if (connectionV2 instanceof PullDatasourceConnection) {
            PullDatasourceConnection pullDatasourceConnection = (PullDatasourceConnection) connectionV2;
            if (pullDatasourceConnection.getMaxResultsPerRequest() != null) {
                this.maxResultsPerRequest = pullDatasourceConnection.getMaxResultsPerRequest();
            }
        }

        int maxConcurrentRequests = Integer.MAX_VALUE;
        if (connectionV2 instanceof PullDatasourceConnection) {
            maxConcurrentRequests =
                    Optional.ofNullable(((PullDatasourceConnection) connectionV2).getMaxConcurrentRequests())
                            .orElse(Integer.MAX_VALUE);
        }

        if (connectionV2 instanceof ConcurrentRequestsHandlerProvider) {
            this.concurrentRequestsHandler =
                    ((ConcurrentRequestsHandlerProvider) connectionV2).getConcurrentRequestsHandler();
        } else {
            this.concurrentRequestsHandler = new DefaultConcurrentRequestsHandler(maxConcurrentRequests);
        }
    }

    private String generateConnectionId() {
        String machineName;

        try {
            machineName = InetAddress.getLocalHost().getHostName();
        } catch (UnknownHostException e) {
            machineName = "Unknown Machine Name";
        }

        return (this.agentService != null ? this.agentService.getDisplayName() + ": " : "") +
                machineName + ": " + this.datasourceClass + ": " + this.datasourceName + ": " + this.datasourceId;
    }

    @Override
    public void destroy() {
        this.disable();
    }

    private final String datasourceClass;

    @Override
    public String getDatasourceClass() {
        return this.datasourceClass;
    }

    private final String datasourceName;

    @Override
    public String getDatasourceName() {
        return this.datasourceName;
    }

    private final String datasourceId;

    @Override
    public String getDatasourceId() {
        return this.datasourceId;
    }

    @Override
    public abstract boolean isPullDatasourceConnection();

    @Override
    public abstract boolean isIndexingDatasourceConnection();

    protected DatasourceService[] services;

    @Override
    public DatasourceService[] getServices() {
        return this.services;
    }

    private final String connectionId;

    private final String connectorDeveloperInfo;

    @Override
    public String getConnectionId() {
        return this.connectionId;
    }

    private IndexingSchedule indexingSchedule = new IndexingSchedule();

    @Override
    public IndexingSchedule getIndexingSchedule() {
        return this.indexingSchedule;
    }

    private SyncMode currentIndexingRequestSyncMode = SyncMode.NONE;

    @Override
    public SyncMode getCurrentIndexingRequestSyncMode() {
        return this.currentIndexingRequestSyncMode;
    }

    @Override
    public void setCurrentIndexingRequestSyncMode(SyncMode syncMode) {
        this.currentIndexingRequestSyncMode = syncMode;
    }

    private final IndexingState indexingState = new IndexingState();
    private int maxResultsPerRequest = Integer.MAX_VALUE;
    private final ConcurrentRequestsHandler concurrentRequestsHandler;

    int itemsWithErrors = 0;

    @Override
    public IndexingState getIndexingState() {
        return this.indexingState;
    }

    @Override
    public String toString() {
        return String.format("%s: %s [%s]",
                this.getDatasourceClass(),
                this.getDatasourceName(),
                this.getState());
    }

    public SignalResponseMessage signalRequest(SignalRequestMessage request) throws Exception {
        throw new NotImplementedException("Connector does not respond to signal requests.");
    }

    public ConditionResponseMessage conditionRequest(ConditionRequestMessage request) throws Exception {
        throw new NotImplementedException("Connector does not respond to condition requests.");
    }

    public ExternalCalculationResponseMessage calculationRequest(ExternalCalculationRequestMessage calculationRequest) {
        throw new NotImplementedException("Connector does not respond to calculation requests.");
    }

    public AuthResponseMessage authRequest(AuthRequestMessage authRequest) {
        throw new NotImplementedException("Connector does not respond to Auth requests.");
    }

    public OAuth2AuthResponseMessage oAuth2AuthRequest(OAuth2AuthRequestMessage oAuth2AuthRequest) {
        throw new NotImplementedException("Connector does not respond to OAuth 2.0 requests.");
    }

    public OAuth2PreAuthResponseMessage oAuth2PreAuthRequest(OAuth2PreAuthRequestMessage oAuth2PreAuthRequest) {
        throw new NotImplementedException("Connector does not respond to OAuth 2.0 requests.");
    }

    @Override
    public void processMessage(AgentMessages.DataDocument data) {
        if (this.getState() == ConnectionState.DISABLED) {
            this.getLog().info("Received data document but DatasourceConnection is disabled. Ignoring.");
            return;
        }

        RequestMessages.TransactionMessage transaction = data.getTransaction();

        if (transaction.hasCancellation()) {
            RequestMessages.CancellationMessage cancellationMessage = transaction.getCancellation();
            if (cancellationMessage.hasRequestIdToCancel()) {
                long requestIdToCancel = cancellationMessage.getRequestIdToCancel();
                this.getBackgroundThreads().interrupt(requestIdToCancel);
                return; // We don't send responses to cancellation requests
            }
        }

        RequestMessages.RequestMessage request = transaction.getRequest();
        long requestId = request.getRequestId();
        long timeoutMillis = request.hasTimeoutNanos() ?
                request.getTimeoutNanos() / 1000000 : ThreadCollection.NO_TIMEOUT;

        Stopwatch queueDuration = Stopwatch.createStarted();
        this.concurrentRequestsHandler.runWhenPermitted(() -> {
            queueDuration.stop();
            Stopwatch datasourceDuration = Stopwatch.createStarted();

            // Set thread name to help with reading log output
            Thread.currentThread().setName(String.format("%s [#%d]", this.getConnectionId(), requestId));

            DataDocument.Builder responseDocument = DataDocument.newBuilder();
            responseDocument.setDestinationConnectionId(this.getConnectionId());

            RequestMessages.TransactionMessage.Builder responseTransaction =
                    RequestMessages.TransactionMessage.newBuilder();

            RequestMessages.ResponseMessage.Builder response = RequestMessages.ResponseMessage.newBuilder();
            response.setRequestId(requestId);

            if (request.hasSignalRequest()) {
                SignalRequestMessage signalRequest = request.getSignalRequest();
                try {
                    // Override sample limit if necessary
                    String overrideLogSnippet = "";
                    if (this.maxResultsPerRequest < signalRequest.getSampleLimit()) {
                        overrideLogSnippet = " (by MaxResultsPerRequest in connection config)";
                        SignalRequestMessage.Builder signalRequestBuilder = signalRequest.toBuilder();
                        signalRequestBuilder.setSampleLimit(Math.min(signalRequest.getSampleLimit(),
                                this.maxResultsPerRequest));
                        signalRequest = signalRequestBuilder.build();
                    }

                    this.getLog().debug(
                            "Received SignalRequestMessage with RequestID {} for SignalId {}" +
                                    " from {} to {}, limited{} to {}",
                            requestId, signalRequest.getSignalId(),
                            new TimeInstant(signalRequest.getStartTime()),
                            new TimeInstant(signalRequest.getEndTime()),
                            overrideLogSnippet,
                            signalRequest.getSampleLimit());

                    SignalResponseMessage signalResponse = this.signalRequest(signalRequest);

                    this.getLog()
                            .debug(
                                    "Returning SignalResponseMessage with RequestID {} for SignalId {}" +
                                            " from {} to {}, with {} samples and HasMoreSamples=={}, took" +
                                            " {} seconds",
                                    requestId, signalRequest.getSignalId(),
                                    new TimeInstant(signalRequest.getStartTime()),
                                    new TimeInstant(signalRequest.getEndTime()),
                                    signalResponse.getSampleCount(), signalResponse.getHasMoreSamples(),
                                    datasourceDuration.elapsed(TimeUnit.MILLISECONDS) / 1000.0d);

                    response.setSignalResponse(signalResponse);
                } catch (Exception e) {
                    SignalResponseMessage.Builder signalResponse = SignalResponseMessage.newBuilder();
                    signalResponse.setErrorInfo(this.buildErrorMessage(e, signalRequest));
                    response.setSignalResponse(signalResponse.build());
                }
            } else if (request.hasConditionRequest()) {
                ConditionRequestMessage conditionRequest = request.getConditionRequest();
                try {
                    // Override capsule limit if necessary
                    String overrideLogSnippet = "";
                    if (this.maxResultsPerRequest < conditionRequest.getCapsuleLimit()) {
                        overrideLogSnippet = " (by MaxResultsPerRequest in connection config)";
                        ConditionRequestMessage.Builder conditionRequestBuilder = conditionRequest.toBuilder();
                        conditionRequestBuilder.setCapsuleLimit(Math.min(conditionRequest.getCapsuleLimit(),
                                this.maxResultsPerRequest));
                        conditionRequest = conditionRequestBuilder.build();
                    }

                    this.getLog().debug(
                            "Received ConditionRequestMessage with RequestID {} for ConditionId {}" +
                                    " from {} to {}, limited{} to {}",
                            requestId, conditionRequest.getConditionId(),
                            new TimeInstant(conditionRequest.getStartTime()),
                            new TimeInstant(conditionRequest.getEndTime()),
                            overrideLogSnippet,
                            conditionRequest.getCapsuleLimit());

                    ConditionResponseMessage conditionResponse = this.conditionRequest(conditionRequest);

                    this.getLog().debug(
                            "Returning ConditionResponseMessage with RequestID {} for " +
                                    "ConditionId {} from {} to {}, with {} capsules and " +
                                    "HasMoreCapsules=={}, took {} seconds",
                            requestId, conditionRequest.getConditionId(),
                            new TimeInstant(conditionRequest.getStartTime()),
                            new TimeInstant(conditionRequest.getEndTime()),
                            conditionResponse.getCapsuleCount(), conditionResponse.getHasMoreCapsules(),
                            datasourceDuration.elapsed(TimeUnit.MILLISECONDS) / 1000.0d);

                    response.setConditionResponse(conditionResponse);
                } catch (Exception e) {
                    ConditionResponseMessage.Builder conditionResponse = ConditionResponseMessage.newBuilder();
                    conditionResponse.setErrorInfo(this.buildErrorMessage(e, conditionRequest));
                    response.setConditionResponse(conditionResponse.build());
                }

            } else if (request.hasExternalCalculationRequest()) {
                ExternalCalculationRequestMessage calculationRequest = request.getExternalCalculationRequest();
                try {
                    int numberOfSamples = 0;
                    if (calculationRequest.getSampleList() != null) {
                        numberOfSamples = calculationRequest.getSampleCount();
                    }
                    this.getLog().debug(
                            "Received ExternalCalculationRequest with RequestID {} for script {} with {} key" +
                                    " (s) in the input signals",
                            requestId, calculationRequest.getScript(), numberOfSamples);

                    ExternalCalculationResponseMessage calculationResponse =
                            this.calculationRequest(calculationRequest);

                    this.getLog().debug(
                            "Returning ExternalCalculationResponse with RequestID {} for " +
                                    "Script {} having {} samples in response, took {} seconds",
                            requestId, calculationRequest.getScript(),
                            calculationResponse.getSampleCount(),
                            datasourceDuration.elapsed(TimeUnit.MILLISECONDS) / 1000.0d);

                    response.setExternalCalculationResponse(calculationResponse);
                } catch (Exception e) {
                    ExternalCalculationResponseMessage.Builder
                            calculationResponse = ExternalCalculationResponseMessage
                            .newBuilder();

                    long start = calculationRequest.getSampleCount() > 0
                            ? calculationRequest.getSampleList().get(0).getTimestamp()
                            : 0L;
                    long end = calculationRequest.getSampleCount() > 0
                            ? calculationRequest.getSampleList()
                            .get(calculationRequest.getSampleCount() - 1).getTimestamp()
                            : 0L;
                    calculationResponse.setErrorInfo(this.buildDataRequestErrorMessage(e,
                            "add-on calculation", calculationRequest.getScript(), start, end));
                    response.setExternalCalculationResponse(calculationResponse.build());
                }
            } else if (request.hasAuthRequest()) {
                AuthRequestMessage AuthRequest = request.getAuthRequest();
                try {
                    this.getLog().debug("Received AuthRequest with RequestID {}, length {}",
                            requestId, data.getSerializedSize());

                    AuthResponseMessage authResponse = this.authRequest(AuthRequest);

                    this.getLog().debug("Returning AuthResponse with RequestID {}, took {} seconds", requestId,
                            datasourceDuration.elapsed(TimeUnit.MILLISECONDS) / 1000.0d);

                    response.setAuthResponse(authResponse);

                } catch (Exception e) {
                    AuthResponseMessage.Builder authResponse = AuthResponseMessage.newBuilder();
                    authResponse.setAuthenticated(false);
                    authResponse.setErrorInfo(this.buildErrorMessage(ErrorMessage.ErrorCode.EXCEPTION, e,
                            String.format("Error processing Auth request with id %s", requestId)));
                    response.setAuthResponse(authResponse.build());
                }
            } else if (request.hasOAuth2AuthRequest()) {
                OAuth2AuthRequestMessage oAuth2AuthRequest = request.getOAuth2AuthRequest();
                try {
                    this.getLog().debug("Received OAuth2AuthRequest with RequestID {}, length {}",
                            requestId, data.getSerializedSize());

                    OAuth2AuthResponseMessage oAuth2AuthResponse = this.oAuth2AuthRequest(oAuth2AuthRequest);

                    this.getLog().debug("Returning OAuth2AuthResponse with RequestID {}, took {} seconds",
                            requestId,
                            datasourceDuration.elapsed(TimeUnit.MILLISECONDS) / 1000.0d);

                    response.setOAuth2AuthResponse(oAuth2AuthResponse);

                } catch (Exception e) {
                    OAuth2AuthResponseMessage.Builder oAuth2AuthResponse = OAuth2AuthResponseMessage.newBuilder()
                            .setAuthenticated(false)
                            .setErrorInfo(this.buildErrorMessage(
                                    ErrorMessage.ErrorCode.EXCEPTION,
                                    e,
                                    String.format(
                                            "Error processing OAuth 2.0 authentication request with id %s",
                                            requestId
                                    )
                            ));
                    response.setOAuth2AuthResponse(oAuth2AuthResponse.build());
                }
            } else if (request.hasOAuth2PreAuthRequest()) {
                OAuth2PreAuthRequestMessage oAuth2PreAuthRequest = request.getOAuth2PreAuthRequest();
                try {
                    this.getLog().debug("Received OAuth2PreAuthRequest with RequestID {}, length {}",
                            requestId, data.getSerializedSize());

                    OAuth2PreAuthResponseMessage oAuth2PreAuthResponse =
                            this.oAuth2PreAuthRequest(oAuth2PreAuthRequest);

                    this.getLog().debug("Returning OAuth2PreAuthResponse with RequestID {}, took {} seconds",
                            requestId,
                            datasourceDuration.elapsed(TimeUnit.MILLISECONDS) / 1000.0d);

                    response.setOAuth2PreAuthResponse(oAuth2PreAuthResponse);

                } catch (Exception e) {
                    OAuth2PreAuthResponseMessage.Builder oAuth2PreAuthResponse =
                            OAuth2PreAuthResponseMessage.newBuilder();
                    oAuth2PreAuthResponse.setErrorInfo(this.buildErrorMessage(
                            ErrorMessage.ErrorCode.EXCEPTION,
                            e,
                            String.format("Error processing OAuth 2.0 pre-auth request with id %s", requestId)
                    ));
                    response.setOAuth2PreAuthResponse(oAuth2PreAuthResponse.build());
                }
            } else if (request.hasConnectionIndexRequest()) {
                ConnectionIndexMessages.ConnectionIndexRequestMessage connectionIndexRequest =
                        request.getConnectionIndexRequest();
                ConnectionIndexMessages.ConnectionIndexResponseMessage.Builder connectionIndexResponse =
                        ConnectionIndexMessages.ConnectionIndexResponseMessage.newBuilder();

                this.getLog().debug("Received ConnectionIndexRequestMessage with RequestID {}, length {}",
                        requestId, data.getSerializedSize());

                this.agentService.requestIndex(this, toSyncMode(connectionIndexRequest.getSyncMode()));
                connectionIndexResponse.setMessage("Connection queued for indexing.");

                response.setConnectionIndexResponse(connectionIndexResponse.build());

            }

            datasourceDuration.stop();

            RequestMessages.MonitorData.Builder monitorData = RequestMessages.MonitorData.newBuilder();
            monitorData.setQueueNanos(queueDuration.elapsed(TimeUnit.NANOSECONDS));
            monitorData.setDatasourceNanos(datasourceDuration.elapsed(TimeUnit.NANOSECONDS));
            monitorData.setRequestNanos(monitorData.getQueueNanos() + monitorData.getDatasourceNanos());
            response.setMonitorData(monitorData.build());

            responseTransaction.setResponse(response.build());
            responseDocument.setTransaction(responseTransaction.build());

            this.sendMessage(responseDocument.build());
        }, this.getBackgroundThreads(), timeoutMillis, requestId, new ManualResetEvent(false));

    }

    @NotNull
    private static SyncMode toSyncMode(ConnectionIndexMessages.ConnectionIndexRequestMessage.SyncMode syncMode) {
        return SyncMode.valueOf(syncMode.name());
    }

    protected void sendMessage(AgentMessages.DataDocument data) {
        this.agentService.sendMessage(this, data);
    }

    public abstract Logger getLog();

    @Override
    public void spawnMetadataSync(SyncMode syncMode, Consumer<SyncResult> callback) {
        this.getBackgroundThreads().spawn(() -> {
            SyncResult syncResult = SyncResult.FAILED;
            try {
                Thread.currentThread().setName("Metadata sync for " + this.getConnectionId());

                this.getLog().info("Metadata sync starting, sync mode " + syncMode.toString());
                this.metadataSync(syncMode);
                this.getLog().info("Metadata sync success");

                syncResult = SyncResult.SUCCESS;
            } catch (OperationCanceledException e) {
                this.getLog().warn("Metadata sync interrupted");
                syncResult = SyncResult.INTERRUPTED;
            } catch (Throwable e) {
                this.getLog().error("Metadata sync failure:", e);
                this.getLog().error("Note: {} was developed by {}", this.connector.getName(),
                        this.connectorDeveloperInfo);
            } finally {
                // This always has to be called so that the agent can schedule the next sync
                callback.accept(syncResult);
            }
        });
    }

    /**
     * A pair of info about a datasource and a boolean indicating whether it was just created or was preexisting.
     */
    public static class DatasourceAndCreationInfo {
        public DatasourceAndCreationInfo(DatasourceOutputV1 datasource, boolean newlyCreated) {
            this.datasource = datasource;
            this.newlyCreated = newlyCreated;
        }

        public DatasourceOutputV1 datasource;
        public boolean newlyCreated;
    }

    /**
     * Get information about a datasource with the specified identifier. If the identifier matches an existing
     * datasource, its information will be returned, otherwise a new datasource will be created and returned.
     *
     * If multiple datasources are found, or a single datasource cannot be created or retrieved, returns null.
     *
     * @param storedInSeeq
     *         Whether this datasource's data will be stored in Seeq (true) or a remote datasource (false). If set to
     *         true, the datasource will also get default "Signal Location" and "Condition Location" properties,
     *         which determine where in Seeq those items are stored. These can be changed after datasource creation.
     * @return Information about the datasource and whether it was newly created
     */
    public DatasourceAndCreationInfo getOrCreateDatasource(boolean storedInSeeq) {
        return this.getOrCreateDatasource(storedInSeeq, false, Collections.emptyList());
    }

    /**
     * Get information about a datasource with the specified identifier. If the identifier matches an existing
     * datasource, its information will be returned, otherwise a new datasource will be created and returned.
     *
     * If multiple datasources are found, or a single datasource cannot be created or retrieved, returns null.
     *
     * @param storedInSeeq
     *         Whether this datasource's data will be stored in Seeq (true) or a remote datasource (false). If set to
     *         true, the datasource will also get default "Signal Location" and "Condition Location" properties,
     *         which determine where in Seeq those items are stored. These can be changed after datasource creation.
     * @param enableCache
     *         Whether this datasource's signal will be cached in Seeq.
     * @return Information about the datasource and whether it was newly created
     */
    public DatasourceAndCreationInfo getOrCreateDatasource(boolean storedInSeeq, boolean enableCache) {
        return this.getOrCreateDatasource(storedInSeeq, enableCache, Collections.emptyList());
    }

    /**
     * Get information about a datasource with the specified identifier. If the identifier matches an existing
     * datasource, its information will be returned, otherwise a new datasource will be created and returned.
     *
     * If multiple datasources are found, or a single datasource cannot be created or retrieved, returns null.
     *
     * @param storedInSeeq
     *         Whether this datasource's data will be stored in Seeq (true) or a remote datasource (false). If set to
     *         true, the datasource will also get default "Signal Location" and "Condition Location" properties,
     *         which determine where in Seeq those items are stored. These can be changed after datasource creation.
     * @param enableCache
     *         Whether this datasource's signal will be cached in Seeq.
     * @param additionalProperties
     *         Additional properties that should be set on the datasource. The 'normal' datasource properties cannot be
     *         set with this parameter, only 'custom' properties.
     * @return Information about the datasource and whether it was newly created
     */
    public DatasourceAndCreationInfo getOrCreateDatasource(boolean storedInSeeq, boolean enableCache,
            List<ScalarPropertyV1> additionalProperties) {
        DatasourcesApi datasourcesApi = this.getAgentService().getIndexingApiProvider().createDatasourcesApi();

        try {
            List<DatasourceOutputV1> datasources = datasourcesApi.getDatasources(this.getDatasourceClass(),
                    this.getDatasourceId(), 0, 100, true).getDatasources();

            if (datasources.size() == 0) {
                // No datasources matched, so create a new datasource and return it.
                DatasourceInputV1 datasourceInput = new DatasourceInputV1();
                datasourceInput.setDatasourceClass(this.getDatasourceClass());
                datasourceInput.setDatasourceId(this.getDatasourceId());
                datasourceInput.setName(this.getDatasourceName());
                datasourceInput.setStoredInSeeq(storedInSeeq);
                datasourceInput.setCacheEnabled(enableCache);
                datasourceInput.setIndexingScheduleSupported(this.isIndexingScheduleSupported());
                datasourceInput.setAdditionalProperties(additionalProperties);
                return new DatasourceAndCreationInfo(datasourcesApi.createDatasource(datasourceInput), true);
            } else if (datasources.size() == 1) {
                // Datasource exists, so modify the properties if necessary, then return the updated datasource model.
                DatasourceOutputV1 datasource = this.updateExistingDatasource(datasources.get(0), storedInSeeq,
                        additionalProperties);
                return new DatasourceAndCreationInfo(datasource, false);
            } else {
                // Multiple datasources matched, don't know how to proceed...
                String errorString = String.format(
                        "Multiple datasources matched when querying for datasourceClass=%s, datasourceIdentifier=%s!",
                        this.getDatasourceClass(), this.getDatasourceId());
                throw new Exception(errorString);
            }
        } catch (Exception e) {
            this.getLog().error("", e);
        }

        return null;
    }

    /**
     * Updates an existing datasource in Seeq by changing its additional properties and storedInSeeq flag
     *
     * @param datasource
     *         the datasource to update
     * @param storedInSeeq
     *         Whether this datasource's data will be stored in Seeq (true) or a remote datasource (false). If set to
     *         true, the datasource will also get default "Signal Location" and "Condition Location" properties,
     *         which determine where in Seeq those items are stored. These can be changed after datasource creation.
     * @param additionalProperties
     *         Additional properties that should be set on the datasource. The 'normal' datasource properties cannot be
     *         set with this parameter, only 'custom' properties.
     * @return the datasource object retrieved from seeq after update or the same datasource object received as
     *         parameter if no update was necessary
     */
    protected DatasourceOutputV1 updateExistingDatasource(DatasourceOutputV1 datasource, boolean storedInSeeq,
            List<ScalarPropertyV1> additionalProperties) {
        DatasourcesApi datasourcesApi = this.getAgentService().getIndexingApiProvider().createDatasourcesApi();

        String datasourceGuid = datasource.getId();
        ItemsApi itemsApi = this.getAgentService().getIndexingApiProvider().createItemsApi();
        List<ScalarPropertyV1> propertiesToChange = new ArrayList<>();
        // Override the "Stored in Seeq" property, regardless of its current value
        if (!Objects.equals(storedInSeeq, datasource.getStoredInSeeq())) {
            propertiesToChange.add(new ScalarPropertyV1()
                    .name(SeeqNames.Properties.StoredInSeeq)
                    .value(storedInSeeq));
        }

        // Ensure this datasource is not archived, since we're going to be using it.
        if (datasource.getIsArchived()) {
            propertiesToChange.add(new ScalarPropertyV1()
                    .name(SeeqNames.Properties.Archived)
                    .value(false));
        }

        if (!Objects.equals(this.isIndexingScheduleSupported(), datasource.getIndexingScheduleSupported())) {
            propertiesToChange.add(new ScalarPropertyV1()
                    .name(SeeqNames.Properties.IndexingScheduleSupported)
                    .value(this.isIndexingScheduleSupported()));
        }

        // "Cache Enabled" can be manually enabled by users. We *don't* want to clobber this value if they've
        // changed it. So we'll only set the default for "Cache Enabled" once, at datasource creation time.

        // For the "additional" properties, we'll update the value if it doesn't match what we we expect.
        // We will not remove or change properties that aren't in the list provided to this method.
        if (!additionalProperties.isEmpty()) {
            Map<String, ScalarPropertyV1> existingAdditionalPropertiesByName =
                    datasource.getAdditionalProperties().stream()
                            .collect(Collectors.toMap(ScalarPropertyV1::getName, p -> p));

            for (ScalarPropertyV1 additionalPropertyToSet : additionalProperties) {
                ScalarPropertyV1 existingProperty =
                        existingAdditionalPropertiesByName.get(additionalPropertyToSet.getName());
                if (existingProperty == null) {
                    propertiesToChange.add(additionalPropertyToSet);
                } else {
                    boolean valueMatches = additionalPropertyToSet.getValue().equals(existingProperty.getValue());
                    boolean uomMatches = Optional.ofNullable(existingProperty.getUnitOfMeasure()).orElse("string")
                            .equals(Optional.ofNullable(additionalPropertyToSet.getUnitOfMeasure()).orElse("string"));
                    if (!valueMatches || !uomMatches) {
                        propertiesToChange.add(additionalPropertyToSet);
                    }
                }
            }
        }

        if (!propertiesToChange.isEmpty()) {
            String summaryString = propertiesToChange.stream().map(p -> "" + p.getName() + " = " + p.getValue())
                    .reduce((x, y) -> x + ", " + y)
                    .orElse("");
            this.getLog().info("Changing properties of datasource with GUID {}: {}", datasourceGuid,
                    summaryString);
            itemsApi.setProperties(datasourceGuid, propertiesToChange);
            datasource = datasourcesApi.getDatasource(datasourceGuid);
        }
        return datasource;
    }

    /**
     * Given data-request-specific information, build an ErrorInfo message.
     *
     * @param e
     *         The exception.
     * @param request
     *         The signal request.
     * @return An ErrorInfo message.
     */
    protected ErrorMessage.ErrorInfo buildErrorMessage(Exception e, SignalRequestMessage request) {
        return this.buildDataRequestErrorMessage(e, "signal", request.getSignalId(),
                request.getStartTime(), request.getEndTime());
    }

    /**
     * Given data-request-specific information, build an ErrorInfo message.
     *
     * @param e
     *         The exception.
     * @param request
     *         The condition request.
     * @return An ErrorInfo message.
     */
    protected ErrorMessage.ErrorInfo buildErrorMessage(Exception e, ConditionRequestMessage request) {
        return this.buildDataRequestErrorMessage(e, "condition", request.getConditionId(),
                request.getStartTime(), request.getEndTime());
    }

    /**
     * Given data-request-specific information, build an ErrorInfo message.
     *
     * @param e
     *         The exception.
     * @param requestType
     *         The type of request the error pertains to, e.g. 'signal' or 'condition'.
     * @param dataId
     *         The dataId requested.
     * @param start
     *         The request start.
     * @param end
     *         The request end.
     * @return An ErrorInfo message.
     */
    protected ErrorMessage.ErrorInfo buildDataRequestErrorMessage(Exception e, String requestType, String dataId,
            long start, long end) {
        return this.buildErrorMessage(ErrorMessage.ErrorCode.EXCEPTION, e,
                String.format("Error processing %s request for %s with start: %s and end: %s.",
                        requestType, dataId, new TimeInstant(start), new TimeInstant(end)));
    }

    /**
     * Given an error code, an optional exception, and an additional error message, build an ErrorInfo message.
     *
     * @param errorCode
     *         An error code indicating the general class of the error.
     * @param e
     *         The exception.
     * @param message
     *         A message to add to the error message.
     * @return An ErrorInfo message.
     */
    protected ErrorMessage.ErrorInfo buildErrorMessage(ErrorMessage.ErrorCode errorCode, Exception e, String
            message) {
        ErrorMessage.ErrorInfo.Builder errorMessage = ErrorMessage.ErrorInfo.newBuilder();
        errorMessage.setCode(errorCode);
        if (e != null) {
            errorMessage.setException(e.getClass().getName());
            errorMessage.setMessage((message + "  " + e.getMessage()).trim());
        } else {
            errorMessage.setMessage(message);
        }

        if (e instanceof OperationCanceledException) {
            this.getLog().debug("Request canceled");
        } else {
            this.getLog().error(errorMessage.getMessage(), e);
            this.getLog().error("Note: {} was developed by {}", this.connector.getName(), this.connectorDeveloperInfo);
        }

        return errorMessage.build();
    }

    /**
     * Save the sync status to be sent to Appserver via the agent status API.
     *
     * @param syncStatus
     *         The current sync status for this datasource.
     * @throws OperationCanceledException
     *         Thrown if indexing is being canceled, often due to a configuration change
     */
    protected void setSyncStatus(SyncStatus syncStatus)
            throws OperationCanceledException {
        // Check for cancellation here, since it's a central place that will be called periodically
        RequestCancellation.check();

        if (syncStatus != this.indexingState.getSyncStatus()) {
            this.indexingState.setSyncStatus(syncStatus);
            this.getAgentService().sendAgentInfoToServer();
        }
    }

    /**
     * Inform Appserver of the sync token for a sync about to take place. Used to track progress.
     *
     * @param datasourceItemId
     *         The item ID for the datasource representing this connection.
     * @param syncToken
     *         The sync token on signals, assets, etc that are part of this sync.
     * @param syncMode
     *         The sync mode (INCREMENTAL, FULL).
     */
    protected void sendSyncToken(String datasourceItemId, String syncToken, String syncMode) {
        ItemsApi itemsApi = this.getAgentService().getIndexingApiProvider().createItemsApi();
        try {
            if (syncToken != null) {
                List<ScalarPropertyV1> body = new ArrayList<>();

                ScalarPropertyV1 syncModeProperty = new ScalarPropertyV1();
                syncModeProperty.setName(SeeqNames.Properties.SyncMode);
                syncModeProperty.setValue(syncMode);
                body.add(syncModeProperty);

                ScalarPropertyV1 syncResultProperty = new ScalarPropertyV1();
                syncResultProperty.setName(SeeqNames.Properties.SyncResult);
                syncResultProperty.setValue(SeeqNames.Connectors.Connections.SyncResult.InProgress);
                body.add(syncResultProperty);

                ScalarPropertyV1 syncTokenProperty = new ScalarPropertyV1();
                syncTokenProperty.setName(SeeqNames.Properties.SyncToken);
                syncTokenProperty.setValue(syncToken);
                body.add(syncTokenProperty);

                itemsApi.setProperties(datasourceItemId, body);
            }
        } catch (ApiException | ProcessingException e) { // Proxy errors can cause ProcessingException (CRAB-12107)
            LOG.error("Could not update sync token to appserver", e);
        }
    }

    protected void logAndCountBadItems(List<ItemUpdateOutputV1> items) {
        List<ItemUpdateOutputV1> badItems =
                items.stream().filter(i -> i.getErrorMessage() != null).collect(Collectors.toList());
        if (badItems.size() > 0) {
            this.itemsWithErrors += badItems.size();
            this.getLog().error("Could not sync the following items:");
            for (ItemUpdateOutputV1 itemUpdate : badItems) {
                this.getLog().error("DataID: {} Item: '{}' ErrorMessage: {}",
                        itemUpdate.getDataId(),
                        itemUpdate.getItem() != null ? itemUpdate.getItem().getName() : "??",
                        itemUpdate.getErrorMessage());
            }
        }
    }

    /**
     * Performs a metadata sync in a consistent way for any datasource, using iterators to draw the items from the
     * source system. Using this function is the preferred route to take when writing a connector. Benefits include:
     *
     * - Automatic batch sizing and throttling
     *
     * - Avoids holding large lists / data structures in memory
     *
     * @param signalInputs
     *         An iterator that produces signals
     * @param hostID
     *         The host identifier for the assets
     * @param assetInputs
     *         An iterator that produces assets
     * @param relationshipInputs
     *         An iterator that produces relationships
     * @param rootAssetInputs
     *         A list of root assets (representing the HDA databases themselves)
     * @param userGroupInputs
     *         An iterator that produces user groups
     * @param newDatasourceVersionCheck
     *         The new datasource version check to use (on success)
     * @param sampleSeriesTransforms
     *         A list of transforms to perform on sample series
     * @param userGroupsTransforms
     *         A list of transforms to perform on user groups
     * @param disableAssetTreeIndexUpdateDuringSync
     *         False if asset tree search index should be kept up to date during the sync, true if it should not.
     *         Setting this to true speeds up FULL relationship syncs considerably, but will require rebuilding the
     *         asset tree search index after the sync completes. Requesting a datasource cleanup after the synccompletes
     *         will automatically trigger a rebuild of the asset tree search index, if it is needed.
     * @return The new datasource version check
     * @throws OperationCanceledException
     *         Thrown if the sync'ing is interrupted from another thread
     * @throws Exception
     *         Thrown if an API call throws an exception
     */
    protected Optional<String> batchSync(
            Iterator<SignalWithIdInputV1> signalInputs,
            String hostID, Iterator<PutAssetInputV1> assetInputs,
            Iterator<AssetTreeSingleInputV1> relationshipInputs,
            List<PutAssetInputV1> rootAssetInputs,
            Iterator<UserGroupWithIdInputV1> userGroupInputs,
            Optional<String> newDatasourceVersionCheck,
            List<PropertyTransformer.Spec> sampleSeriesTransforms,
            // TODO: CRAB-17512 Remove usergroups transforms
            List<PropertyTransformer.Spec> userGroupsTransforms,
            boolean disableAssetTreeIndexUpdateDuringSync)
            throws Exception {
        Exception apiException = null;

        BatchSizeHelper batchSizeHelper = this.getAgentService().createBatchSizeHelper();

        SignalsApi signalsApi = this.getAgentService().getIndexingApiProvider().createSignalsApi();
        AssetsApi assetsApi = this.getAgentService().getIndexingApiProvider().createAssetsApi();
        TreesApi treesApi = this.getAgentService().getIndexingApiProvider().createTreesApi();
        UserGroupsApi userGroupsApi = this.getAgentService().getIndexingApiProvider().createUserGroupsApi();

        long progress = 0;

        // Send initial status to kick it off
        this.setSyncStatus(SyncStatus.SYNC_INITIALIZING);

        // Sync the signals - use a scope to group related code
        {
            PutSignalsInputV1 putSignalsInput = new PutSignalsInputV1();
            List<SignalWithIdInputV1> signalList = new ArrayList<>();
            putSignalsInput.setSignals(signalList);

            boolean hasMore = signalInputs.hasNext();

            if (hasMore) {
                this.getLog().info("Sync Progress (Signals) [{}] first batch size: {}",
                        progress, batchSizeHelper.getBatchSize());
            }

            while (hasMore) {
                SignalWithIdInputV1 signalInput = signalInputs.next();
                hasMore = signalInputs.hasNext();

                if (sampleSeriesTransforms != null) {
                    signalInput = PropertyTransformer.transform(signalInput, sampleSeriesTransforms);
                }
                signalList.add(signalInput);

                progress++;

                if (!hasMore || signalList.size() >= batchSizeHelper.getBatchSize()) {
                    batchSizeHelper.start();
                    try {
                        this.logAndCountBadItems(signalsApi.putSignals(putSignalsInput).getItemUpdates());
                    } catch (Exception e) {
                        this.getLog().error("Error calling PutSignals:", e);

                        if (apiException == null) {
                            apiException = e;
                        }
                    }
                    batchSizeHelper.stop(signalList.size());

                    String batchStats = "last batch";
                    if (hasMore) {
                        batchStats = String.format("%.2f seconds at %.1f per second - next batch size: %d",
                                batchSizeHelper.getLastDuration().toMillis() / 1000d,
                                batchSizeHelper.getLastItemsPerSecond(),
                                batchSizeHelper.getBatchSize());

                        this.setSyncStatus(SyncStatus.SYNC_IN_PROGRESS);
                    }

                    this.getLog().info("Sync Progress (Signals) [{}] {}",
                            progress, batchStats);

                    putSignalsInput = new PutSignalsInputV1();
                    signalList = new ArrayList<>();
                    putSignalsInput.setSignals(signalList);
                }
            }
        }

        if (!rootAssetInputs.isEmpty()) {
            AssetBatchInputV1 assetBatchInput = new AssetBatchInputV1();
            assetBatchInput.setHostId(hostID);
            assetBatchInput.setAssets(rootAssetInputs);

            progress += rootAssetInputs.size();
            this.setSyncStatus(SyncStatus.SYNC_IN_PROGRESS);

            ItemBatchOutputV1 assetBatchOutput = assetsApi.batchCreateAssets(assetBatchInput);

            // Move database node to root of asset tree so it will appear in Workbench
            ItemIdListInputV1 itemIdList = new ItemIdListInputV1();
            itemIdList.setItems(assetBatchOutput.getItemUpdates().stream().map(o -> o.getItem().getId()).collect(
                    Collectors.toList()));
            treesApi.moveNodesToRootOfTree(itemIdList);
        }

        if (assetInputs != null) {
            AssetBatchInputV1 assetBatchInput = new AssetBatchInputV1();
            assetBatchInput.setHostId(hostID);
            assetBatchInput.setAssets(new ArrayList<>());

            boolean hasMore = assetInputs.hasNext();

            if (hasMore) {
                this.getLog().info("Sync Progress (Assets) [{}] first batch size: {}",
                        progress, batchSizeHelper.getBatchSize());
            }

            while (hasMore) {
                PutAssetInputV1 assetInput = assetInputs.next();
                hasMore = assetInputs.hasNext();

                assetBatchInput.getAssets().add(assetInput);

                progress++;

                if (!hasMore || assetBatchInput.getAssets().size() >= batchSizeHelper.getBatchSize()) {
                    batchSizeHelper.start();
                    try {
                        this.logAndCountBadItems(assetsApi.batchCreateAssets(assetBatchInput).getItemUpdates());
                    } catch (Exception e) {
                        this.getLog().error("Error calling BatchCreateAssets:", e);

                        if (apiException == null) {
                            apiException = e;
                        }
                    }
                    batchSizeHelper.stop(assetBatchInput.getAssets().size());

                    String batchStats = "last batch";
                    if (hasMore) {
                        batchStats = String.format("%.2f seconds at %.1f per second - next batch size: %d",
                                batchSizeHelper.getLastDuration().toMillis() / 1000d,
                                batchSizeHelper.getLastItemsPerSecond(),
                                batchSizeHelper.getBatchSize());

                        this.setSyncStatus(SyncStatus.SYNC_IN_PROGRESS);
                    }

                    this.getLog().info("Sync Progress (Signals) [{}] {}", progress, batchStats);

                    assetBatchInput = new AssetBatchInputV1();
                    assetBatchInput.setHostId(hostID);
                    assetBatchInput.setAssets(new ArrayList<>());
                }
            }
        }

        // Sync the asset tree - use a scope to group related code
        {
            AssetTreeBatchInputV1 treeBatchInput = new AssetTreeBatchInputV1();
            treeBatchInput.setParentHostId(hostID);
            treeBatchInput.setChildHostId(hostID);
            treeBatchInput.setDisableAssetTreeIndexUpdateDuringSync(disableAssetTreeIndexUpdateDuringSync);
            treeBatchInput.setRelationships(new ArrayList<>());

            boolean hasMore = relationshipInputs.hasNext();

            if (hasMore) {
                this.getLog().info("Sync Progress (Relationships) [{}] first batch size: {}",
                        progress, batchSizeHelper.getBatchSize());
            }

            while (hasMore) {
                AssetTreeSingleInputV1 relationshipInput = relationshipInputs.next();
                hasMore = relationshipInputs.hasNext();

                treeBatchInput.getRelationships().add(relationshipInput);

                progress++;

                if (!hasMore || treeBatchInput.getRelationships().size() >= batchSizeHelper.getBatchSize()) {
                    batchSizeHelper.start();
                    try {
                        this.logAndCountBadItems(treesApi.batchMoveNodesToParents(treeBatchInput).getItemUpdates());
                    } catch (Exception e) {
                        this.getLog().error("Error calling BatchMoveNodesToParents:", e);

                        if (apiException == null) {
                            apiException = e;
                        }
                    }
                    batchSizeHelper.stop(treeBatchInput.getRelationships().size());

                    String batchStats = "last batch";
                    if (hasMore) {
                        batchStats = String.format("%.2f seconds at %.1f per second - next batch size: %d",
                                batchSizeHelper.getLastDuration().toMillis() / 1000d,
                                batchSizeHelper.getLastItemsPerSecond(),
                                batchSizeHelper.getBatchSize());

                        this.setSyncStatus(SyncStatus.SYNC_IN_PROGRESS);
                    }

                    this.getLog().info("Sync Progress (Relationships) [{}] {}", progress, batchStats);

                    treeBatchInput = new AssetTreeBatchInputV1();
                    treeBatchInput.setParentHostId(hostID);
                    treeBatchInput.setChildHostId(hostID);
                    treeBatchInput.setDisableAssetTreeIndexUpdateDuringSync(disableAssetTreeIndexUpdateDuringSync);
                    treeBatchInput.setRelationships(new ArrayList<>());
                }
            }
        }

        // Sync the user groups - use a scope to group related code
        {
            PutUserGroupsInputV1 putUserGroupsInput = new PutUserGroupsInputV1();
            List<UserGroupWithIdInputV1> userGroupList = new ArrayList<>();
            putUserGroupsInput.setUserGroups(userGroupList);

            boolean hasMore = userGroupInputs.hasNext();

            if (hasMore) {
                this.getLog().info("Sync Progress (UserGroups) [{}] first batch size: {}",
                        progress, batchSizeHelper.getBatchSize());
            }

            while (hasMore) {
                UserGroupWithIdInputV1 userGroupInput = userGroupInputs.next();
                hasMore = userGroupInputs.hasNext();

                if (userGroupsTransforms != null) {
                    userGroupInput = PropertyTransformer.transform(userGroupInput, userGroupsTransforms);
                }
                userGroupList.add(userGroupInput);

                progress++;

                if (!hasMore || userGroupList.size() >= batchSizeHelper.getBatchSize()) {
                    batchSizeHelper.start();
                    try {
                        this.logAndCountBadItems(userGroupsApi.putUserGroups(putUserGroupsInput).getItemUpdates());
                    } catch (Exception e) {
                        this.getLog().error("Error calling PutUserGroups:", e);

                        if (apiException == null) {
                            apiException = e;
                        }
                    }
                    batchSizeHelper.stop(userGroupList.size());

                    String batchStats = "last batch";
                    if (hasMore) {
                        batchStats = String.format("%.2f seconds at %.1f per second - next batch size: %d",
                                batchSizeHelper.getLastDuration().toMillis() / 1000d,
                                batchSizeHelper.getLastItemsPerSecond(),
                                batchSizeHelper.getBatchSize());

                        this.setSyncStatus(SyncStatus.SYNC_IN_PROGRESS);
                    }

                    this.getLog().info("Sync Progress (UserGroups) [{}] {}",
                            progress, batchStats);

                    putUserGroupsInput = new PutUserGroupsInputV1();
                    userGroupList = new ArrayList<>();
                    putUserGroupsInput.setUserGroups(userGroupList);
                }
            }
        }


        if (apiException == null) {
            return newDatasourceVersionCheck;
        } else {
            // Rethrow the first exception so that datasource cleanup doesn't run. Also avoids sending complete
            // status or updating datasource version check. If we do a cleanup but some API calls failed, we may
            // archive signals that still exist but simply failed to sync. See CRAB-9605.
            throw apiException;
        }
    }

    /**
     * Divides changed signal metadata into batches and writes those batches to the REST API, creating or
     * updating those signals.
     *
     * @param signalInputs
     *         The SignalWithIdInputV1 objects to write to the REST API
     * @param newDatasourceVersionCheck
     *         The optional new version check for this datasource, to be applied if sync is successful
     * @return The new datasource version check
     * @throws InterruptedException
     *         Thrown if the sync'ing is interrupted from another thread
     * @throws Exception
     *         Thrown if an API call throws an exception
     */
    protected Optional<String> batchSyncSampleSeries(Iterator<SignalWithIdInputV1> signalInputs,
            Optional<String> newDatasourceVersionCheck, List<PropertyTransformer.Spec> signalTransforms)
            throws Exception, InterruptedException {
        return this.batchSync(signalInputs, null, Collections.emptyIterator(),
                Collections.emptyIterator(), Collections.emptyList(), Collections.emptyIterator(),
                newDatasourceVersionCheck, signalTransforms, null, true);
    }

    /**
     * Cancel a running request. We use this to prevent a previous long-running clean-up request from overlapping
     * with a subsequent metadata sync.
     *
     * Note: this shouldn't be necessary during normal operation. This is to guard from inconsistent data if an
     * agent is restarted while the clean-up process continues to run on Appserver.
     *
     * @param requestId
     *         The unique identifier that was provided when creating the request
     * @throws ApiException
     *         Thrown if request cannot be cancelled
     */
    protected void cancelRequest(String requestId) throws ApiException {
        RequestsApi requestsApi = this.getAgentService().getIndexingApiProvider().createRequestsApi();
        try {
            requestsApi.cancelRequest(URLEncoder.encode(requestId, "UTF-8"));
        } catch (UnsupportedEncodingException e) {
            // UTF-8 is definitely supported, so we turn this into an unchecked exception.
            throw new RuntimeException(e);
        } catch (ApiException e) {
            if (e.getCode() == 404) {
                // No need to cancel the request. It has already completed!
                return;
            }
            throw e;
        }
    }

    /**
     * Call the Seeq Appserver to clean up (archive) any stale items in this datasource.
     *
     * @param datasourceItemId
     *         The datasource item ID to clean up stale items on
     * @param syncToken
     *         The sync token that specifies which items to preserve
     * @param itemTypesFilter
     *         The list of item types on which cleanup will be done. When no filter is specified, all types of items
     *         will be included in the cleanup process
     * @param datasourceItemDataIdRegexFilter
     *         The datasource item DataId Regex filter. When set, only items having DataId matching the RegEx will
     *         be included in the cleanup process.
     * @param datasourceItemDataIdExcludeRegexFilter
     *         The datasource item DataId exclude Regex filter. When set, the items having DataID matching the
     *         RegEx will be excluded from cleanup process.
     * @param datasourceItemNameRegexFilter
     *         The datasource item name Regex filter. When set, only items having the name matching the RegEx will
     *         be included in the cleanup process.
     * @param datasourceItemNameExcludeRegexFilter
     *         The datasource item name exclude Regex filter. When set, the items having the name matching the
     *         RegEx will be excluded from cleanup process.
     * @param requestId
     *         A requestID to use for cancellation purposes
     * @throws ApiException
     *         Thrown if an unhandled Seeq API exception is thrown
     */
    protected void cleanUpStaleItems(String datasourceItemId, String syncToken, List<String> itemTypesFilter,
            String datasourceItemDataIdRegexFilter, String datasourceItemDataIdExcludeRegexFilter,
            String datasourceItemNameRegexFilter, String datasourceItemNameExcludeRegexFilter,
            String requestId)
            throws ApiException {
        ApiClient apiClient = this.getAgentService().getIndexingApiProvider().getApiClient();

        int originalReadTimeout = apiClient.getReadTimeout();

        try {
            apiClient.setReadTimeout(24 * 60 * 60 * 1000); // 24 hours, because archiving may take a long time

            DatasourceCleanUpInputV1 datasourceCleanUpInput = new DatasourceCleanUpInputV1();
            datasourceCleanUpInput.setSyncToken(syncToken);
            datasourceCleanUpInput.setItemTypeFilter(itemTypesFilter);
            datasourceCleanUpInput.setItemDataIdRegexFilter(datasourceItemDataIdRegexFilter);
            datasourceCleanUpInput.setItemDataIdExcludeRegexFilter(datasourceItemDataIdExcludeRegexFilter);
            datasourceCleanUpInput.setItemNameRegexFilter(datasourceItemNameRegexFilter);
            datasourceCleanUpInput.setItemNameExcludeRegexFilter(datasourceItemNameExcludeRegexFilter);

            this.setSyncStatus(SyncStatus.SYNC_ARCHIVING_DELETED_ITEMS);

            try {
                requestId = URLEncoder.encode(requestId, "UTF-8");
            } catch (UnsupportedEncodingException ignored) {}

            this.getLog().debug("Datasource clean-up starting for sync token {}", syncToken);

            // We want to add a request ID to the potentially-long running request so we can cancel it if necessary.
            // Unfortunately, the Swagger generated client doesn't let us set headers so we have to use the more
            // primitive "invokeAPI" method.
            String path = SeeqNames.API.Datasources + "/" + datasourceItemId + "/cleanup";
            String method = "POST";
            List<Pair> queryParams = Collections.emptyList();
            Map<String, String> headerParams = Maps.newHashMap();
            headerParams.put("x-sq-request-id", requestId);
            String seeqContentType = "application/vnd.seeq.v1+json";
            String accept = apiClient.selectHeaderAccept(new String[] { seeqContentType });
            String contentType = apiClient.selectHeaderContentType(new String[] { seeqContentType });
            String[] authNames = new String[] {};

            DatasourceCleanUpOutputV1 datasourceCleanUpOutput = apiClient.invokeAPI(path, method, queryParams,
                    datasourceCleanUpInput, headerParams, Collections.emptyMap(), accept, contentType, authNames,
                    new GenericType<DatasourceCleanUpOutputV1>() {});

            int numNewlyArchivedItems = datasourceCleanUpOutput.getNumNewlyArchivedItems();
            if (numNewlyArchivedItems > 0) {
                this.getLog()
                        .debug("Datasource clean-up complete for sync token {}; archived {} stale items", syncToken,
                                numNewlyArchivedItems);
            } else {
                this.getLog()
                        .debug("Datasource clean-up complete for sync token {}; no items were stale", syncToken);
            }
        } finally {
            // Restore the original timeout no matter what.
            apiClient.setReadTimeout(originalReadTimeout);
        }
    }

    @Override
    public void saveConfig() {
        this.connector.saveConfig();
    }
}
