package com.seeq.link.sdk;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableList;
import com.seeq.link.messages.agent.AgentMessages;
import com.seeq.link.sdk.interfaces.AgentService;
import com.seeq.link.sdk.interfaces.ConfigService;
import com.seeq.link.sdk.interfaces.Connection;
import com.seeq.link.sdk.interfaces.Connection.ConnectionState;
import com.seeq.link.sdk.interfaces.Connection.StateChangedEventArgs;
import com.seeq.link.sdk.interfaces.Connector;
import com.seeq.link.sdk.interfaces.DatasourceConnection;
import com.seeq.link.sdk.utilities.Event;

import lombok.extern.slf4j.Slf4j;

/**
 * Facilitates a consistent approach to connector implementation, including initialization and message routing.
 *
 * See the {@link Connector} interface for more documentation.
 *
 * @param <TConfig>
 *         The class for the top-level configuration object.
 */
@Slf4j
abstract class BaseConnector<TConfig> extends Configurable<TConfig> {

    private final List<DatasourceConnection> connectionsUnsafe = new ArrayList<>();

    private AgentService agentService;

    protected AgentService getAgentService() {
        return this.agentService;
    }

    public abstract void initialize(AgentService agentService) throws Exception;

    protected synchronized void initialize(AgentService agentService, ConfigObject[] defaultConfigObjects)
            throws IOException {
        this.agentService = agentService;

        super.initialize(agentService.getConfigService(), defaultConfigObjects);

        this.agentService.markConnectionsAsReinitialized(
                this.getConnections().stream().map(DatasourceConnection::getConnectionId).collect(Collectors.toList()));
    }

    @Override
    protected synchronized void initialize(ConfigService configService, ConfigObject[] defaultConfigObjects) {
        throw new UnsupportedOperationException(
                "initialize(IConfigService configService, Object defaultConfigObject) should not " +
                        "be called from derived classes directly. Use initialize(AgentService agentService, " +
                        "ConfigObject[] defaultConfigObject) instead.");
    }

    /**
     * Called by a derived class when a new connection Object is created. This should be called regardless of whether
     * the connection was actually established.
     *
     * @param connection
     *         The connection to be added to the internal list of connections.
     * @throws Exception
     *         thrown if connection had an issue initializing
     */
    protected synchronized void initializeConnection(DatasourceConnection connection) throws Exception {
        connection.getStateChangedEvent().add((sender, args) -> {
            this.connectionStateChanged(args.getSender(), args.getState(), args.getMessage());
        });

        connection.initialize();

        synchronized (this.connectionsUnsafe) {
            this.connectionsUnsafe.add(connection);
        }
    }

    @Override
    public synchronized void destroy() {
        super.destroy();

        // We want to avoid synchronizing on connectionsUnsafe during connection.destroy() because this can take
        // long time.
        for (DatasourceConnection connection : this.getConnections()) {
            connection.destroy();
        }

        synchronized (this.connectionsUnsafe) {
            this.connectionsUnsafe.clear();
        }
    }

    public List<DatasourceConnection> getConnections() {
        synchronized (this.connectionsUnsafe) {
            return ImmutableList.copyOf(this.connectionsUnsafe);
        }
    }

    public boolean processMessage(String destinationConnectorInstanceId, AgentMessages.DataDocument data) {
        Optional<DatasourceConnection> connection = this.getConnections().stream()
                .filter(c -> c.getConnectionId().equals(destinationConnectorInstanceId)).findFirst();
        if (connection.isPresent()) {
            connection.get().processMessage(data);
            return true;
        } else {
            return false;
        }
    }

    private final Event<StateChangedEventArgs> connectionStateChangedEvent = new Event<>();

    public Event<StateChangedEventArgs> getConnectionStateChangedEvent() {
        return this.connectionStateChangedEvent;
    }

    protected void connectionStateChanged(Connection sender, ConnectionState state, String message) {
        this.connectionStateChangedEvent.dispatch(sender, new StateChangedEventArgs(sender, state, message));
    }

    @Override
    protected synchronized void onConfigChanged() {
        LOG.info("{} configuration changed", this.getName());

        try {
            this.destroy();
        } catch (Exception e) {
            LOG.error("Error destroying connector", e);
        }

        try {
            this.initialize(this.getAgentService());
        } catch (Exception e) {
            LOG.error("Error initializing connector", e);
        }
    }
}
